"""ParameterData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.srh.components.parameters.bc_parameter import BcParameter
from xms.srh.components.parameters.initial_condition_parameter import InitialConditionParameter
from xms.srh.components.parameters.inlet_q_parameter import InletQParameter
from xms.srh.components.parameters.mannings_n_parameter import ManningsNParameter
from xms.srh.components.parameters.time_step_parameter import TimeStepParameter


class ParameterData:
    """Parameter data used in ParametersDialog."""
    def __init__(
        self,
        use_parameters=False,
        run_type='Scenarios',
        show_type='All',
        max_iterations=5,
        max_iterations_no_improvement=3,
        run_count=1,
        xy_series=None,
        params=None,
        runs=None
    ):
        """Initializes the class."""
        self.use_parameters = use_parameters
        self.run_type = run_type
        self.show_type = show_type
        self.max_iterations = max_iterations
        self.max_iterations_no_improvement = max_iterations_no_improvement
        self.run_count = run_count
        self.xy_series = xy_series or []
        self.params = params or []  # List of Parameter objects
        self.runs = runs or {}  # The runs table

    def to_dict(self):
        """Returns a dict representation of this class.

        Returns:
            (dict): See description.
        """
        param_data_dict = {
            'use_parameters': self.use_parameters,
            'run_type': self.run_type,
            'show_type': self.show_type,
            'max_iterations': self.max_iterations,
            'max_iterations_no_improvement': self.max_iterations_no_improvement,
            'run_count': self.run_count,
            'xy_series': self.xy_series,
            'params': self.dict_from_parameter_list(),
            'runs': self.runs,
        }
        return param_data_dict

    @staticmethod
    def from_dict(the_dict):
        """Returns a ParameterData class built from the dict.

        Args:
            the_dict (dict): A dict.
        """
        param_data = ParameterData()
        param_data.use_parameters = the_dict.get('use_parameters', param_data.use_parameters)
        param_data.run_type = the_dict.get('run_type', param_data.run_type)
        param_data.show_type = the_dict.get('show_type', param_data.show_type)
        param_data.max_iterations = the_dict.get('max_iterations', param_data.max_iterations)
        param_data.max_iterations_no_improvement = the_dict.get(
            'max_iterations_no_improvement', param_data.max_iterations_no_improvement
        )
        param_data.run_count = the_dict.get('run_count', param_data.run_count)
        param_data.xy_series = the_dict.get('xy_series', param_data.xy_series)
        param_data.params = ParameterData.parameter_list_from_dict(the_dict)
        param_data.populate_runs_table(the_dict)
        return param_data

    def parameter_from_column_name(self, column_name):
        """Given the column name in the runs table, returns the corresponding Parameter.

        Args:
            column_name (str): Name of the column

        Returns:
            (Parameter): The parameter.
        """
        # Just ask each the parameter if they have the column
        for parameter in self.params:
            if parameter.has_column_name(column_name):
                return parameter
        raise RuntimeError('Error matching column to parameter.')  # pragma no cover - hard to test Exceptions

    @staticmethod
    def is_bc(_id):
        """Returns true if the _id string indicates it is a bc.

        Args:
            _id (str): id from the params table.

        Returns
            tuple: (bool, int) True if the specified id is a BC, the next available type for the BC
        """
        if _id.startswith('inlet_q'):
            return True, len('inlet_q') + 1
        elif _id.startswith('exit_h'):
            return True, len('exit_h') + 1
        elif _id.startswith('internal_sink'):
            return True, len('internal_sink') + 1
        return False, -1  # pragma: no cover

    @staticmethod
    def _parse_id_number(id_string, _type):
        id_number = None
        if _type == "Manning's N":
            id_number = int(float(id_string[4:]))
        else:
            is_bc, bc_length = ParameterData.is_bc(id_string)
            if is_bc:
                id_number = int(float(id_string[bc_length:]))
        return id_number

    def dict_from_parameter_list(self):
        """Returns a dict with the info from the list of parameters.

        Returns:
            (dict): See description.
        """
        parameters_dict = {}
        for parameter in self.params:
            parameter.to_dict(parameters_dict)
        return parameters_dict

    @staticmethod
    def parameter_list_from_dict(the_dict):
        """Returns a list of parameter objects by building them from the dict.

        Args:
            the_dict (dict): The parameters as a dict (as read from the json file).
        """
        params = []
        for i, _id in enumerate(the_dict['params']['id']):
            _type = the_dict['params']['Type'][i]
            id_number = ParameterData._parse_id_number(_id, _type)
            if _type == "Manning's N":
                param = ManningsNParameter(id_number=id_number)
            elif _type == 'Inlet Q':
                param = InletQParameter(id_number=id_number)
            elif _type == 'Time step':
                param = TimeStepParameter()
            elif _type == 'Initial condition':
                param = InitialConditionParameter()
            else:
                param = BcParameter(use=the_dict['params']['Use'][i], id_number=id_number, type=_type)

            param.copy_disk_data(the_dict)
            params.append(param)
        return params

    def apply_disk_data(self, disk_data):
        """Updates the Parameters with what was on disk.

        Args:
            disk_data (dict): Parameter data that was read from disk.
        """
        if not disk_data:
            return

        # Copy some these items from disk
        self.use_parameters = disk_data['use_parameters']
        self.run_type = disk_data['run_type']
        self.show_type = disk_data['show_type']
        self.max_iterations = disk_data['max_iterations']
        self.max_iterations_no_improvement = disk_data['max_iterations_no_improvement']
        self.run_count = disk_data['run_count']
        self.xy_series = disk_data['xy_series']

        # Update the parameters table with disk data if it exists
        for parameter in self.params:
            # If the disk data includes info for this parameter, use it.
            if parameter.id_string() in disk_data['params']['id']:
                parameter.copy_disk_data(disk_data)

    def populate_runs_table(self, disk_data):
        """Populates the runs table.

        Args:
            disk_data (dict): Param data from file.
        """
        # Add 'Run' column
        run_count = self.run_count  # This can't have changed since the last time
        if disk_data:
            self.runs['Run'] = disk_data['runs']['Run']
        else:
            self.runs['Run'] = [f'Run {i + 1}' for i in range(run_count)]

        # Add all the other columns
        for parameter in self.params:
            parameter.add_run_data(disk_data, run_count, self)

    def get_defaults(self):
        """Returns a dict containing the default values for each column in the runs table except the Run column.

        Returns:
            (dict): See description.
        """
        defaults = {}
        for column_name in self.runs.keys():
            if column_name == 'Run':
                continue
            parameter = self.parameter_from_column_name(column_name)
            defaults[column_name] = parameter.column_default(column_name)
        return defaults
