"""BcDataSedInflow class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import pandas as pd
import param

# 3. Aquaveo modules
from xms.guipy.param.param_enabler import ParamEnabler  # noqa: I202

# 4. Local modules


def empty_table() -> pd.DataFrame:
    """Returns an empty sediment table."""
    df = pd.DataFrame({'Time(hr)': [0.0], 'Qs1(cfs)': [0.0]})
    df.index += 1
    return df


class BcDataSedInflow(param.Parameterized):
    """Bc type sediment inflow data."""
    sediment_discharge_type = param.ObjectSelector(
        default='Capacity',
        objects=['Capacity', 'File', 'Table'],
        doc='sediment discharge type',
        precedence=1,
    )
    sediment_file = param.String(
        label='Rates per sediment size class defined in a file',
        doc='sediment input file',
        precedence=2,
    )

    # Table options

    # "Import File..." button
    import_sediment_file = param.Action(label='Import File...', doc='import sediment file', precedence=3.1)
    # File path of the imported file so the Open File dialog remembers for the next time
    imported_sediment_file = param.String(label='', doc='file path of the imported file', precedence=3.2)
    # The table
    sediment_table = param.DataFrame(default=empty_table(), doc='sediment table', precedence=3.3)
    # "Edit Table..." button
    edit_table = param.Action(label='Edit Table...', doc='edit table button', precedence=3.4)
    # Scale factor label
    scale_factor_label = param.String(default='Scale factor: ', doc='scale factor label', label='', precedence=3.5)
    # Scale factor
    scale_factor = param.Number(default=1.0, bounds=(None, None), doc='scale factor', label='', precedence=3.6)
    # File name of the .csv file that stores the table next to the component main file
    sediment_table_file_name = param.String(label='', doc='name of the csv file for the table', precedence=3.7)

    def __init__(self):
        """Initializes the class."""
        super().__init__()
        self.sediment_file = ''
        self.enabler = ParamEnabler(
            param_cls=self,
            enablers=['sediment_discharge_type'],
            depends={
                'File': ['sediment_file'],
                'Table': ['import_sediment_file', 'sediment_table', 'edit_table', 'scale_factor_label', 'scale_factor'],
            },
        )

    @param.depends('sediment_discharge_type', watch=True)
    def _update_sediment_discharge_type(self):
        """Watches changes made to the discharge_option object."""
        self.enabler.do_enabling(force_disable=False)
