"""SedimentCohesive class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import param

# 3. Aquaveo modules
from xms.guipy.param.param_enabler import ParamEnabler

# 4. Local modules


class SedimentCohesive(param.Parameterized):
    """Manages data file for the hidden simulation component."""
    bulk_density = param.Number(
        default=110.0,
        bounds=(0.0, None),
        doc='',
        precedence=1,
    )
    bulk_density_units = param.ObjectSelector(
        label='',
        default='English',
        objects=['SI', 'English'],
        doc='',
        precedence=1.1,
    )
    fall_velocity = param.ObjectSelector(
        default='Kaolinite Properties',
        objects=['Kaolinite Properties', 'Severn River Properties', 'Velocity Data File (mm/sec)'],
        doc='',
        precedence=2,
    )
    fall_velocity_data_file = param.String(
        doc='',
        precedence=2.1,
    )
    # erosion rate group
    erosion_rate = param.ObjectSelector(
        default='Parameters',
        objects=['Parameters', 'Erosion Data File'],
        doc='',
        precedence=3,
    )
    surface_erosion = param.Number(
        label='Critical shear stress for surface erosion (Pa or lb/ft^2)',
        default=0.0,
        bounds=(0.0, None),
        doc='',
        precedence=3.1,
    )
    mass_erosion = param.Number(
        label='Critical shear stress for mass erosion (Pa or lb/ft^2)',
        default=0.0,
        bounds=(0.0, None),
        doc='',
        precedence=3.2,
    )
    surface_erosion_constant = param.Number(
        label='Surface erosion constant (mm/sec or lb/ft^2/hr)',
        default=0.0,
        bounds=(0.0, None),
        doc='',
        precedence=3.3,
    )
    mass_erosion_constant = param.Number(
        label='Mass erosion constant (mm/sec or lb/ft^2/hr)',
        default=0.0,
        bounds=(0.0, None),
        doc='',
        precedence=3.4,
    )
    erosion_units = param.ObjectSelector(
        default='SI',
        objects=['SI', 'English'],
        doc='',
        precedence=3.5,
    )
    erosion_rate_data_file = param.String(
        doc='',
        precedence=3.6,
    )
    # deposition group
    full_depostion = param.Number(
        label='Critical shear stress for full deposition (Pa or lb/ft^2)',
        default=0.0,
        bounds=(0.0, None),
        doc='',
        precedence=4,
    )
    partial_depostion = param.Number(
        label='Critical shear stress for partial deposition (Pa or lb/ft^2)',
        default=0.0,
        bounds=(0.0, None),
        doc='',
        precedence=5,
    )
    equilibrium_concentration = param.Number(
        label='Equilibrium concentration (g/l or kg/m^3 [1 g/l approx 1000 ppm-wt])',
        default=0.0,
        bounds=(0.0, None),
        doc='',
        precedence=6,
    )
    deposition_units = param.ObjectSelector(
        default='SI',
        objects=['SI', 'English'],
        doc='',
        precedence=7,
    )

    def __init__(self):
        """Initializes the data class.

        Args:
            data_file (:obj:`str`): The netcdf file (with path) associated with this instance data. Probably the owning
                component's main file.

        """
        super().__init__()
        params = [
            'surface_erosion', 'mass_erosion', 'surface_erosion_constant', 'mass_erosion_constant', 'erosion_units'
        ]
        self.enabler = ParamEnabler(
            param_cls=self,
            enablers=['fall_velocity', 'erosion_rate'],
            depends={
                'Velocity Data File (mm/sec)': ['fall_velocity_data_file'],
                'Parameters': params,
                'Erosion Data File': ['erosion_rate_data_file'],
            },
        )

    @param.depends('fall_velocity', watch=True)
    def _update_fall_velocity(self):
        self.enabler.do_enabling(force_disable=False)

    @param.depends('erosion_rate', watch=True)
    def _update_erosion_rate(self):
        self.enabler.do_enabling(force_disable=False)
