"""Writes SRH-2D boundary condition data to the hydro file."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
from pathlib import Path

# 2. Third party modules
import pandas as pd

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as xmf
from xms.guipy.validators.number_corrector import format_float

# 4. Local modules
from xms.srh.data.par.bc_data_culvert_hy8 import hy8_file_dict_from_hy8_file
from xms.srh.data.par.bc_data_sed_inflow import BcDataSedInflow

# Constants
PREC = 3  # Precision for float formatting


def _bc_data_lines_error(direction, label, arc_id):
    """Formats an error meesage for when a BCDATA line label is not found.

    Args:
        direction (:obj:`str`): 'upstream' or 'downstream'
        label (:obj:`str`): BCDATA line label
        arc_id (:obj:`int`): the id of the arc in the coverage

    Returns:
        (:obj:`str`): the error message
    """
    error_msg = f'The {direction} "BCDATA" line label: "{label}" was not found for structure associated with ' \
                f'arc id: {arc_id}.\nEdit the structure and select the appropriate "BCDATA" line.'
    return error_msg


def _convert_to_grid_units(grid_units: str, value: float) -> float:
    """Returns the value converted to the units of the grid.

    This exists for ease of testing.

    Args:
        grid_units: The units of the grid.
        value: A value in ft or meter.

    Returns:
        The converted value.
    """
    if grid_units == 'GridUnit "METER"':
        value = value / 3.281  # Conversion from ft to meter
    return value


def _culvert_elevation_warning_up(culvert_elev: float, grid_elev: float, tolerance: float, arc_id: int) -> str:
    """Returns a warning message for when the upstream culvert invert elevation is not within tolerance of the grid.

    Args:
        culvert_elev: Upstream invert + embedment elevation.
        grid_elev: Minimum grid elevation.
        tolerance: HY8 elevation tolerance.
        arc_id: The arc id.

    Returns:
        See description.
    """
    grid_elev_str = format_float(grid_elev, PREC)
    diff_str = format_float(abs(culvert_elev - grid_elev), PREC)
    return (
        f'Upstream culvert invert elevation is not within tolerance of mesh elevation for arc {arc_id}.\n'
        f'    Invert + embedment: {culvert_elev}; Minimum mesh elevation: {grid_elev_str};'
        f' Difference: {diff_str}; Tolerance: {tolerance}.'
    )


def _culvert_elevation_warning_down(culvert_elev: float, grid_elev: float, tolerance: float, arc_id: int) -> str:
    """Returns a warning message for when the upstream culvert invert elevation is not within tolerance of the grid.

    Args:
        culvert_elev: Downstream invert + embedment elevation.
        grid_elev: Minimum grid elevation.
        tolerance: HY8 elevation tolerance.
        arc_id: The arc id.

    Returns:
        See description.
    """
    grid_elev_str = format_float(grid_elev, PREC)
    diff_str = format_float(abs(culvert_elev - grid_elev), PREC)
    return (
        f'Downstream culvert invert elevation is not within tolerance of mesh elevation for arc {arc_id}.\n'
        f'    Invert + embedment: {culvert_elev}; Minimum mesh elevation: {grid_elev_str};'
        f' Difference: {diff_str}; Tolerance: {tolerance}.'
    )


def _culvert_elevation_warning_range(highest: float, lowest: float, rise_diameter: float, arc_id: int) -> str:
    """Returns a warning message for when the downstream grid elevation range > 67% of the culvert rise/diameter.

    Args:
        highest: Highest grid elevation along arc.
        lowest: Lowest grid elevation along arc.
        rise_diameter: 67% of the culvert rise/diameter.
        arc_id: The arc id.

    Returns:
        See description.
    """
    range_str = format_float(highest - lowest, PREC)
    rise_diameter_str = format_float(rise_diameter * 2 / 3, PREC)
    return (
        f'Downstream mesh elevation range is greater than 67% of culvert rise/diameter for arc {arc_id}.\n'
        f'    Grid elevation range: {range_str}; 67% of culvert rise/diameter: {rise_diameter_str}.'
    )


class HydroBcWriter:
    """Writes SRH-2D boundary condition data to the hydro file."""
    def __init__(self, file, hydro_writer):
        """Constructor.

        Args:
            file: Output *.srhhydro file stream
            hydro_writer (:obj:`HydroWriter`): The main hydro file writer
        """
        self._file = file
        self._logger = hydro_writer.logger
        self._hydro_writer = hydro_writer
        self._sediment_on = hydro_writer.sediment_on
        self._offset = hydro_writer.num_monitor_lines
        self._bc_data = hydro_writer.bc_data
        self._bc_arc_id_to_bc_id = hydro_writer.bc_arc_id_to_bc_id
        self._bc_arc_id_to_grid_pts = hydro_writer.bc_arc_id_to_grid_pts
        self._bc_bc_id_to_structure = hydro_writer.bc_bc_id_to_structure
        self._hy8_file = hydro_writer.bc_hy8_file
        self._bc_3d_structures = hydro_writer.bc_3d_structures
        self._monitor_3d_structures = hydro_writer.monitor_3d_structures
        self._inlet_coefficients = HydroBcWriter.get_inlet_coefficients()
        self._bc_lines = []
        self._cur_arc_id = -1
        self._cur_bc_param = None
        self._bc_data_line_labels = None

    def write(self):
        """Write BC data to the *.srhhydro file."""
        bcs = {
            'Inlet-Q (subcritical inflow)': ('INLET-Q', '_write_inlet_q', 'inlet_q'),
            'Exit-H (subcritical outflow)': ('EXIT-H', '_write_exit_h', 'exit_h'),
            'Exit-Q (known-Q outflow)': ('EXIT-Q', '_write_exit_q', 'exit_q'),
            'Inlet-SC (supercritical inflow)': ('INLET-SC', '_write_inlet_sc', 'inlet_sc'),
            'Exit-EX (supercritical outflow)': ('EXIT-EX', None, None),
            'Wall (no-slip boundary)': ('WALL', '_write_wall', 'wall'),
            'Symmetry (slip boundary)': ('SYMMETRY', None, None),
            'Internal sink': ('INTERNAL', '_write_internal_sink', 'internal_sink'),
            'Bc Data': ('BCDATA', None, None),
        }
        sed_bcs = {
            'Inlet-Q (subcritical inflow)': ('INLET-Q', '_write_sed_inlet_q', 'inlet_q', 'sediment_inflow'),
            'Inlet-SC (supercritical inflow)': ('INLET-SC', '_write_sed_inlet_sc', 'inlet_sc', 'sediment_inflow'),
        }
        struct_ids = {}
        arc_structure_idx = []
        self._bc_data_line_labels = {}
        inlet_q_index = 0
        internal_count = 0
        exit_h_count = 0
        for arc_id, bc_param in self._bc_data.items():
            self._cur_arc_id = arc_id
            self._cur_bc_param = bc_param
            id_in_file = arc_id + self._offset
            if bc_param.bc_type in bcs.keys():
                bc_info = bcs[bc_param.bc_type]
                type_str = bc_info[0]
                if self._sediment_on and bc_param.bc_type == 'Inlet-Q (subcritical inflow)':
                    inlet_q_index += 1
                    arc_structure_idx.append((arc_id, 'InletQ', inlet_q_index, ''))
                self._file.write(f'BC {id_in_file} {type_str}\n')
                if bc_info[1]:
                    if self._sediment_on and bc_param.bc_type in sed_bcs:  # Write extra sediment cards if applicable
                        sed_bc_info = sed_bcs[bc_param.bc_type]
                        writer_method = getattr(self, sed_bc_info[1])
                        par_cls1 = getattr(bc_param, sed_bc_info[2])
                        par_cls2 = getattr(bc_param, sed_bc_info[3])
                        writer_method(id_in_file, par_cls1, par_cls2)
                    else:
                        writer_method = getattr(self, bc_info[1])
                        par_cls = getattr(bc_param, bc_info[2])
                        writer_method(id_in_file, par_cls)
                if bc_param.bc_type == 'Bc Data':
                    if bc_param.bc_data.label not in self._bc_data_line_labels:
                        self._bc_data_line_labels[bc_param.bc_data.label] = id_in_file
                    else:
                        msg = f'More than one arc is using the "BCDATA" label: {bc_param.bc_data.label}.\n' \
                              f'The "BCDATA" label must be unique. Check arc id: {arc_id}.'
                        self._logger.error(msg)
                elif bc_param.bc_type == 'Internal sink':
                    internal_count += 1
                    arc_structure_idx.append((arc_id, 'INTERNAL', internal_count, ''))
                elif bc_param.bc_type == 'Exit-H (subcritical outflow)':
                    if bc_param.exit_h.water_surface_elevation_option == 'Rating curve':
                        exit_h_count += 1
                        arc_structure_idx.append((arc_id, 'EXITH_RC', exit_h_count, ''))

            else:
                struct_ids[arc_id] = bc_param
                type_str = bc_param.bc_type.upper()
                if 'HY-8' in type_str:
                    type_str = 'HY8'
                    if bc_param.hy8_culvert.simulate_as_link:
                        type_str = 'LINK'
                self._file.write(f'BC {id_in_file} {type_str}\n')

        for struct in self._bc_3d_structures:
            self._file.write(f'BC {struct["up_nodestring_id"]} LINK\n')
            self._file.write(f'BC {struct["down_nodestring_id"]} LINK\n')

        for bc in self._bc_lines:
            self._file.write(f'{bc}\n')
        self._bc_lines = []

        structs = {
            'Culvert HY-8': ['HY8Nodestrings', '_write_hy8_culvert', 'hy8_culvert', 0, 'HY'],
            'Culvert': ['CulvertNodestrings', '_write_culvert', 'culvert', 0, ''],
            'Weir': ['WeirNodestrings', '_write_weir', 'weir', 0, 'WEIR'],
            'Pressure': ['PressureNodestrings', '_write_pressure', 'pressure', 0, 'INTERNAL'],
            'Gate': ['GateNodestrings', '_write_gate', 'gate', 0, 'GATE'],
            'Link': ['LinkNodestrings', '_write_link', 'link', 0, 'INTERNAL'],
        }

        # get the correct index for structures if there were no internal ss boundaries
        internal_count -= 1

        used_bc_ids = set()
        pressure_structure_idx = []
        for arc_id, bc_param in struct_ids.items():
            self._cur_arc_id = arc_id
            self._cur_bc_param = bc_param
            bc_id = self._bc_arc_id_to_bc_id[arc_id]
            if bc_id in used_bc_ids:
                continue  # pragma: no cover
            used_bc_ids.add(bc_id)
            struct = self._bc_bc_id_to_structure[bc_id]
            up_id = struct['up'] + self._offset
            down_id = struct['down'] + self._offset
            bc_data_line_str = self._get_bcdata_lines_str()

            bc_type = bc_param.bc_type
            info = structs[bc_param.bc_type]
            # extra work for HY8 link
            if bc_param.bc_type == 'Culvert HY-8' and bc_param.hy8_culvert.simulate_as_link:
                info = structs['Link']
                bc_type = 'Link'

            info[3] += 1  # count for the number of structures for this type
            self._file.write(f'{info[0]} {info[3]} {up_id} {down_id}{bc_data_line_str}\n')

            writer_method = getattr(self, info[1])
            par_cls = getattr(bc_param, info[2])
            writer_method(par_cls, info[3])
            self._write_flow_direction(info[3])

            # don't do this if the structure is culvert or pressure without overtopping
            if bc_type == 'Culvert' or (bc_type == 'Pressure' and not bc_param.pressure.overtopping):
                pass
            else:
                # get data to write to the structure index file
                struct_type = structs[bc_type][4]
                struct_index = structs[bc_type][3]
                if bc_type == 'Pressure':
                    pressure_structure_idx.append((struct['up'], struct['down']))
                else:
                    if struct_type == 'INTERNAL':
                        internal_count += 2
                        struct_index = internal_count
                    arc_structure_idx.append((struct['up'], struct_type, struct_index, ''))
                    arc_structure_idx.append((struct['down'], struct_type, struct_index, ''))
        for item in pressure_structure_idx:
            internal_count += 2
            struct_type = 'INTERNAL'
            struct_index = internal_count
            arc_structure_idx.append((item[0], struct_type, struct_index, ''))
            arc_structure_idx.append((item[1], struct_type, struct_index, ''))

        info = structs['Link']
        for struct in self._bc_3d_structures:
            info[3] += 1  # count for the number of structures for this type
            internal_count += 2
            up_id = struct['up_nodestring_id']
            down_id = struct['down_nodestring_id']
            self._file.write(f'{info[0]} {info[3]} {up_id} {down_id}\n')
            self._write_link(struct['bc_data'].link, info[3])
            arc_structure_idx.append((struct['arc_id'], 'INTERNAL', internal_count, struct['cov_uuid']))

        for monitor in self._monitor_3d_structures:
            arc_structure_idx.append(monitor)

        for bc in self._bc_lines:
            self._file.write(f'{bc}\n')
        struct_plot_file = os.path.join(os.path.dirname(self._file.name), 'structure_index.txt')
        with open(struct_plot_file, 'w') as f:
            f.write('arc_id,structure_type,structure_index,coverage_uuid\n')
            for item in arc_structure_idx:
                f.write(f'{item[0]},{item[1]},{item[2]},{item[3]}\n')

    def _get_bcdata_lines_str(self):
        """Gets the bc id for upstream and downstream bc data lines.

        Returns:
            (:obj:`tuple(str,str)`): the values of the upstream and downstream ids (they can be empty strings)
        """
        up_id = down_id = ''
        lines = self._cur_bc_param.bc_data_lines
        if lines.specify_upstream_bcdata_line or lines.specify_downstream_bcdata_line:
            up_id = 'NO'
            if lines.specify_upstream_bcdata_line:
                if lines.upstream_line_label not in self._bc_data_line_labels:
                    msg = _bc_data_lines_error('upstream', lines.upstream_line_label, self._cur_arc_id)
                    self._logger.error(msg)
                else:
                    up_id = f'{self._bc_data_line_labels[lines.upstream_line_label]}'
            if lines.specify_downstream_bcdata_line:
                if lines.downstream_line_label not in self._bc_data_line_labels:
                    msg = _bc_data_lines_error('downstream', lines.downstream_line_label, self._cur_arc_id)
                    self._logger.error(msg)
                else:
                    down_id = f' {self._bc_data_line_labels[lines.downstream_line_label]}'
        if not up_id and not down_id:
            return ''
        return f' {up_id}{down_id}'

    def _write_flow_direction(self, struct_id):
        """Writes flow direction to the hydro file for a structure.

        Args:
            struct_id (:obj:`int`): id of the structure
        """
        prefix = self._cur_bc_param.bc_type
        if prefix == 'Culvert HY-8':
            prefix = 'HY8'
        elif prefix == 'Pressure':
            prefix = 'PressWeir'
        fd = self._cur_bc_param.flow_direction
        if fd.specify_flow_direction:
            line = f'{prefix}FlowDirection {struct_id} {fd.x_direction} {fd.y_direction}'
            self._bc_lines.append(line)

    def _write_xys(self, df, curve_name='Curve', swap_x_y=False):
        """Write an XY series.

        Args:
            df (:obj:`pandas.DataFrame`): The XY series data
            curve_name (:obj:`str`): Name to assign to the curve
            swap_x_y (:obj:`bool`): flag to swap the x and y data

        Returns:
            (:obj:`str`): Path from the model export location to the xys file
        """
        if not swap_x_y:
            ts = [tuple(r) for r in df.to_numpy()]
        else:
            ts = [(r[1], r[0]) for r in df.to_numpy()]
        self._check_xys(ts)
        return self._hydro_writer.write_xys(ts, curve_name)

    def _check_xys(self, ts):
        """Make sure the xys always increases in time and has no duplicate times.

        Args:
            ts(:obj:`list[tuple]`): List of the XY series time steps
        """
        if not self._logger:
            return
        msg = f'Errors encountered with boundary condition: {self._cur_bc_param.bc_type} on arc: {self._cur_arc_id}.'
        for i in range(1, len(ts)):
            if ts[i][0] == ts[i - 1][0]:  # duplicate time
                self._logger.error(
                    f'{msg}\n'
                    f'Duplicate times found with time value: {ts[i][0]}. SRH-2D does not support '
                    f'the defined time series.'
                )
            elif ts[i][0] < ts[i - 1][0]:  # time is not increasing
                self._logger.error(
                    f'{msg}\n'
                    f'Time must always increase in a time series. Value {ts[i-1][0]} found '
                    f'before value {ts[i][0]}. SRH-2D does not support the defined time series.'
                )

    def _get_pest_inlet_q_string(self, inlet_q):
        """Get the PEST card for an inlet Q BC boundary.

        Args:
            inlet_q (:obj:`BcDataInletQ`): Inlet Q data

        Returns:
            (:obj:`str`): See description
        """
        value = f'{inlet_q.constant_q}'
        for row, _id in enumerate(self._hydro_writer.param_data['params']['id']):
            if _id.startswith('inlet_q'):
                q_idx = int(_id[8:])
                if q_idx == self._cur_arc_id:
                    if self._hydro_writer.param_data['params']['Use'][row]:
                        value = f"${self._hydro_writer.param_data['params']['id'][row]}$"
                    break
        return value

    def _get_sediment_table_file(self, sed_inlet: BcDataSedInflow) -> str:
        """Writes the sediment file using the sediment table file, checks for errors, and returns the file path.

        Args:
            sed_inlet: Sediment inflow data.

        Returns:
            The path.
        """
        if sed_inlet.sediment_table_file_name == '':
            self._logger.error(f'Sediment table for arc {self._cur_arc_id} is not defined.')
            return ''

        sed_table_file = Path(self._hydro_writer.bc_comp_file).parent / sed_inlet.sediment_table_file_name
        df = pd.read_csv(sed_table_file, sep=' ')

        # Check for errors
        df_particle_diameter_threshold = self._hydro_writer._model_control.sediment.particle_diameter_threshold
        if len(df) == 0:
            msg = f'Sediment table for arc {self._cur_arc_id} contains no data.'
            self._logger.error(msg)
        elif len(df.columns) - 1 < len(df_particle_diameter_threshold) - 1:
            msg = (
                f'Sediment table for arc {self._cur_arc_id} has fewer Qs columns than required by the particle '
                'diameter threshold table in Model Control.'
            )
            self._logger.error(msg)
        elif len(df.columns) - 1 > len(df_particle_diameter_threshold) - 1:
            msg = (
                f'Sediment table for arc {self._cur_arc_id} has more Qs columns than required by the particle '
                'diameter threshold table in Model Control.'
            )
            self._logger.warning(msg)

        # Apply scale factor
        df.iloc[:, 1:] *= sed_inlet.scale_factor

        # Write a temp file without the headings. This will later get copied and renamed.
        sed_file = xmf.temp_filename(suffix='.csv')
        df.to_csv(sed_file, sep=' ', header=False, index=False)
        return sed_file

    def _write_inlet_q(self, node_string_id, inlet_q, sed_inlet=None):
        """Write inlet Q bc data to the hydro file.

        Args:
            node_string_id (:obj:`int`): SRH-2D nodestring id for the inlet Q
            inlet_q (:obj:`BcDataInletQ`): Inlet Q data
            sed_inlet (:obj:`BcDataSedInflow`): Sediment inflow data
        """
        sed_curve = ''
        if sed_inlet is not None:
            sed_curve = 'CAPACITY '
            if sed_inlet.sediment_discharge_type != 'Capacity':
                sed_file = ''
                if sed_inlet.sediment_discharge_type == 'File':
                    sed_file = sed_inlet.sediment_file
                elif sed_inlet.sediment_discharge_type == 'Table':
                    sed_file = self._get_sediment_table_file(sed_inlet)

                if not os.path.isfile(sed_file):
                    msg = f'Sediment load file does not exist: {sed_file}\n'
                    self._logger.error(msg)
                else:
                    sed_out_file = f'sed_input_{node_string_id}.qs'
                    sed_out_w_path = os.path.join(os.path.dirname(self._file.name), sed_out_file)
                    xmf.copyfile(sed_file, sed_out_w_path)
                    sed_curve = f'"{sed_out_file}" '

        bc_line = f'IQParams {node_string_id}'
        if inlet_q.discharge_option == 'Constant':
            unit_str = 'EN' if inlet_q.constant_q_units == 'cfs' else 'SI'
            if self._hydro_writer.is_template and self._hydro_writer.param_data:
                value = self._get_pest_inlet_q_string(inlet_q)
            else:
                value = f'{inlet_q.constant_q}'
            bc_line = f'{bc_line} {value} {sed_curve}{unit_str}'
        else:
            filename = self._write_xys(inlet_q.time_series_q)
            unit_str = 'EN' if inlet_q.time_series_q_units == 'hrs -vs- cfs' else 'SI'
            bc_line = f'{bc_line} "{filename}" {sed_curve}{unit_str}'

        self._bc_lines.append(f'{bc_line} {inlet_q.distribution_at_inlet.upper()}')

    def _write_sed_inlet_q(self, node_string_id, inlet_q, sed_inlet):
        """Write sediment inlet Q bc data to the hydro file.

        Args:
            node_string_id (:obj:`int`): SRH-2D nodestring id for the sediment inlet Q
            inlet_q (:obj:`BcDataInletQ`): Inlet Q data
            sed_inlet (:obj:`BcDataSedInflow`): Sediment inlet data

        """
        self._write_inlet_q(node_string_id, inlet_q, sed_inlet)

    def _write_exit_h(self, node_string_id, exit_h):
        """Write exit H bc data to the hydro file.

        Args:
            node_string_id (:obj:`int`): SRH-2D nodestring id for the exit H
            exit_h (:obj:`BcDataExitH`): Exit H data
        """
        if exit_h.water_surface_elevation_option == 'Constant':
            opt_str = 'C'
            unit_str = 'EN' if exit_h.constant_wse_units == 'Feet' else 'SI'
            card = 'EWSParamsC'
            value = exit_h.constant_wse
        elif exit_h.water_surface_elevation_option == 'Time series':
            opt_str = 'TS'
            card = 'EWSParamsTS'
            filename = self._write_xys(exit_h.time_series_wse)
            unit_str = 'EN' if exit_h.time_series_wse_units == 'hrs -vs- feet' else 'SI'
            value = f'"{filename}"'
        elif exit_h.water_surface_elevation_option == 'Rating curve':
            opt_str = 'RC'
            card = 'EWSParamsRC'
            filename = self._write_xys(exit_h.rating_curve, 'Rating_Curve')
            unit_str = 'EN' if exit_h.rating_curve_units == 'cfs -vs- feet' else 'SI'
            value = f'"{filename}"'
        bc_line = f'{card} {node_string_id} {value} {unit_str} {opt_str}'
        self._bc_lines.append(bc_line)

    def _write_exit_q(self, node_string_id, exit_q):
        """Write exit Q bc data to the hydro file.

        Args:
            node_string_id (:obj:`int`): SRH-2D nodestring id for the exit Q
            exit_q (:obj:`BcDataExitQ`): Exit H data
        """
        bc_line = f'EQParams {node_string_id} DISCHARGE'
        if exit_q.discharge_option == 'Constant':
            unit_str = 'EN' if exit_q.constant_q_units == 'cfs' else 'SI'
            bc_line = f'{bc_line} {exit_q.constant_q} {unit_str}'
        elif exit_q.discharge_option == 'Time series':
            filename = self._write_xys(exit_q.time_series_q)
            unit_str = 'EN' if exit_q.time_series_q_units == 'hrs -vs- cfs' else 'SI'
            bc_line = f'{bc_line} "{filename}" {unit_str}'
        self._bc_lines.append(bc_line)

    def _write_inlet_sc(self, node_string_id, inlet_sc):
        """Write inlet sc bc data to the hydro file.

        Args:
            node_string_id (:obj:`int`): SRH-2D nodestring id for the inlet SC
            inlet_sc (:obj:`BcDataInletSc`): Inlet SC data
        """
        if inlet_sc.discharge_q_option == 'Constant':
            val1 = inlet_sc.constant_q
            val2 = inlet_sc.constant_wse
        elif inlet_sc.discharge_q_option == 'Time series':
            filename = self._write_xys(inlet_sc.time_series_q)
            val1 = f'"{filename}"'
            wse = inlet_sc.water_surface_elevation
            if wse.water_elevation_option == 'Time series':
                filename = self._write_xys(wse.time_series_wse)
            elif wse.water_elevation_option == 'Rating curve':
                filename = self._write_xys(wse.rating_curve, 'Rating_Curve')
            val2 = f'"{filename}"'

        unit_str = 'EN' if inlet_sc.discharge_q_units == 'English' else 'SI'
        dist_str = f'{inlet_sc.distribution_at_inlet.upper()}'
        bc_line = f'ISupCrParams {node_string_id} {val1} {val2} {unit_str} {dist_str}'
        self._bc_lines.append(bc_line)

    def _write_sed_inlet_sc(self, node_string_id, inlet_sc, sed_inlet):
        """Write sediment inlet SC bc data to the hydro file.

        Args:
            node_string_id (:obj:`int`): SRH-2D nodestring id for the sediment inlet SC
            inlet_sc (:obj:`BcDataInletSc`): Inlet SC data
            sed_inlet (:obj:`BcDataSedInflow`): Sediment inlet data
        """
        if not self._logger:
            return
        msg = f'Warning encountered with boundary condition: {self._cur_bc_param.bc_type} on arc: ' \
              f'{self._cur_arc_id}.\n{self._cur_bc_param.bc_type} for an SRH-2D sediment simulation is not ' \
              f'supported.\nThis BC will not be written.'
        self._logger.error(msg)
        # Remove this code for now since Alan Z and Scott H suggest that this is not supported and we have no examples.
        # if inlet_sc.discharge_q_option == 'Constant':
        #     val1 = inlet_sc.constant_q
        #     val2 = inlet_sc.constant_wse
        # elif inlet_sc.discharge_q_option == 'Time series':
        #     filename = self._write_xys(inlet_sc.time_series_q)
        #     val1 = f'"{filename}"'
        #     wse = inlet_sc.water_surface_elevation
        #     if wse.water_elevation_option == 'Time series':
        #         filename = self._write_xys(wse.time_series_wse)
        #     elif wse.water_elevation_option == 'Rating curve':
        #         filename = self._write_xys(wse.rating_curve, 'Rating_Curve')
        #     val2 = f'"{filename}"'
        #
        # if sed_inlet.sediment_discharge_type == 'Capacity':
        #     sed_curve = 'CAPACITY'
        # else:
        #     sed_curve = f'"{sed_inlet.sediment_file}"'
        #
        # unit_str = 'EN' if inlet_sc.discharge_q_units == 'English' else 'SI'
        # dist_str = f'{inlet_sc.distribution_at_inlet.upper()}'
        # bc_line = f'ISupCrParams {node_string_id} {val1} {val2} {sed_curve} {unit_str} {dist_str}'
        # self._bc_lines.append(bc_line)

    def _write_wall(self, node_string_id, wall):
        """Write wall bc data to the hydro file.

        Args:
            node_string_id (:obj:`int`): SRH-2D nodestring id for the wall
            wall (:obj:`BcDataWall`): Wall data
        """
        if not wall.extra_wall_roughness:
            return
        bc_line = f'WallRoughness {node_string_id} {wall.roughness}'
        self._bc_lines.append(bc_line)

    def _write_internal_sink(self, node_string_id, internal_sink):
        """Write internal sink bc data to the hydro file.

        Args:
            node_string_id (:obj:`int`): SRH-2D nodestring id for the internal sink
            internal_sink (:obj:`BcDataInternalSink`): Internal Sink data
        """
        if internal_sink.sink_flow_type == 'Constant':
            opt_str = 'CONSTANT'
            unit_str = 'EN' if internal_sink.constant_q_units == 'cfs' else 'SI'
            line2 = f'IntConstantFlow {node_string_id} {internal_sink.constant_q} {unit_str}'
        if internal_sink.sink_flow_type == 'Time series':
            opt_str = 'TIMESERIES'
            filename = self._write_xys(internal_sink.time_series_q)
            unit_str = 'EN' if internal_sink.time_series_q_units == 'hrs -vs- cfs' else 'SI'
            line2 = f'IntTSFlow {node_string_id} "{filename}" {unit_str}'
        elif internal_sink.sink_flow_type == 'Weir':
            opt_str = 'WEIR'
            unit_str = 'EN' if internal_sink.weir_units == 'Feet' else 'SI'
            use_total_head = 0 if internal_sink.weir_use_total_head else 1
            line2 = f'IntWeirFlow {node_string_id} {internal_sink.weir_coeff}' \
                    f' {internal_sink.weir_crest_elevation} {internal_sink.weir_length_across}' \
                    f' {unit_str} {use_total_head}'
        elif internal_sink.sink_flow_type == 'Rating curve':
            opt_str = 'RATING'
            filename = self._write_xys(internal_sink.rating_curve, 'Rating_Curve', swap_x_y=True)
            unit_str = 'EN' if internal_sink.rating_curve_units == 'cfs -vs- feet' else 'SI'
            line2 = f'IntRCFlow {node_string_id} "{filename}" {unit_str}'

        bc_line = f'IntFlowType {node_string_id} {opt_str}'
        self._bc_lines.append(bc_line)
        self._bc_lines.append(line2)

    def _write_hy8_culvert(self, hy8_culvert, structure_id):
        """Write HY8 culvert bc data to the hydro file.

        Args:
            hy8_culvert (:obj:`BcDataCulvertHy8`): Hy8 Culvert data
            structure_id (:obj:`int`): id for the structure
        """
        name = ''
        hy8_culvert.hy8_input_file = self._hy8_file
        guid = hy8_culvert.hy8_crossing_guid
        hy8_names = hy8_culvert.name_guid_dict_from_hy8_file(include_crest_length=True)
        hy8_crest_length = None
        for cross_name, cross_guid in hy8_names.items():
            if guid == cross_guid:
                name = cross_name
                crest_dict = hy8_names.get('##crest_length##', {})
                if guid in crest_dict:
                    hy8_crest_length = crest_dict[guid]
        struct_bc_id = self._bc_arc_id_to_bc_id[self._cur_arc_id]
        arc_ids = [arc_id for arc_id, bc_id in self._bc_arc_id_to_bc_id.items() if bc_id == struct_bc_id]
        if not name:
            msg = f'Error writing HY8 culvert associated with arcs: {arc_ids}.' \
                  f'\nEdit the boundary condition attributes for the arc ids {arc_ids} ' \
                  f'and ensure that a valid HY8 crossing is selected!' \
                  f"\nID missing from HY8 file: '{guid}'." \
                  f'\nIDs in HY8 file (crossing name: ID):\n{str(hy8_names)}'
            raise RuntimeError(msg)
        the_name = name
        if hy8_culvert.simulate_as_link:
            the_name = f'{name}-NoHY8Overtopping'
        self._check_hy8_crossing_name_length(the_name)
        self._check_hy8_crest_length(hy8_crest_length, arc_ids, name, hy8_culvert.simulate_as_link)
        self._check_hy8_invert_with_mesh_elevation(hy8_culvert.tolerance, arc_ids, name, struct_bc_id)

        unit_str = 'EN' if hy8_culvert.units == 'English' else 'SI'
        if not hy8_culvert.simulate_as_link:
            bc_line = f'HY8Params {structure_id} -999 {unit_str} "{the_name}" {guid}'
        else:
            tab_file = xmf.compute_relative_path(os.path.dirname(self._file.name), self._hy8_file)
            tab_file = f'{os.path.splitext(tab_file)[0]}.table'
            bc_line = f'LinkFlowType2 {structure_id} HY8 "{the_name}" "{tab_file}"'
        if hy8_culvert.total_head:
            bc_line = f'{bc_line} HEAD'
        self._bc_lines.append(bc_line)

    def _check_hy8_invert_with_mesh_elevation(
        self, tolerance: float, arc_ids: list[int], name: str, struct_bc_id: int
    ) -> None:
        """Check for inconsistencies between the HY8 invert and the mesh elevations.

        Args:
            tolerance: The HY8 elevation tolerance from Model Control.
            arc_ids: The arc IDs.
            name: Name of the crossing.
            struct_bc_id: ID of the structure boundary condition.
        """
        if not self._bc_arc_id_to_grid_pts:  # This can be False when testing
            return

        hy8_file_dict = hy8_file_dict_from_hy8_file(self._hy8_file)

        # Get the upstream and downstream arc IDs
        struct = self._bc_bc_id_to_structure[struct_bc_id]
        if 'up' in struct and 'down' in struct:
            up_id = struct['up']  # Upstream
            down_id = struct['down']  # Downstream

        for culvert_arc_id in arc_ids:
            if culvert_arc_id not in self._bc_arc_id_to_grid_pts:
                continue

            # Get elevation data of the grid points
            grid_pts = self._bc_arc_id_to_grid_pts[culvert_arc_id]
            elevation_values = grid_pts[2::3]  # Every 3rd value is the z-position(elevation)
            lowest = min(elevation_values)
            highest = max(elevation_values)

            crossing = hy8_file_dict['crossings'][name]
            for culvert in crossing['culverts'].values():
                embed_depth = culvert['EMBEDDEPTH'] / 12  # conversion from inches to ft
                upstream_elevation = culvert['upstream_invert'] + embed_depth
                downstream_elevation = culvert['downstream_invert'] + embed_depth
                rise_diameter = culvert['BARRELDATA']['RISE']

                # hy-8 stores data in ft
                upstream_elevation = self._convert_to_grid_units(upstream_elevation)
                downstream_elevation = self._convert_to_grid_units(downstream_elevation)
                rise_diameter = self._convert_to_grid_units(rise_diameter)

                # Upstream check
                if culvert_arc_id == up_id:
                    if abs(upstream_elevation - lowest) > tolerance:
                        msg = _culvert_elevation_warning_up(upstream_elevation, lowest, tolerance, up_id)
                        self._logger.warning(msg)

                # Downstream check
                elif culvert_arc_id == down_id:
                    if (lowest - downstream_elevation) > tolerance:  # Only warn if the invert is below the mesh
                        msg = _culvert_elevation_warning_down(downstream_elevation, lowest, tolerance, down_id)
                        self._logger.warning(msg)
                    if (highest - lowest) > (rise_diameter * 2 / 3):
                        msg = _culvert_elevation_warning_range(highest, lowest, rise_diameter, down_id)
                        self._logger.warning(msg)

    def _convert_to_grid_units(self, value: float) -> float:
        """Returns the value converted to the units of the grid.

        Args:
            value: A value in ft (HY8 always stores values in feet, even when displaying them as SI units).

        Returns:
            The converted value.
        """
        return _convert_to_grid_units(self._hydro_writer.grid_units, value)

    def _check_hy8_crossing_name_length(self, name):
        """Check the HY8 crossing name length."""
        if len(name) > 36:
            msg = f'Error: HY8 crossing name is too long (max length is 36). Crossing: "{name}" must be changed.'
            if '-NoHY8Overtopping' in name:
                msg += ' When using 2d terrain for overtopping, the "-NoHY8Overtopping" suffix is added automatically.'
            raise RuntimeError(msg)

    def _check_hy8_crest_length(self, hy8_crest_length, arc_ids, name, simulate_as_link):
        """Check the HY8 crest length."""
        if hy8_crest_length is None:
            return
        msg = (
            f'Check HY8 crest length for boundary condition associated with arcs: {arc_ids}.\n'
            f'    HY8 crossing: "{name}" has a crest length of {hy8_crest_length}.'
        )
        if not simulate_as_link:
            lengths = self._hydro_writer.bc_arc_id_to_node_string_length
            ave_length = (lengths[arc_ids[0]] + lengths[arc_ids[1]]) / 2
            factor = hy8_crest_length / ave_length
            if factor > 1.1 or factor < 0.9:
                msg += (
                    f'\n    Arc id {arc_ids[0]} has a node string length of {lengths[arc_ids[0]]}.'
                    f'\n    Arc id {arc_ids[1]} has a node string length of {lengths[arc_ids[1]]}.'
                    '\n    Crest length recommended to be within 10% of average node string length.'
                )
                self._logger.warning(msg)
        # no longer needed because hy8 will produce a flow table with and without overtopping
        # elif hy8_crest_length > 0.1:
        #     msg += '\n    Crest length recommended to be less than 0.1 when using "2d terrain for overtopping".'
        #     self._logger.warning(msg)

    def _write_culvert(self, culvert, structure_id):
        """Write culvert bc data to the hydro file.

        Args:
            culvert (:obj:`BcDataCulvert`): Culvert data
            structure_id (:obj:`int`): id for the structure
        """
        self._logger.warning(
            'Culvert structure encountered in simulation. This feature is not recommended to use and '
            'should be replaced by Culvert-HY8.'
        )
        unit_str = 'EN' if culvert.units == 'Feet' else 'SI'
        entrance_str = '0.7' if culvert.entrance_type == 'mitered' else '-0.5'
        inlet_str = self._inlet_coefficients[culvert.inlet_coefficients]
        type_str = culvert.weir.type.upper()
        bc_line = f'CulvertParams {structure_id} {culvert.invert_elevation} -999 {culvert.barrel_height}' \
                  f' {culvert.barrel_length} {culvert.barrel_area} {culvert.barrel_hydraulic_radius}' \
                  f' {culvert.barrel_slope} {unit_str} {culvert.num_barrels} {entrance_str}' \
                  f' {inlet_str} {culvert.loss_coefficient} {culvert.mannings_n}'
        if culvert.total_head:
            bc_line = f'{bc_line} HEAD'
        self._bc_lines.append(bc_line)
        bc_line = f'CulvertWeirParams {structure_id} {culvert.crest_elevation}' \
                  f' {culvert.length_of_weir_over_culvert} {type_str}'
        if culvert.weir.type == 'User':
            bc_line = f'{bc_line} {culvert.weir.cw} {culvert.weir.a} {culvert.weir.b}'
        self._bc_lines.append(bc_line)

    def _write_weir(self, weir, structure_id):
        """Write weir bc data to the hydro file.

        Args:
            weir (:obj:`BcDataWeir`): Culvert data
            structure_id (:obj:`int`): id for the structure
        """
        unit_str = 'EN' if weir.units == 'Feet' else 'SI'
        bc_line = f'WeirParams {structure_id} {weir.crest_elevation} -999 {weir.length} {unit_str}' \
                  f' {weir.type.type.upper()}'
        if weir.type.type == 'User':
            bc_line = f'{bc_line} {weir.type.cw} {weir.type.a} {weir.type.b}'
        if weir.total_head:
            bc_line = f'{bc_line} HEAD'
        self._bc_lines.append(bc_line)

    def _write_pressure(self, pressure, structure_id):
        """Write pressure bc data to the hydro file.

        Args:
            pressure (:obj:`BcDataPressure`): Pressure data
            structure_id (:obj:`int`): id for the structure
        """
        unit_str = 'EN' if pressure.units == 'Feet' else 'SI'
        type_str = 'FLAT' if pressure.ceiling_type == 'Flat' else 'PARA'
        bc_line = f'PressureParams {structure_id} {type_str} {pressure.upstream_elevation}' \
                  f' {pressure.downstream_elevation} {pressure.roughness} {unit_str}'
        self._bc_lines.append(bc_line)
        overtop_str = '1' if pressure.overtopping else '0'
        bc_line = f'PressOvertop {structure_id} {overtop_str}'
        self._bc_lines.append(bc_line)
        if pressure.overtopping:
            weir_type_str = pressure.weir_type.type.upper()
            user_str = ''
            bc_line = f'PressWeirParams2 {structure_id} {pressure.weir_crest_elevation} {pressure.weir_length} ' \
                      f'{unit_str}'
            if pressure.weir_type.type == 'User':
                user_str = f' {pressure.weir_type.cw} {pressure.weir_type.a} {pressure.weir_type.b}'
            if pressure.total_head:
                user_str = f'{user_str} HEAD'
            bc_line = f'{bc_line} {weir_type_str}{user_str}'
            self._bc_lines.append(bc_line)

    def _write_gate(self, gate, structure_id):
        """Write gate bc data to the hydro file.

        Args:
            gate (:obj:`BcDataGate`): Pressure data
            structure_id (:obj:`int`): id for the structure
        """
        unit_str = 'EN' if gate.units == 'Feet' else 'SI'
        type_str = gate.type.type.upper()
        bc_line = f'GateParams {structure_id} {gate.crest_elevation} -999 {gate.height} {gate.width}' \
                  f' {unit_str} {gate.contract_coefficient} {type_str}'
        if gate.type.type == 'User':
            bc_line = f'{bc_line} {gate.type.cw} {gate.type.a} {gate.type.b}'
        self._bc_lines.append(bc_line)

    def _write_link(self, link, structure_id):
        """Write link bc data to the hydro file.

        Args:
            link (:obj:`BcDataLink`): Link data
            structure_id (:obj:`int`): id for the structure
        """
        # from SRH_PRE
        # (1) DISCHARGE Qvalue UNIT
        # (2) RATING    Fname  UNIT [M_ID]
        # (3) WEIR ZC LW UNIT TYPE [CW] [a b] [HEAD]
        # (4) HY8  CROSSING_NAME  TABLE_NAME  [HEAD]
        if self._cur_bc_param.bc_type == 'Culvert HY-8':
            self._write_hy8_culvert(self._cur_bc_param.hy8_culvert, structure_id)
        else:
            if link.inflow_type == 'Constant':
                unit_str = 'EN' if link.constant_q_units == 'cfs' else 'SI'
                type2_str = f'DISCHARGE {link.constant_q} {unit_str}'
            elif link.inflow_type == 'Time series':
                unit_str = 'EN' if link.time_series_q_units == 'hrs -vs- cfs' else 'SI'
                filename = self._write_xys(link.time_series_q)
                type2_str = f'DISCHARGE "{filename}" {unit_str}'
            elif link.inflow_type == 'Weir':
                unit_str = 'EN' if link.weir.units == 'Feet' else 'SI'
                weir_type = link.weir.type
                type2_str = f'WEIR {link.weir.crest_elevation} {link.weir.length} {unit_str} {weir_type.type.upper()}'
                if link.weir.type.type == 'User':
                    type2_str = f'{type2_str} {weir_type.cw} {weir_type.a} {weir_type.b}'
                if link.weir.total_head:
                    type2_str = f'{type2_str} HEAD'
            elif link.inflow_type == 'Rating curve':
                unit_str = 'EN' if link.rating_curve_units == 'cfs -vs- feet' else 'SI'
                filename = self._write_xys(link.rating_curve, 'Rating_Curve', swap_x_y=True)
                type2_str = f'RATING "{filename}" {unit_str}'
            bc_line = f'LinkFlowType2 {structure_id} {type2_str}'
            self._bc_lines.append(bc_line)
        if link.link_lag_method == 'Specified':
            bc_line = f'LinkLagTimeParameters {structure_id} {link.specified_lag}'
        elif link.link_lag_method == 'Computed':
            unit_str = 'EN' if link.conduit_units == 'Feet' else 'SI'
            bc_line = f'LinkLagTimeParameters {structure_id} {link.conduit_length} {link.conduit_diameter}' \
                      f' {link.conduit_slope} {link.conduit_mannings} {unit_str}'
        self._bc_lines.append(bc_line)

    @staticmethod
    def get_inlet_coefficients():
        """Returns a dict to look up export strings from an option string."""
        objects = {
            'Concrete - Circular - Headwall; square edge':
                '0.3153 2.000 1.2804 0.670',
            'Concrete - Circular - Headwall; grooved edge':
                '0.2509 2.000 0.9394 0.740',
            'Concrete - Circular - Projecting; grooved edge':
                '0.1448 2.000 1.0198 0.690',
            'Cor metal - Circular - Headwall':
                '0.2509 2.000 1.2192 0.690',
            'Cor metal - Circular - Mitered to slope':
                '0.2112 1.330 1.4895 0.750',
            'Cor metal - Circular - Projecting':
                '0.4593 1.500 1.7790 0.540',
            'Concrete - Circular - Beveled ring; 45 deg bevels':
                '0.1379 2.500 0.9651 0.740',
            'Concrete - Circular - Beveled ring; 33.7 deg bevels':
                '0.1379 2.500 0.7817 0.830',
            'Concrete - Rectangular - Wingwalls; 30-75 deg flares; square edge':
                '0.1475 1.000 1.2385 0.810',
            'Concrete - Rectangular - Wingwalls; 90 and 15 deg flares; square edge':
                '0.2242 0.750 1.2868 0.800',
            'Concrete - Rectangular - Wingwalls; 0 deg flares; square edge':
                '0.2242 0.750 1.3608 0.820',
            'Concrete - Rectangular - Wingwalls; 45 deg flares; beveled edge':
                '1.6230 0.667 0.9941 0.800',
            'Concrete - Rectangular - Wingwalls; 18-33.7 deg flare; beveled edge':
                '1.5466 0.667 0.8010 0.830',
            'Concrete - Rectangular - Headwall; 3/4 in chamfers':
                '1.6389 0.667 1.2064 0.790',
            'Concrete - Rectangular - Headwall; 45 deg bevels':
                '1.5752 0.667 1.0101 0.820',
            'Concrete - Rectangular - Headwall; 33.7 deg bevels':
                '1.5466 0.667 0.8107 0.865',
            'Concrete - Rectangular - Headwall; 45 deg skew; 3/4 in chamfers':
                '1.6611 0.667 1.2932 0.730',
            'Concrete - Rectangular - Headwall; 30 deg skew; 3/4 in chamfers':
                '1.6961 0.667 1.3672 0.705',
            'Concrete - Rectangular - Headwall; 15 deg skew; 3/4 in chamfers':
                '1.7343 0.667 1.4493 0.680',
            'Concrete - Rectangular - Headwall; 10-45 deg skew; 45 deg bevels':
                '1.5848 0.667 1.0520 0.750',
            'Concrete - Rectangular - Wingwalls; non-offset 45 deg flares; 3/4 in chamfers':
                '1.5816 0.667 1.0906 0.803',  # noqa
            'Concrete - Rectangular - Wingwalls; non-offset 18.4 deg flares; 3/4 in chamfers':
                '1.5689 0.667 1.1613 0.806',  # noqa
            'Concrete - Rectangular - Wingwalls; non-offset 18.4 deg flares; 30 deg skewed barrel':
                '1.5752 0.667 1.2418 0.710',  # noqa
            'Concrete - Rectangular - Wingwalls; offset 45 deg flares; beveled top edge':
                '1.5816 0.667 0.9715 0.835',
            'Concrete - Rectangular - Wingwalls; offset 33.7 deg flares; beveled top edge':
                '1.5752 0.667 0.8107 0.881',
            'Concrete - Rectangular - Wingwalls; offset 18.4 deg flares; beveled top edge':
                '1.5689 0.667 0.7303 0.887',
            'Cor metal - Rectangular - Headwall':
                '0.2670 2.000 1.2192 0.690',
            'Cor metal - Rectangular - Projecting; thick wall':
                '0.3023 1.750 1.3479 0.640',
            'Cor metal - Rectangular - Projecting; thin wall':
                '0.4593 1.500 1.5956 0.570',
            'Concrete - Horizontal ellipse - Headwall; square edge':
                '0.3217 2.000 1.2804 0.670',
            'Concrete - Horizontal ellipse - Headwall; grooved edge':
                '0.1379 2.500 0.9394 0.740',
            'Concrete - Horizontal ellipse - Projecting; grooved edge':
                '0.1448 2.000 1.0198 0.690',
            'Cor metal - Pipe arch (18in corner) - Headwall':
                '0.2670 2.000 1.5956 0.570',
            'Cor metal - Pipe arch (18in corner) - Mitered to slope':
                '0.1702 1.000 1.4895 0.750',
            'Cor metal - Pipe arch (18in corner) - Projecting':
                '0.4593 1.500 1.5956 0.530',
            'Structural plate - Pipe arch (18in corner) - Projecting':
                '0.3998 1.500 1.5667 0.550',
            'Structural plate - Pipe arch (18in corner) - Headwall; square edge':
                '0.2799 2.000 1.1613 0.660',
            'Structural plate - Pipe arch (18in corner) - Headwall; beveled edge':
                '0.0965 2.000 0.8493 0.750',
            'Structural plate - Pipe arch (31in corner) - Projecting':
                '0.3998 1.500 1.5667 0.550',
            'Structural plate - Pipe arch (31in corner) - Headwall; square edge':
                '0.2799 2.000 1.1613 0.660',
            'Structural plate - Pipe arch (31in corner) - Headwall; beveled edge':
                '0.0965 2.000 0.8493 0.750',
            'Cor metal - Arch - Headwall':
                '0.2670 2.000 1.2192 0.690',
            'Cor metal - Arch - Mitered to slope':
                '0.9651 2.000 1.4895 0.750',
            'Cor metal - Arch - Projecting':
                '0.4593 1.500 1.5956 0.570',
            'Concrete - Circular - Tapered throat':
                '1.3991 0.555 0.6305 0.890',
            'Cor metal - Circular - Tapered throat':
                '1.5760 0.640 0.9297 0.900',
            'Concrete - Rectangular - Tapered throat':
                '1.5116 0.667 0.5758 0.970',
        }
        return objects
