"""Manages writing the SRH-2D control file."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.core.filesystem import filesystem

# 4. Local modules
from xms.srh.components.parameters.parameters_manager import ParametersManager
from xms.srh.file_io.hydro_bc_writer import HydroBcWriter
from xms.srh.file_io.hydro_model_control_writer import HydroModelControlWriter


class HydroWriter:
    """Writes SRH-2D geom file."""
    def __init__(
        self,
        file_name,
        model_control,
        file_list,
        logger,
        main_file='',
        is_template=False,
        grid_units='GridUnit "FOOT"'
    ):
        """Constructor.

        Args:
            file_name (:obj:`str`): Path to the output *.srhhydro file
            model_control (:obj:`ModelControl`): The simulation data
            file_list (:obj:`list[str]`): Additional files that need to be exported
            logger (:obj:`logging.Logger`): Logger to use
            main_file (:obj:`str`): Path to simulation component main file
            is_template (:obj:`bool`): True if writing a template
            grid_units (:obj:`str`): The grid units
        """
        self._file_name = file_name
        self._model_control = model_control
        self._file_list = file_list
        self._num_xy_files_written = 0  # track the number of curves we have written
        self.sediment_on = model_control.enable_sediment
        self.logger = logger
        self.is_template = is_template
        self.grid_units = grid_units
        self.param_data = ParametersManager.read_parameter_file(main_file)

        # these are set outside of this class if the data exists
        self.sed_material_data = None
        self.obstruction_decks = None
        self.obstruction_piers = None
        self.num_monitor_lines = 0
        self.materials_manning = None
        self.materials_bed_shear = None
        self.bc_data = None
        self.bc_arc_id_to_comp_id = None
        self.bc_arc_id_to_bc_id = None
        self.bc_arc_id_to_grid_pts = None
        self.bc_arc_id_to_node_string_length = None
        self.bc_bc_id_to_structure = None
        self.bc_hy8_file = None
        self.bc_3d_structures = []
        self.monitor_3d_structures = []
        self.bc_comp_file = None  # BcComponent main file
        self.restart_file = None
        self.run_name = None

    def write(self, run_name=None):
        """Write the *.srhhydro file.

        Args:
            run_name (:obj:`str`): Name of the run if this is a parameter run model
        """
        self.run_name = run_name  # Non-empty subfolder run name if doing parameters
        with open(self._file_name, 'w') as file:
            if self.is_template:
                file.write('ptf $\n')
            file.write('SRHHYDRO 30\n')
            self._write_model_control(file)
            self._write_materials(file)
            self._write_bcs(file)
            self._write_obstructions(file)

    def _write_model_control(self, file):
        """Write the materials data to the hydro file.

        Args:
            file (:obj:`file`): The output hydro file
        """
        writer = HydroModelControlWriter(
            file, self._model_control, self._file_list, grid_units=self.grid_units, logger=self.logger
        )
        writer.write(self.run_name)

    def _get_pest_material_string(self, index):
        """Get the PEST material string for a specified material.

        Args:
            index (:obj:`int`): Index of the material in the materials list

        Returns:
            (:obj:`str`): See description
        """
        value = f'{self.materials_manning[index]}'
        for row, _id in enumerate(self.param_data['params']['id']):
            if _id.startswith('mat'):
                mat_idx = int(_id[4:])
                if mat_idx == index:
                    if self.param_data['params']['Use'][row]:
                        value = f"${self.param_data['params']['id'][row]}$"
                    break
        return value

    def _write_materials(self, file):
        """Write the materials data to the hydro file.

        Args:
            file: The output hydro file stream
        """
        if self.materials_manning:
            bed_shear = ''
            for i in range(len(self.materials_manning)):
                if self.materials_bed_shear:
                    bed_shear = f' {self.materials_bed_shear[i]}'
                if type(self.materials_manning[i]) is list:
                    curve_name = '//ManningsN 30'
                    value = f'"{self.write_xys(self.materials_manning[i], curve_name=curve_name)}"'
                else:
                    if self.is_template and self.param_data:
                        value = self._get_pest_material_string(i)
                    else:
                        value = f'{self.materials_manning[i]}'
                file.write(f'ManningsN {i} {value}{bed_shear}\n')
        if self.sed_material_data:
            for line in self.sed_material_data:
                if type(line) is str:
                    file.write(f'{line}\n')
                else:
                    mat_id = line[0]
                    lay_id = line[1]
                    curve = line[2]
                    mat_name = f'ID: {mat_id}'
                    if len(line) > 3:
                        mat_name = line[3]
                    self._check_sed_gradation_curve(curve, mat_name, lay_id)
                    curve_file = self.write_xys(curve, curve_name='//SEDGRAD 30')
                    file.write(f'BedSedComposition {mat_id} {lay_id} GRADATION "{curve_file}"\n')

    def _check_sed_gradation_curve(self, curve, mat_name, lay_id):
        """Perform error checking on the sediment gradation curve.

        Args:
            curve (:obj:`list[x,y]`): list of x, y values
            mat_name (:obj:`str`): material name
            lay_id (:obj:`int`): layer id for the sediment layer
        """
        msg = f'Material "{mat_name}" has an invalid sediment gradation curve for layer: {lay_id}. '
        if len(curve) < 2:
            msg_1 = msg + 'The curve has less than 2 points. All sediment gradation curves must have at least 2 points.'
            self.logger.error(msg_1)
        if curve[-1][1] != 100.0:
            msg_1 = msg + 'The percent finer values should end at 100.0. SRH-Pre will set the final value to 100.0.'
            self.logger.warning(msg_1)
        x_vals = [t[0] for t in curve]
        y_vals = [t[1] for t in curve]
        x_sort = sorted(x_vals.copy())
        y_sort = sorted(y_vals.copy())
        if x_vals != x_sort or y_vals != y_sort:
            msg_1 = msg + 'The curve must have increasing values for "particle diameter" and "% finer".'
            self.logger.error(msg_1)

    def _write_bcs(self, file):
        """Write the bc data to the hydro file.

        Args:
            file: The output hydro file stream
        """
        #  write hy8 if needed
        if self.bc_data:
            for _, bc in self.bc_data.items():
                if bc.bc_type == 'Culvert HY-8':
                    hy8_file = 'culvert.hy8'
                    if self.bc_hy8_file:
                        hy8_file = filesystem.compute_relative_path(os.path.dirname(file.name), self.bc_hy8_file)
                    file.write(f'HY8File "{hy8_file}"\n')
                    # if self.run_name:  # Needs to be relative if a parameter run
                    #     file.write(f'HY8File "./{self.run_name}/culvert.hy8"\n')
                    # else:
                    #     file.write('HY8File "culvert.hy8"\n')
                    break

        for i in range(self.num_monitor_lines):
            file.write(f'BC {i+1} MONITORING\n')

        if self.bc_data:
            bc_writer = HydroBcWriter(file, self)
            bc_writer.write()

    def _write_obstructions(self, file):
        """Write the obstruction data to the hydro file.

        Args:
            file: The output hydro file stream
        """
        if self.obstruction_decks:
            file.write(f'NumDeckObstruction {len(self.obstruction_decks)}\n')
        if self.obstruction_piers:
            file.write(f'NumPierObstruction {len(self.obstruction_piers)}\n')

        if self.obstruction_decks:
            for item in self.obstruction_decks:
                file.write(f'{item}\n')
        if self.obstruction_piers:
            for item in self.obstruction_piers:
                file.write(f'{item}\n')

    def write_xys(self, xys_list, curve_name='Curve'):
        """Write the xys file and return the file name.

        Args:
            xys_list (:obj:`list`): The list of tuples (x, y)
            curve_name (:obj:`str`): Curve name or 'Rating Curve'. First line of file.

        Returns:
            (:obj:`str`): Path from the model export location to the xys file
        """
        path = os.sep.join(self._file_name.split(os.sep)[:-1])
        self._num_xy_files_written += 1
        filename = os.path.join(path, f'curve_{self._num_xy_files_written}.xys')
        with open(filename, 'w') as file:
            file.write(f'{curve_name} {self._num_xy_files_written}\n')
            for item in xys_list:
                file.write(f'{item[0]} {item[1]}\n')
        return os.path.basename(filename)
