"""Generate and SMS SRH-2D project summary report."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import collections
from datetime import datetime
import logging
import math
import os
import sys

# 2. Third party modules
from jinja2 import Environment, FileSystemLoader, select_autoescape
import numpy as np
from PySide2.QtCore import QThread, Signal

# 3. Aquaveo modules
from xms.api.tree import tree_util
from xms.constraint import read_grid_from_file
from xms.core.filesystem import filesystem
from xms.coverage.grid.grid_cell_to_polygon_coverage_builder import GridCellToPolygonCoverageBuilder
from xms.data_objects.parameters import FilterLocation
from xms.gdal.utilities.gdal_utils import wkt_to_sr
from xms.guipy.dialogs.process_feedback_dlg import ProcessFeedbackDlg
from xms.guipy.notes import Notes
from xms.guipy.validators.number_corrector import NumberCorrector  # noqa AQU103

# 4. Local modules
from xms.srh.file_io.report import plots
from xms.srh.file_io.report import report_util
from xms.srh.file_io.report.bc_coverage_looper import BcCoverageLooper
from xms.srh.file_io.report.bridge_coverage_looper import BridgeCoverageLooper
from xms.srh.file_io.report.material_coverage_looper import MaterialCoverageLooper
from xms.srh.file_io.report.mesh_quality_reporter import MeshQualityReporter, QualityMeasure
from xms.srh.file_io.report.monitor_coverage_looper import MonitorCoverageLooper
from xms.srh.file_io.report.obstruction_coverage_looper import ObstructionCoverageLooper
from xms.srh.file_io.report.plots import SummaryReportPlotter
from xms.srh.file_io.report.structure_coverage_looper import StructureCoverageLooper

MeshBoundaryTuple = collections.namedtuple('MeshBoundary', 'multi_polys ugrid_points')
BIGLY = 1.5e6  # noqa If we have more than this many points, we skip some stuff


class SummaryReportGeneratorWorkerThread(QThread):
    """Worker thread for the SummaryReportGenerator."""

    processing_finished = Signal()

    def __init__(self, query, report_dialog_data, testing):  # pragma: no cover
        """Construct the worker.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with SMS
            report_dialog_data (:obj:`SummaryReportDialogData`): Data entered in the Summary Report dialog.
            testing (:obj:`bool`): If True, the date is hardwired to always be the same.
        """
        super().__init__()
        self._query = query
        self._report_dialog_data = report_dialog_data
        self._testing = testing
        self.file = ''  # The file that gets created

    def run(self):  # pragma: no cover
        """Runs the SummaryReportGenerator ."""
        generator = SummaryReportGenerator(
            query=self._query, report_dialog_data=self._report_dialog_data, testing=self._testing
        )
        self.file = generator.generate()
        self.processing_finished.emit()


class SummaryReportGenerator:
    """Class to generate the SRH project summary report."""

    log_name = 'Summary Report'

    def __init__(self, query, report_dialog_data, testing):  # pragma: no cover
        """Initializes the class.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with SMS
            report_dialog_data (:obj:`SummaryReportDialogData`): Data entered in the Summary Report dialog.
            testing (:obj:`bool`): If True, the date is hardwired to always be the same.
        """
        self._query = query
        self._report_dialog_data = report_dialog_data
        self._testing = testing

        self._logger = logging.getLogger(self.log_name)

        # Data dict used to fill in html template using jinja
        self._report_jinja = {}  # The whole report
        self._scatter_jinja = {}
        self._meshes_jinja = {}
        self._bcs_jinja = {}
        self._monitor_coverages_jinja = {}
        self._obstructions_jinja = {}
        self._bridge_jinja = {}
        self._structure_jinja = {}
        self._material_jinja = {}

        self._project_path = self._query.xms_project_path if self._query else ''
        self._horizontal_datum = 'None'
        self._vertical_datum = 'None'
        self._notes_db = Notes()
        self._report_dir = ''  # Path to directory where report files will be created
        self._mesh_boundaries = {}  # Boundary of each mesh. key = mesh uuid, value = MeshBoundaryTuple
        self._bc_coverage_meshes = {}  # Dict of BC coverage uuids and their associated mesh uuids
        self._monitor_coverage_meshes = {}  # Dict of monitor coverage uuids and their associated mesh uuids

    def run(self, win_cont, feedback=True):  # pragma: no cover
        """Generates the project summary report.

        Args:
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.
            feedback (:obj:`bool`): True to show the feedback dialog.

        Returns:
            The filepath of the html file created.
        """
        if feedback:
            worker = SummaryReportGeneratorWorkerThread(
                query=self._query, report_dialog_data=self._report_dialog_data, testing=self._testing
            )
            display_text = {
                'title': 'SRH-2D Summary Report',
                'working_prompt': 'Creating SRH-2D Summary Report...',
                'error_prompt': 'Error(s) encountered while creating the report.',
                'warning_prompt': 'Warning(s) encountered while creating the report.',
                'success_prompt': 'Successfully created the report.',
                'note': '',
                'auto_load': 'Close this dialog automatically when report is finished.'
            }
            # debug_util.ensure_qapplication_exists()
            feedback_dlg = ProcessFeedbackDlg(
                display_text=display_text, logger_name=self.log_name, worker=worker, parent=win_cont
            )
            feedback_dlg.exec()
            return worker.file
        else:
            generator = SummaryReportGenerator(
                query=self._query, report_dialog_data=self._report_dialog_data, testing=self._testing
            )
            return generator.generate()

    def generate(self):  # pragma: no cover
        """Generates the project summary report.

        Returns:
            The filepath of the html file created.
        """
        self._make_report_directory()
        self._store_datums()

        self._fill_mesh_summary()  # Do meshes first to get mesh boundaries
        self._fill_simulation_summary()  # Do 1st though it's later in the report. Stores things we need elsewhere.
        self._fill_project_name_and_date()
        self._fill_project_summary()
        self._fill_versions()
        self._fill_project_datum()
        self._fill_terrain_data()
        self._fill_summary_of_boundary_conditions()
        self._fill_summary_of_monitor_coverages()
        self._fill_summary_of_obstructions()
        self._fill_summary_of_bridges()
        self._fill_structures_jinja()
        self._fill_materials_roughness_summary()
        self._fill_calibration_summary_results()

        html = self._render()
        filename = self._create_file(html)
        return filename

    def _store_datums(self):  # pragma: no cover
        """Gets some project level data such as display projection."""
        self._logger.info('Getting project projection.')
        try:
            self._horizontal_datum, self._vertical_datum = datums_from_query(self._query)
        except:  # noqa
            raise RuntimeError('Error getting project data.')

    def _fill_project_name_and_date(self):  # pragma: no cover
        """Fills in data for the 'Project filename and date' section."""
        self._logger.info('Getting project file name and report date.')
        self._report_jinja['project_filename'] = os.path.basename(self._project_path)
        if not self._testing:
            self._report_jinja['report_date'] = datetime.now().strftime('%d %B %Y %H:%M:%S')
        else:
            self._report_jinja['report_date'] = '03 December 2020 08:30:50'

    def _fill_project_summary(self):  # pragma: no cover
        """Fills in data for the 'Project summary ...' section."""
        self._logger.info('Getting project summary information.')
        project_name = os.path.splitext(os.path.basename(self._project_path))[0]
        project_summary_jinja = {
            'name': f'{project_name}',
            'river': self._report_dialog_data.river,
            'purpose': self._report_dialog_data.project_purpose,
            'model_developer': self._report_dialog_data.model_developer_name,
            'terrain_source': self._report_dialog_data.terrain_source,
            'bathymetry_source': self._report_dialog_data.bathymetry_source,
            'additional_survey_source': self._report_dialog_data.additional_survey_source
        }

        self._report_jinja['project_summary'] = project_summary_jinja

    def _fill_versions(self):  # pragma: no cover
        """Fills in data for the 'Versions of software' section."""
        self._logger.info('Getting software versions.')
        sms_version = os.environ.get('XMS_PYTHON_APP_VERSION', 'unknown')
        self._report_jinja['sms_version'] = sms_version

        # import pkg_resources
        # self._report_jinja['srh_version'] = pkg_resources.get_distribution("srh2d_exe").version
        # This is now done in self._fill_results_summary_by_simulation() because we get the version
        # from the OUT.dat file.

    def _fill_project_datum(self):  # pragma: no cover
        """Fills in data for the 'Project datum' section."""
        self._logger.info('Getting horizontal and vertical datums.')
        self._report_jinja['horizontal_datum'] = self._horizontal_datum
        self._report_jinja['vertical_datum'] = self._vertical_datum

    @staticmethod
    def get_scatter_jinja(
        scatter_name, scatter_uuid, co_grid, projection, notes_db, report_dir, logger
    ):  # pragma: no cover
        """Returns the jinja dict for the scatter set.

        Args:
            scatter_name (:obj:`str`): Name of the scatter set.
            scatter_uuid (:obj:`str`): Uuid string of the scatter tree item.
            co_grid (:obj:`xms.constraint.ugrid_2d.UGrid2d`): The constrained UGrid.
            projection (:obj:`xms.data_objects.parameters.Projection`): The projection.
            notes_db (:obj:`Notes`): Notes object.
            report_dir (:obj:`str`): Path to directory where report files are saved.
            logger(:obj:`logger`): The logger.

        Returns:
            (:obj:`dict`): jinja dict for scatter set.
        """
        ugrid = co_grid.ugrid
        scatter_jinja = {'name': scatter_name, 'point_count': str(ugrid.point_count)}
        extents = ugrid.extents
        scatter_jinja['min_elevation'] = fmt(extents[0][2])
        scatter_jinja['max_elevation'] = fmt(extents[1][2])
        # scatter_jinja['projection'] = projection.get_projection_name()
        scatter_jinja['projection'] = projection.well_known_text
        average_point_spacing = get_average_edge_length(ugrid) if ugrid.point_count < BIGLY else -1.0
        if average_point_spacing != -1.0:
            scatter_jinja['average_point_spacing'] = fmt(average_point_spacing)
        else:
            scatter_jinja['average_point_spacing'] = 'Unknown'
        report_util.add_object_notes(notes_db, scatter_uuid, scatter_jinja)
        if logger:
            logger.info('Creating plot of scatter set geometry.')
        if ugrid.point_count < BIGLY:
            scatter_jinja['plot'] = plots.plot_scatter(ugrid, projection, scatter_name, report_dir)
        else:
            scatter_jinja['plot'] = ''
        return scatter_jinja

    def _fill_summary_of_scatter_sets(self):  # pragma: no cover
        """Fills in data for the 'Summary of scatter sets' section."""
        self._scatter_jinja['scatter_sets'] = []

        # Find 'Scatter Data'
        scatter_data_node = tree_util.child_from_name(self._query.project_tree, 'Scatter Data')
        if not scatter_data_node:
            return

        # Iterate through all scatter sets
        for child in scatter_data_node.children:
            self._logger.info(f'Getting data for scatter set "{child.name}".')
            geom = self._query.item_with_uuid(child.uuid)
            if geom:
                co_grid = read_grid_from_file(geom.cogrid_file)
                scatter_jinja = self.get_scatter_jinja(
                    child.name, child.uuid, co_grid, geom.projection, self._notes_db, self._report_dir, self._logger
                )
                self._scatter_jinja['scatter_sets'].append(scatter_jinja)

    def _fill_raster_data_sets(self):  # pragma: no cover
        """Fills in data for the 'Raster data set' section."""
        pass

    def _fill_terrain_data(self):  # pragma: no cover
        """Fills in data for the 'Terrain Data' section."""
        self._fill_summary_of_scatter_sets()
        self._fill_raster_data_sets()

    @staticmethod
    def _store_mesh_boundary(co_grid, projection, mesh_uuid, mesh_boundaries):  # pragma: no cover
        """Stores the mesh boundary for later use.

        Args:
            co_grid (:obj:`xms.constraint.ugrid_2d.UGrid2d`): The UGrid.
            projection (:obj:`xms.data_objects.parameters.Projection`): Projection.
            mesh_uuid (:obj:`str`): UUID of the mesh.
            mesh_boundaries (:obj:`dict`): Dict of mesh uuid and MeshBoundaryTuple
        """
        cell_materials = [1] * co_grid.ugrid.cell_count
        cov_builder = GridCellToPolygonCoverageBuilder(co_grid, cell_materials, projection, '')
        boundary_polygons = cov_builder.find_polygons()
        mesh_boundaries[mesh_uuid] = MeshBoundaryTuple(
            multi_polys=boundary_polygons, ugrid_points=co_grid.ugrid.locations
        )

    @staticmethod
    def get_mesh_jinja(
        mesh_name, mesh_uuid, co_grid, projection, notes_db, report_dir, logger, mesh_boundaries
    ):  # pragma: no cover
        """Returns the jinja dict for the mesh.

        Args:
            mesh_name (:obj:`str`): Name of the mesh.
            mesh_uuid (:obj:`str`): Uuid string of the mesh tree item.
            co_grid (:obj:`xms.constraint.ugrid_2d.UGrid2d`): The constrained UGrid.
            projection (:obj:`xms.data_objects.parameters.Projection`): The projection.
            notes_db (:obj:`Notes`): Notes object.
            report_dir (:obj:`str`): Path to directory where report files are saved.
            logger(:obj:`logger`): The logger.
            mesh_boundaries (:obj:`dict`): Dict of MeshBoundaryTuple

        Returns:
            (:obj:`dict`): jinja dict for mesh.
        """
        ugrid = co_grid.ugrid
        mesh_jinja = {'name': mesh_name, 'node_count': str(ugrid.point_count), 'element_count': str(ugrid.cell_count)}

        # Do extents
        mn, mx = ugrid.extents
        mesh_jinja['x_size'] = fmt(mx[0] - mn[0])
        mesh_jinja['y_size'] = fmt(mx[1] - mn[1])

        # Get largest and smallest element sizes
        smallest, largest = get_smallest_and_largest_element_sizes(ugrid)
        mesh_jinja['smallest_element_size'] = fmt(smallest)
        mesh_jinja['largest_element_size'] = fmt(largest)

        report_util.add_object_notes(notes_db, mesh_uuid, mesh_jinja)
        SummaryReportGenerator._store_mesh_boundary(co_grid, projection, mesh_uuid, mesh_boundaries)

        # Mesh plot
        if logger:
            logger.info('Creating plot of mesh geometry.')
        mesh_jinja['plot'] = plots.plot_mesh(ugrid, projection, mesh_name, report_dir)

        # Mesh quality report
        if logger:
            logger.info('Creating mesh quality report.')
        generator = MeshQualityReporter()
        mesh_jinja['arr_plot'] = generator.get_html_plot_files(
            ugrid, projection, [QualityMeasure.Q_ALS], mesh_name, report_dir
        )
        return mesh_jinja

    def _fill_mesh_summary(self):  # pragma: no cover
        """Fills in data for the 'Mesh Summary' section."""
        self._meshes_jinja['meshes'] = []

        # Find 'Mesh Data'
        mesh_data = tree_util.child_from_name(self._query.project_tree, 'Mesh Data')
        if not mesh_data:
            return

        # Iterate through all meshes
        for child in mesh_data.children:
            self._logger.info(f'Getting data for mesh "{child.name}".')
            geom = self._query.item_with_uuid(child.uuid)
            if geom:
                co_grid = read_grid_from_file(geom.cogrid_file)
                mesh_jinja = self.get_mesh_jinja(
                    child.name, child.uuid, co_grid, geom.projection, self._notes_db, self._report_dir, self._logger,
                    self._mesh_boundaries
                )
                self._meshes_jinja['meshes'].append(mesh_jinja)

    def _fill_summary_of_boundary_conditions(self):  # pragma: no cover
        """Fills in data for the 'Summary of boundary conditions' section."""
        coverage_looper = BcCoverageLooper(
            self._notes_db, self._report_dir, self._query, self._mesh_boundaries, self._bc_coverage_meshes, self._logger
        )
        self._bcs_jinja['bc_coverages'] = coverage_looper.visit_coverages()

    def _fill_summary_of_monitor_coverages(self):  # pragma: no cover
        """Fills in data for the 'Summary of monitor coverages' section."""
        coverage_looper = MonitorCoverageLooper(
            self._notes_db, self._report_dir, self._query, self._mesh_boundaries, self._monitor_coverage_meshes,
            self._logger
        )
        self._monitor_coverages_jinja['monitor_coverages'] = coverage_looper.visit_coverages()

    def _fill_summary_of_obstructions(self):  # pragma: no cover
        """Fills in data for the 'Summary of hydraulic structures' section."""
        coverage_looper = ObstructionCoverageLooper(self._notes_db, self._report_dir, self._query, self._logger)
        self._obstructions_jinja['obstruction_coverages'] = coverage_looper.visit_coverages()

    def _fill_summary_of_bridges(self):  # pragma: no cover
        """Fills in data for the 'Summary of bridges' section."""
        coverage_looper = BridgeCoverageLooper(self._notes_db, self._report_dir, self._query, self._logger)
        self._bridge_jinja['bridge_coverages'] = coverage_looper.visit_coverages()

    def _fill_structures_jinja(self):
        """Fills in data for the 'Structures map' section."""
        if self._logger:
            self._logger.info('Creating map of structures.')

        coverage_looper = StructureCoverageLooper(self._notes_db, self._report_dir, self._query, self._logger)
        structures_list = coverage_looper.visit_coverages()
        if structures_list:
            plot_file = plots.plot_structure(structures_list, self._query.display_projection, self._report_dir)
            self._structure_jinja['structures_plots'] = [{'plot': plot_file}]

    def _fill_materials_roughness_summary(self):  # pragma: no cover
        """Fills in data for the 'Materials roughness summary' section."""
        coverage_looper = MaterialCoverageLooper(self._notes_db, self._report_dir, self._query, self._logger)
        self._material_jinja['material_coverages'] = coverage_looper.visit_coverages()

    def _fill_simulation_summary(self):  # pragma: no cover
        """Fills in data for the 'Simulation Summary' section."""
        self._report_jinja['simulations'] = []

        # Find 'SRH-2D Simulations'
        srh_sims_node = self._find_srh_sims_node()
        if not srh_sims_node:
            return

        # Iterate through all SRH-2D simulations
        for sim_node in srh_sims_node.children:
            sim_component = self._sim_component_from_sim_node(sim_node)

            # Store all jinja stuff in a dict
            sim_jinja = {'name': sim_node.name}

            self._logger.info(f'Getting simulation summary data for "{sim_node.name}".')

            self._fill_model_control_data(sim_component, sim_jinja)

            # Go through all the children of this simulation getting stuff for jinja
            data = {
                'bc_coverage_uuid': None,
                'monitor_coverage_uuid': None,
                'coverage': None,
                'mesh_uuid': None,
                'results_jinja': None,
                'datasets_jinja': []  # List of dicts for each data set
            }
            for sim_child in sim_node.children:
                self._visit_tree_node(sim_child, sim_jinja, data)

            # Associate the BC coverage with the mesh
            if data['mesh_uuid'] and data['bc_coverage_uuid']:
                self._bc_coverage_meshes[data['bc_coverage_uuid']] = data['mesh_uuid']
            if data['mesh_uuid'] and data['monitor_coverage_uuid']:
                self._monitor_coverage_meshes[data['monitor_coverage_uuid']] = data['mesh_uuid']

            # Get _INF.dat file for plots
            inf_file = self._path_to_dat_file(sim_component, sim_node.name, 'INF')
            plotter = SummaryReportPlotter()

            # Add the monitor coverage plot either with or without the mesh boundary
            if 'monitor_coverage' in sim_jinja:
                coverage = data.get('coverage')
                if not coverage:
                    self._logger.error('Monitor coverage indicated but no coverage set.')
                    return
                self._logger.info(f'Creating plot of monitor coverage "{coverage.name}".')
                mesh_boundary = self._mesh_boundaries.get(data['mesh_uuid'])
                sim_jinja['monitor_coverage']['plot'] = plotter.plot_monitor_coverage(
                    coverage, mesh_boundary, self._report_dir
                )

                # Solution plots
                if os.path.isfile(inf_file):
                    sim_jinja['monitor_coverage']['plot_points_wse'] =\
                        plotter.plot_monitor_points_wse(inf_file, self._report_dir)
                    sim_jinja['monitor_coverage']['plot_lines_q'] =\
                        plotter.plot_monitor_lines_q(inf_file, self._report_dir)
                    if sim_component.data.enable_sediment:
                        sim_jinja['monitor_coverage']['plot_points_z'] = \
                            plotter.plot_monitor_points_z(inf_file, self._report_dir)
                        sim_jinja['monitor_coverage']['plot_lines_qs'] = \
                            plotter.plot_monitor_lines_qs(inf_file, self._report_dir)

            # Solution plots
            if os.path.isfile(inf_file):
                sim_jinja['plot_net_q_inlet_q'] = plotter.plot_net_q_inlet_q(inf_file, self._report_dir)
                sim_jinja['plot_wet_elements'] = plotter.plot_wet_elements(inf_file, self._report_dir)
                sim_jinja['plot_mass_balance'] = plotter.plot_mass_balance(inf_file, self._report_dir)

            sim_jinja['cpu_time'] = self._get_cpu_time(sim_component, sim_node.name)

            # Results
            if data['datasets_jinja']:
                srh_version = self._get_srh_version(sim_component, sim_node.name)
                results_jinja = {
                    'sim_name': sim_node.name,
                    'datasets': data['datasets_jinja'],
                    'srh_version': srh_version
                }
                sim_jinja['results'] = results_jinja

            self._report_jinja['simulations'].append(sim_jinja)

    def _visit_tree_node(self, node, sim_jinja, data):  # pragma: no cover
        """Recursively traverses the Project Explorer tree collecting data.

        Args:
            node: Project Explorer tree node.
            sim_jinja (:obj:`dict`): Dict of stuff for jinja.
            data (:obj:`dict`): Dict of various data that we will need.
        """
        if node.item_typename == 'TI_MESH2D_PTR':
            sim_jinja['mesh_name'] = node.name
            data['mesh_uuid'] = node.uuid
        elif node.item_typename == 'TI_DATASET_PTR':
            dataset = self._query.item_with_uuid(node.uuid)
            if dataset.num_components == 1:  # 1 for scalar datasets (vectors have > 1)
                dataset_jinja = {
                    'name': dataset.name,
                    'min': fmt(np.amin(dataset.mins[:])),
                    'max': fmt(np.amax(dataset.maxs[:]))
                }
                data['datasets_jinja'].append(dataset_jinja)
        elif node.item_typename == 'TI_SOLUTION_FOLDER':
            for child_node in node.children:
                self._visit_tree_node(child_node, sim_jinja, data)
        elif node.item_typename == 'TI_COVER_PTR':
            if node.coverage_type == 'Boundary Conditions':
                sim_jinja['bc_coverage'] = {'name': node.name}
                data['bc_coverage_uuid'] = node.uuid
            elif node.coverage_type == 'Materials':
                sim_jinja['material_coverage'] = {'name': node.name}
            elif node.coverage_type == 'Monitor':
                sim_jinja['monitor_coverage'] = {'name': node.name}
                data['coverage'] = self._query.item_with_uuid(node.uuid)
                points = data['coverage'].get_points(FilterLocation.PT_LOC_DISJOINT)
                sim_jinja['monitor_coverage']['point_count'] = len(points)
                arcs = data['coverage'].arcs
                sim_jinja['monitor_coverage']['line_count'] = len(arcs)
                data['monitor_coverage_uuid'] = node.uuid
            elif node.coverage_type == 'Obstructions':
                sim_jinja['obstruction_coverage'] = {'name': node.name}

    def _sim_component_from_sim_node(self, sim_node):  # pragma: no cover
        """Returns the SimComponent given the simulation tree node.

        Args:
            sim_node (:obj:`TreeNode`): Simulation tree node.

        Returns:
            (:obj:`SimComponent`): The simulation component.
        """
        component = self._query.item_with_uuid(sim_node.uuid, model_name='SRH-2D', unique_name='Sim_Manager')
        from xms.srh.components.sim_component import SimComponent  # Must import here or circular dependencies
        sim_component = SimComponent(component.main_file)
        return sim_component

    def _get_cpu_time(self, sim_component, simulation_name):  # pragma: no cover
        """Returns the CPU time found in the OUT.dat file if it can be found.

        Args:
            sim_component (:obj:`SimComponent`): The SimComponent.
            simulation_name (:obj:`str`): Name of the simulation.

        Returns:
            (:obj:`str`): CPU time, or 'Not found' if we couldn't find it.
        """
        out_file = self._path_to_dat_file(sim_component, simulation_name, 'OUT')
        time = 'Not found'
        if not os.path.isfile(out_file):
            return time

        # Open the file, find the "CPU-TIME" string and read the time
        with open(out_file, 'r') as file:
            contents = file.read()
            pos = contents.find('CPU-TIME')
            if pos > -1:
                pos = contents.find('=', pos)
                if pos > -1:
                    pos2 = contents.find('\n', pos)
                    if pos2 > -1:
                        time_string = contents[pos + 1:pos2]
                        time = str(float(time_string.strip('')))
        return time

    def _get_srh_version(self, sim_component, simulation_name):  # pragma: no cover
        """Returns the SRH-2D version found in the OUT.dat file if it can be found.

        Args:
            sim_component (:obj:`SimComponent`): The SimComponent.
            simulation_name (:obj:`str`): Name of the simulation.

        Returns:
            (:obj:`str`): The SRH-2D version string.
        """
        out_file = self._path_to_dat_file(sim_component, simulation_name, 'OUT')
        version = 'Unknown'
        if not os.path.isfile(out_file):
            return version

        # Open the file, find the "SRH-2D Version" string and read it
        search_term = 'SRH-2D Version'
        with open(out_file, 'r') as file:
            contents = file.read()
            pos = contents.find(search_term)
            if pos > -1:
                pos2 = contents.find('*', pos)
                if pos2 > -1:
                    version = contents[pos + len(search_term):pos2].strip()
        return version

    def _path_to_dat_file(self, sim_component, simulation_name, suffix):  # pragma: no cover
        """Returns the path to the OUT.dat file (<CaseName>_OUT.dat) which may or may not exist.

        Args:
            sim_component (:obj:`SimComponent`): The SimComponent.
            simulation_name (:obj:`str`): Name of the simulation.
            suffix (:obj:`str`): e.g. "OUT", "INF". File name will be <case_name>_<suffix>.dat, e.g. "Steady_OUT.dat"

        Returns:
            (:obj:`str`): See description.
        """
        project_path = os.path.dirname(self._project_path)
        project_name = os.path.splitext(os.path.basename(self._project_path))[0]
        case_name = sim_component.data.hydro.case_name
        out_file = os.path.join(
            project_path, f'{project_name}_models', 'SRH-2D', simulation_name, f'{case_name}_{suffix}.dat'
        )
        return out_file

    @staticmethod
    def _fill_model_control_data(sim_component, sim_jinja):  # pragma: no cover
        """Fills in the model control data.

        Args:
            sim_component (:obj:`SimComponent`): The SimComponent.
            sim_jinja (:obj:`dict`): Dict of jinja data for the simulation.
        """
        model_control = sim_component.data
        sim_jinja['type'] = 'Transport' if model_control.enable_sediment else 'Flow'
        sim_jinja['start_time'] = str(model_control.hydro.start_time)
        sim_jinja['time_step'] = str(model_control.hydro.time_step)
        sim_jinja['end_time'] = str(model_control.hydro.end_time)
        sim_jinja['initial_condition'] = model_control.hydro.initial_condition
        sim_jinja['initial_value'] = 'NA'
        if model_control.hydro.initial_condition == 'Initial Water Surface Elevation':
            sim_jinja['initial_value'] = model_control.hydro.initial_water_surface_elevation
        elif model_control.hydro.initial_condition == 'Restart File':
            sim_jinja['initial_value'] = model_control.hydro.restart_file
        elif model_control.hydro.initial_condition == 'Water Surface Elevation Dataset':
            sim_jinja['initial_value'] = model_control.hydro.water_surface_elevation_dataset
        sim_jinja['turbulence_model'] = model_control.advanced.turbulence_model
        sim_jinja['turbulence_parameter'] = 'NA'
        if model_control.advanced.turbulence_model == 'Parabolic':
            sim_jinja['turbulence_parameter'] = model_control.advanced.parabolic_turbulence
        sim_jinja['unsteady_output'] = model_control.advanced.unsteady_output
        sim_jinja['pressure_dataset'] = 'NA'
        if model_control.advanced.specify_pressure_dataset:
            sim_jinja['pressure_dataset'] = model_control.advanced.pressure_dataset
        sim_jinja['output_method'] = model_control.output.output_method
        sim_jinja['output_frequency'] = 'NA'
        if model_control.output.output_method == 'Specified Frequency':
            output = model_control.output
            sim_jinja['output_frequency'] = f'{output.output_frequency} ({output.output_frequency_units})'

    def _find_srh_sims_node(self):  # pragma: no cover
        """Finds and returns the 'SRH-2D Simulations' tree node.

        Returns:
            See description.
        """
        sim_data_node = tree_util.child_from_name(self._query.project_tree, 'Simulation Data')
        srh_sims_node = None
        if sim_data_node:
            srh_sims_node = tree_util.child_from_name(sim_data_node, 'SRH-2D Simulations')
        return srh_sims_node

    def _fill_calibration_summary_results(self):  # pragma: no cover
        """Fills in data for the 'Calibration summary results' section."""
        # TODO: calibration summary results
        # self._logger.info(f'Getting calibration data.')
        pass

    def _make_report_directory(self):  # pragma: no cover
        """Creates the directory where the report files will be created."""
        self._report_dir = os.path.join(os.path.dirname(self._project_path), 'reports', 'srh-2d_summary_report')
        self._logger.info(f'Creating/clearing report directory "{self._report_dir}"')
        filesystem.make_or_clear_dir(self._report_dir)

    def _render(self):  # pragma: no cover
        """Renders the data as html using a template.

        Returns:
            (:obj:`str`): The report html as a string.
        """
        self._logger.info('Rendering the report.')
        template_dir = os.path.join(os.path.dirname(__file__), 'templates')
        env = Environment(loader=FileSystemLoader(template_dir), autoescape=select_autoescape(['html']))
        # Convert scatter jinja dict to html and store html as a string in jinja dict
        template = env.get_template('scatter_summary.html')
        self._report_jinja['scatter_summary'] = template.render(self._scatter_jinja)

        # Convert mesh jinja dict to html and store html as a string in jinja dict
        template = env.get_template('mesh_summary.html')
        self._report_jinja['mesh_summary'] = template.render(self._meshes_jinja)

        # Convert bc coverage jinja dict to html and store html as a string in jinja dict
        template = env.get_template('bc_coverage_summary.html')
        self._report_jinja['bc_coverage_summary'] = template.render(self._bcs_jinja)

        # Convert monitor coverage jinja dict to html and store html as a string in jinja dict
        template = env.get_template('monitor_coverage_summary.html')
        self._report_jinja['monitor_coverage_summary'] = template.render(self._monitor_coverages_jinja)

        # Convert obstruction coverage jinja dict to html and store html as a string in jinja dict
        template = env.get_template('obstructions_summary.html')
        self._report_jinja['obstruction_coverage_summary'] = template.render(self._obstructions_jinja)

        # Convert bridge jinja dict to html and store html as a string in jinja dict
        template = env.get_template('bridge_summary.html')
        self._report_jinja['bridge_summary'] = template.render(self._bridge_jinja)

        # Convert structure jinja dict to html and store html as a string in jinja dict
        template = env.get_template('structures_map.html')
        self._report_jinja['structures_map'] = template.render(self._structure_jinja)

        # Convert material jinja dict to html and store html as a string in jinja dict
        template = env.get_template('material_summary.html')
        self._report_jinja['material_summary'] = template.render(self._material_jinja)

        # Convert report to html
        template = env.get_template('project_summary_report.html')
        html = template.render(self._report_jinja)
        return html

    def _create_file(self, html):  # pragma: no cover
        """Creates the .html file and returns the filename.

        Args:
            html (:obj:`str`): The report html string.

        Returns:
            (:obj:`str`): The filepath of the file created.
        """
        report_file = os.path.join(self._report_dir, 'index.html')
        self._logger.info(f'Writing the file "{report_file}"')
        with open(report_file, 'w') as f:
            f.write(html)

        # Copy .css file
        template_dir = os.path.join(os.path.dirname(__file__), 'templates')
        src = os.path.join(template_dir, 'report_styles.css')
        dst = os.path.join(self._report_dir, 'report_styles.css')
        filesystem.copyfile(src, dst)

        return report_file


def fmt(number, prec=-1):  # pragma: no cover
    """Convenience function to make lines shorter.

    Args:
        number (:obj:`float`):  The number to format.
        prec (:obj:`int`): The number of digits after the decimal point.

    Returns:
        The number formatted as a string.
    """
    return NumberCorrector.format_double(number, prec)


def datums_from_query(query):  # pragma: no cover
    """Returns the horizontal and vertical datums.

    Returns:
        (:obj:`tuple`): tuple containing:

            horizontal datum (:obj:`str`):

            vertical datum (:obj:`str`):
    """
    # Get the horizontal datum
    spatial_ref = wkt_to_sr(query.display_projection.well_known_text)
    return spatial_ref.GetAttrValue('DATUM'), query.display_projection.vertical_datum


def distance_xy(point1, point2):
    """Returns the distance between two points.

    Args:
        point1: The first point.
        point2: The second point.

    Returns:
        See description.
    """
    return math.sqrt(((point2[0] - point1[0])**2) + ((point2[1] - point1[1])**2))


def mean_edge_length(poly):
    """Returns the average length of the edges of the polygon.

    Args:
        poly: Array of points

    Returns:
        See description.
    """
    point_count = len(poly)
    if point_count == 0:
        return 0.0

    total_length = 0.0
    for i in range(point_count - 1):
        total_length += distance_xy(poly[i], poly[i + 1])
    total_length += distance_xy(poly[point_count - 1], poly[0])
    return total_length / point_count


def get_smallest_and_largest_element_sizes(ugrid):
    """Returns the largest and smallest element sizes based on average cell edge lengths.

    Args:
        ugrid (:obj:`xms.grid.ugrid.ugrid.UGrid`): The UGrid.

    Returns:
        (:obj:`tuple`): tuple containing:

            smallest_element_size (:obj:`float`): The smallest element size.

            largest_element_size (:obj:`float`): The largest element size.
    """
    smallest_element_size = sys.float_info.max
    largest_element_size = -1
    for cell in range(ugrid.cell_count):
        success, poly = ugrid.get_cell_plan_view_polygon(cell)
        if success:
            length = mean_edge_length(poly)
            if length < smallest_element_size:
                smallest_element_size = length
            if length > largest_element_size:
                largest_element_size = length
    return smallest_element_size, largest_element_size


def get_average_edge_length(ugrid):
    """Computes and returns the average edge length for all cells in the UGrid.

    Args:
        ugrid (:obj:`xms.grid.ugrid.ugrid.UGrid`): The UGrid.

    Returns:
        (:obj:`float`) average edge length or -1.0 if no edges.
    """
    total_edge_lengths = 0.0
    edge_count = 0
    locations = ugrid.locations
    edges = set()
    for cell in range(ugrid.cell_count):
        for edge in ugrid.get_cell_edges(cell):
            sorted_edge = (min(edge), max(edge))
            if sorted_edge not in edges:
                total_edge_lengths += distance_xy(locations[edge[0]], locations[edge[1]])
                edge_count += 1
                edges.add(sorted_edge)

    if edge_count > 0:
        average_edge_length = total_edge_lengths / edge_count
    else:  # pragma: no cover
        average_edge_length = -1.0
    return average_edge_length
