"""Migrate old DMI material and sediment material coverage attributes."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.guipy.data.polygon_texture import PolygonTexture

# 4. Local modules

UNASSIGNED_DB_MAT_ID = -2
UNASSIGNED_COMP_MAT_ID = 0


class MigrateMaterialCoverage:
    """Convert old SRH-2D material coverage material attributes and polygon assignments to component-based coverage."""
    def __init__(self, proj_dir, widget_map, display_atts):
        """Construct the migrator.

        Args:
            proj_dir (:obj:`str`): Path to the saved project's directory.
            widget_map (:obj:`dict`): Dict of material widget values for a particular coverage keyed by material id
            display_atts (:obj:`dict`): Dict of material display attributes for a particular coverage keyed by
                material id

        """
        self._proj_dir = proj_dir
        self._widgets = widget_map
        self._display_atts = display_atts

        # Output variables
        self.mat_names = {}  # {mat_id: name}
        self.mat_display = {}  # {mat_id: 'r g b texture'}
        self.manningsn = {}  # Non-sediment material parameters
        self.sed_mat_data = []  # Sediment material parameters

    def _get_display_atts(self):
        """Extract material names, colors, and texture ids from the database."""
        for mat_id, display_atts in self._display_atts.items():
            if mat_id == UNASSIGNED_DB_MAT_ID:
                mat_id = UNASSIGNED_COMP_MAT_ID
            # display_atts = (name, r, g, b, alpha, texture)
            self.mat_names[mat_id] = display_atts[0]
            texture_id = display_atts[5]
            if texture_id == PolygonTexture.solid_pattern:  # Adjust texture id stored in database to proper enum
                texture_id = int(PolygonTexture.null_pattern)
            self.mat_display[mat_id] = f'{display_atts[1]} {display_atts[2]} {display_atts[3]} {texture_id}'

    def _migrate_materials(self):
        """Migrate data of an old Materials coverage."""
        for mat_id in self.mat_names:
            mat_widgets = self._widgets.get(mat_id, {})
            if not mat_widgets:  # Use defaults for this material.
                self.manningsn[mat_id] = 0.02
                continue

            # Determine if using a constant Manning's N or a depth curve
            mannings_method = 'Constant'
            if 'cbxManningMethod' in mat_widgets:
                mannings_method = mat_widgets['cbxManningMethod'][0][2]
            if mannings_method == 'Constant' and 'edtManningRough' in mat_widgets:
                self.manningsn[mat_id] = mat_widgets['edtManningRough'][0][2]
            elif 'crvDepthVaryManning' in mat_widgets:
                depth_x = mat_widgets['crvDepthVaryManning'][0][2][2]
                depth_y = mat_widgets['crvDepthVaryManning'][0][2][3]
                self.manningsn[mat_id] = [(x, y) for x, y in zip(depth_x, depth_y)]

    def _migrate_sediment_materials(self):
        """Migrate data of an old Sediment Materials coverage."""
        for mat_id in self.mat_names:
            mat_widgets = self._widgets.get(mat_id, {})
            if not mat_widgets:  # No layers defined for this material.
                continue
            num_layers = len(mat_widgets['tblColLayerThickness'])
            for row in range(num_layers):
                layer_id = row + 1  # Convert to 1-based layer id for GUI
                units = 'EN' if mat_widgets['tblColLayerUnits'][row][2] == 'English' else 'SI'
                self.sed_mat_data.append(
                    f"SUBSURFACETHICKNESS {mat_id} {layer_id} "
                    f"{mat_widgets['tblColLayerThickness'][row][2]} "
                    f"{units} "
                    f"{mat_widgets['tblColLayerClayDensity'][row][2]}"
                )
                gradation_curve = [
                    (x, y) for x, y in
                    zip(mat_widgets['tblColGradation'][row][2][2], mat_widgets['tblColGradation'][row][2][3])
                ]
                self.sed_mat_data.append((mat_id, layer_id, gradation_curve))

    def migrate(self, is_sediment):
        """Migrate the material parameters, display attributes, and polygon assignments.

        Args:
            is_sediment (:obj:`bool`): True if the coverage we are migrating is a Sediment Materials type.

        """
        self._get_display_atts()
        if is_sediment:
            self._migrate_sediment_materials()
        else:
            self._migrate_materials()
