"""Provide progress feedback when running an SRH-2D PEST model in SMS."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import json
import logging
import os
import shlex

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query

# 4. Local modules
from xms.srh.model.pest_progress_plots import PestPlotManager


class SrhPestTracker:
    """Class that tracks an SRH-2D PEST model running in SMS."""
    query = None
    logger = None
    echo_file = None
    echo_pos = 0
    params = None
    pest_iter = 0
    prog_increment = 0
    model_prog_increment = 0
    current_progress = 0
    plots_manager = None

    @staticmethod
    def progress_function():
        """Calculates the progress and sends it to SMS."""
        if not SrhPestTracker.echo_file:
            SrhPestTracker.echo_file = SrhPestTracker.plots_manager.looper.command_line_output_file
            SrhPestTracker.logger.info(f'echo_file: {SrhPestTracker.echo_file}')
            SrhPestTracker.logger.info(f'prog_increment: {SrhPestTracker.prog_increment}')

        update_prog = False
        try:
            with open(SrhPestTracker.echo_file, "r") as f:
                f.seek(SrhPestTracker.echo_pos)
                echo_line = f.readline()
                while echo_line:
                    # if (echo_line.endswith('\n') or echo_line.endswith('\r')) and \
                    #         echo_line.strip().startswith("Processed timestep: "):
                    if echo_line.endswith('\n') or echo_line.endswith('\r'):
                        SrhPestTracker.logger.info(f'echo_line: {echo_line}')
                        if 'OPTIMISATION ITERATION NO.        : 1' in echo_line:
                            # first model run complete
                            update_prog = True
                            SrhPestTracker.current_progress = SrhPestTracker.prog_increment * 0.5 * 100
                            SrhPestTracker.pest_iter = 1
                        elif 'OPTIMISATION ITERATION NO.        : ' in echo_line:
                            items = shlex.split(echo_line)
                            SrhPestTracker.pest_iter = int(items[4])
                            SrhPestTracker.logger.info(f'pest_iter: {SrhPestTracker.pest_iter}')
                        elif 'Calculating Jacobian matrix: running model ' in echo_line:
                            items = shlex.split(echo_line)
                            num_runs = int(items[5])
                            SrhPestTracker.model_prog_increment = SrhPestTracker.prog_increment / num_runs
                            SrhPestTracker.logger.info(f'num_runs: {num_runs}')
                            SrhPestTracker.logger.info(f'model_prog_increment: {SrhPestTracker.model_prog_increment}')
                        elif ' runs completed.' in echo_line:
                            items = shlex.split(echo_line)
                            run_iter = int(items[0])
                            SrhPestTracker.logger.info(f'run_iter: {run_iter}')
                            update_prog = True
                            SrhPestTracker.current_progress += SrhPestTracker.model_prog_increment * 100
                        elif 'Recording run statistics .....' in echo_line:
                            pass

                        SrhPestTracker.echo_pos = f.tell()
                    echo_line = f.readline()
        except Exception:
            pass  # File might not exist yet

        # Update Run Queue plots
        SrhPestTracker.plots_manager.update_plots()

        if update_prog:
            SrhPestTracker.logger.info(f'current_progress: {SrhPestTracker.current_progress}')
            SrhPestTracker.query.update_progress_percent(SrhPestTracker.current_progress)

    @staticmethod
    def start_tracking():
        """Entry point for the SRH-Post progress script."""
        SrhPestTracker.logger = logging.getLogger('xms.srh')
        log_file = 'srh_pest_progress.log'
        if os.path.isfile(log_file):
            os.remove(log_file)
        handler = logging.FileHandler(log_file)
        SrhPestTracker.logger.addHandler(handler)
        SrhPestTracker.logger.setLevel(logging.INFO)
        SrhPestTracker.logger.info('Begin logging')

        SrhPestTracker.query = Query(progress_script=True)
        sim_uuid = SrhPestTracker.query.current_item_uuid()
        sim_do_comp = SrhPestTracker.query.item_with_uuid(sim_uuid, model_name='SRH-2D', unique_name='Sim_Manager')
        sim_dir = os.path.dirname(sim_do_comp.main_file)

        # Read/write the params.json file using json instead of orjson because params.json could have lists
        # with more than one type (both floats and strings) and orjson does not seem to support this.
        params_file = os.path.join(sim_dir, 'params.json')
        with open(params_file, 'r') as file:
            SrhPestTracker.params = json.loads(file.read())
            SrhPestTracker.prog_increment = 1 / (SrhPestTracker.params['max_iterations'] + 1)
        SrhPestTracker.logger.info(f'Param data: {SrhPestTracker.params}')

        # do this last
        prog = SrhPestTracker.query.xms_agent.session.progress_loop
        SrhPestTracker.plots_manager = PestPlotManager(prog)
        prog.set_progress_function(SrhPestTracker.progress_function)
        prog.start_loop()
