"""AdvectiveCourantNumber Algorithm."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from logging import Logger

# 2. Third party modules
import numpy

# 3. Aquaveo modules
from xms.constraint.ugrid_activity import CellToPointActivityCalculator
from xms.datasets.dataset_reader import DatasetReader
from xms.datasets.dataset_writer import DatasetWriter
from xms.grid.ugrid import UGrid
from xms.mesher.meshing import mesh_utils

# 4. Local modules
from xms.tool.utilities.dataset_tool import set_builder_activity_flags


def advective_courant_number(dataset_name: str, dset_grid: UGrid, logger: Logger, timestep_seconds: float,
                             input_dset: DatasetReader) -> DatasetWriter:
    """Computes advective courant number for a vector dataset.

    Args:
        dataset_name: The name for the output dataset
        dset_grid: (xms.constraint.Grid): The dataset's constrained grid.
        logger: Logger for user output.
        timestep_seconds: The time step in seconds.
        input_dset: Dataset reader for the input dataset.

    Returns:
        The DatasetWriter for the output dataset
    """
    # Create and set up the DatasetWriter for the output dataset file
    builder = DatasetWriter()
    builder.name = dataset_name
    builder.geom_uuid = input_dset.geom_uuid
    builder.ref_time = input_dset.ref_time
    builder.time_units = input_dset.time_units
    builder.null_value = input_dset.null_value

    # Calculate the advective time step data
    if input_dset.activity and input_dset.values:
        if input_dset.values.shape != input_dset.activity.shape:
            # Nodal dataset values with cell activity
            input_dset.activity_calculator = CellToPointActivityCalculator(dset_grid)
            builder.activity_calculator = CellToPointActivityCalculator(dset_grid)

    # Calculate the size function and set the dataset values default
    logger.info('Calculating size function...')
    size_func = mesh_utils.size_function_from_edge_lengths(dset_grid)
    size_func = numpy.array(size_func)

    # Loop on the timesteps
    time_count = len(input_dset.times)
    for tsidx in range(time_count):
        logger.info(f'Processing time step {tsidx + 1} of {time_count}...')
        data, activity = input_dset.timestep_with_activity(tsidx)
        set_builder_activity_flags(activity, builder)

        # Calculate a value for each data value, after defaulting to the dataset null value
        magnitude = numpy.sqrt((data ** 2).sum(-1))
        new_data = numpy.full([len(data)], numpy.nan)
        new_data[size_func > 0.0] = magnitude * timestep_seconds / size_func

        # Append the values to the time step
        builder.append_timestep(input_dset.times[tsidx], new_data, activity)

    # Write output advective courant number dataset to XMDF file
    logger.info('Writing output advective courant number dataset to XMDF file...')
    builder.appending_finished()

    return builder
