"""PolygonsFromUGridBoundaryTool class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.tool_core import IoDirection, Tool

# 4. Local modules
from xms.tool.algorithms.coverage.lines_from_points import LinesFromPoints
from xms.tool.utilities.coverage_conversion import convert_lines_to_coverage


class ArcsFromUGridPointsTool(Tool):
    """Tool to convert ugrid points to arcs."""
    ARG_INPUT_UGRID = 0
    ARG_NUM_NEAREST_NEIGHBORS = 1
    ARG_CONNECT_ISOLATED_POINTS = 2
    ARG_ENFORCE_MAX_DISTANCE = 3
    ARG_MAX_DISTANCE = 4
    ARG_OUTPUT_COVERAGE = 5

    def __init__(self):
        """Initializes the class."""
        super().__init__(name='Arcs from UGrid Points')
        self.new_cov = None

    def initial_arguments(self):
        """Get initial arguments for tool.

        Must override.

        Returns:
            (list): A list of the initial tool arguments.
        """
        arguments = [
            self.grid_argument(name='input_grid', description='Input grid'),
            self.integer_argument(name='num_nearest_neighbors', description='Number of nearest neighbors',
                                  value=3, min_value=2),
            self.bool_argument(name='connect_isolated_points', description='Connect isolated points', value=False),
            self.bool_argument(name='enforce_max_distance', description='Enforce max distance', value=False),
            self.float_argument(name='max_distance', description='Max distance', min_value=0.001, value=1.0),
            self.coverage_argument(name='output_coverage', description='Output coverage name',
                                   io_direction=IoDirection.OUTPUT, optional=True)
        ]
        return arguments

    def run(self, arguments):
        """Override to run the tool.

        Args:
            arguments (list): The tool arguments.
        """
        ug = self.get_input_grid(arguments[self.ARG_INPUT_UGRID].value)
        if ug is not None:
            points = ug.ugrid.locations.tolist()
            cov_name = arguments[self.ARG_OUTPUT_COVERAGE].value
            if cov_name == '' or cov_name is None:  # default the coverage name to match the ugrid
                ug_path = arguments[self.ARG_INPUT_UGRID].text_value.split('/')
                if ug_path:
                    cov_name = ug_path[-1]
            num_nearest_neighbors = min(arguments[self.ARG_NUM_NEAREST_NEIGHBORS].value, ug.ugrid.point_count - 1)
            connect_isolated_pts = arguments[self.ARG_CONNECT_ISOLATED_POINTS].value
            enforce_max_dist = arguments[self.ARG_ENFORCE_MAX_DISTANCE].value
            max_dist = arguments[self.ARG_MAX_DISTANCE].value
            lfp = LinesFromPoints(points, num_nearest_neighbors, connect_isolated_pts, enforce_max_dist, max_dist)
            lines = lfp.lines_from_points()
            line_pts = []
            for line in lines:
                cur_line_pts = []
                for idx in line:
                    cur_line_pts.append(points[idx])
                line_pts.append(cur_line_pts)
            self.new_cov = convert_lines_to_coverage(line_pts, cov_name, self.default_wkt)
            arguments[self.ARG_OUTPUT_COVERAGE].value = cov_name
            self.set_output_coverage(self.new_cov, arguments[self.ARG_OUTPUT_COVERAGE])
