"""PolygonsFromIndexRasterTool tool class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules
from xms.gdal.rasters import RasterInput
from xms.gdal.rasters.raster_utils import convert_index_raster_data_to_polygons
from xms.gdal.vectors.vector_input import get_poly_features
from xms.tool_core import IoDirection, Tool

# 4. Local modules
from xms.tool.utilities.coverage_conversion import convert_polygons_to_coverage, get_polygons_from_list
from xms.tool.whitebox.whitebox_tool import get_argument_from_name

ARG_INPUT_INDEX_RASTER = 0
ARG_INDEX_POLYGON_COVERAGE = 1


class PolygonsFromIndexRasterTool(Tool):
    """PolygonsFromIndexRasterTool tool class."""

    def __init__(self):
        """Initializes the class."""
        super().__init__('Polygons from Index Raster')
        self._raster = None

    def validate_arguments(self, arguments):
        """Called to determine if arguments are valid.

        Args:
            arguments (list): The tool arguments.

        Returns:
            (dict): Dictionary of errors for arguments.
        """
        errors = {}
        raster_name = get_argument_from_name(arguments, 'input_index_raster').value
        raster = self.get_input_raster(raster_name)
        # Get the number of unique raster values.  If the number of values is less than 10% of the total number of
        # cells, I'll assume this is an index raster.
        raster_type = raster.data_type
        if raster_type in {RasterInput.GDT_Float32, RasterInput.GDT_Float64, RasterInput.GDT_CFloat32,
                           RasterInput.GDT_CFloat64}:
            num_cells = raster.resolution[0] * raster.resolution[1]
            if num_cells > 20000:
                threshold = int(num_cells * 0.05)
                values = raster.get_raster_values()
                num_values = len(np.unique(values))
                if num_values > threshold:
                    errors['input_index_raster'] = f'The input raster {raster_name} does not appear to be an index ' \
                                                   f'raster. This raster has {num_values} unique values. Please ' \
                                                   'select a raster with fewer unique values.'
        return errors

    def initial_arguments(self):
        """Get initial arguments for tool.

        Returns:
            (list): A list of the initial tool arguments.
        """
        arguments = [
            self.raster_argument(name='input_index_raster', description='Input index raster'),
            self.coverage_argument(name='index_polygon_coverage', description='Index polygon coverage',
                                   io_direction=IoDirection.OUTPUT),
        ]
        return arguments

    def run(self, arguments):
        """Runs the tool.

        Args:
            arguments (list): A list of the tool's arguments.
        """
        input_filename = arguments[ARG_INPUT_INDEX_RASTER].value
        self._raster = self.get_input_raster(input_filename)
        input_path_and_filename = self.get_input_raster_file(input_filename)

        # Convert the index raster to polygons
        self.logger.info('Converting index raster to polygons...')
        index_ri = RasterInput(input_path_and_filename)
        vec_ds = convert_index_raster_data_to_polygons(index_ri, None, True)
        index_ri = None

        # Get polygons
        index_hole_data, index_poly_data = get_polygons_from_list(get_poly_features(vec_ds))
        # Flush the buffer for the shapefile
        vec_ds = None

        # Generate index map coverage or shapefile
        index_coverage_filename = arguments[ARG_INDEX_POLYGON_COVERAGE].value
        polys = []
        for index_poly in index_poly_data:
            polys.append([index_poly['poly_pts']])
        new_cov = convert_polygons_to_coverage(polys, index_coverage_filename, self._raster.wkt)
        self.set_output_coverage(new_cov, arguments[ARG_INDEX_POLYGON_COVERAGE])
