"""PolygonsFromRasterNodataTool class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.tool_core import IoDirection, Tool

# 4. Local modules
from xms.tool.algorithms.coverage.polygons_from_raster_nodata import polygons_from_raster_nodata

ARG_INPUT_RASTER = 0
ARG_INPUT_CELL_THRESHOLD = 1
ARG_OUTPUT_COVERAGE = 2


class PolygonsFromRasterNodataTool(Tool):
    """Tool to make a coverage with polygons for the interior active regions of the input raster."""

    def __init__(self):
        """Initializes the class."""
        super().__init__(name='Polygons from Raster Nodata')

    def initial_arguments(self):
        """Get initial arguments for tool.

        Must override.

        Returns:
            (list): A list of the initial tool arguments.
        """
        arguments = [
            self.raster_argument(name='input_raster', description='Input raster'),
            self.integer_argument(name='cell_tolerance', description='Number of cells required to make a polygon',
                                  min_value=1, value=5),
            self.coverage_argument(name='output_coverage', description='Output coverage',
                                   io_direction=IoDirection.OUTPUT)
        ]
        return arguments

    def run(self, arguments):
        """Override to run the tool.

        Args:
            arguments (list): The tool arguments.
        """
        input_raster = self.get_input_raster(arguments[ARG_INPUT_RASTER].value)
        cell_threshold = arguments[ARG_INPUT_CELL_THRESHOLD].value
        out_cov = arguments[ARG_OUTPUT_COVERAGE].value

        polygon_cov = polygons_from_raster_nodata(input_raster, cell_threshold, out_cov, self.default_wkt,
                                                  self.vertical_datum, self.vertical_units)

        if not polygon_cov.empty:
            self.set_output_coverage(polygon_cov, arguments[ARG_OUTPUT_COVERAGE])
