"""VoronoiUGridFromUGridTool class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.constraint import UnconstrainedGrid
from xms.constraint.ugrid_activity import CellToPointActivityCalculator
from xms.tool_core import IoDirection, Tool

# 4. Local modules
from xms.tool.utilities.dataset_tool import copy_datasets_to_new_geometry


class UGridFromSurfaceTool(Tool):
    """Tool to convert UGrids to Voronoi grids."""
    ARG_INPUT_GRID = 0
    ARG_OUTPUT_GRID = 1

    def __init__(self):
        """Initializes the class."""
        super().__init__(name='UGrid from Surface')
        self._args = None
        self._available_datasets = None
        self._datasets = []
        self._input_cogrid = None
        self._out_grid = None
        self._ug_name = ''

    def initial_arguments(self):
        """Get initial arguments for tool.

        Must override.

        Returns:
            (list): A list of the initial tool arguments.
        """
        arguments = [
            self.grid_argument(name='input_grid', description='Input grid'),
            self.grid_argument(name='output_grid', description='Output grid name', io_direction=IoDirection.OUTPUT),
        ]
        return arguments

    def validate_arguments(self, arguments):
        """Called to determine if arguments are valid.

        Args:
            arguments (list): The tool arguments.

        Returns:
            (dict): Dictionary of errors for arguments.
        """
        errors = {}

        # Make sure output name specified
        self._validate_input_grid(errors, arguments[self.ARG_INPUT_GRID])

        return errors

    def _validate_input_grid(self, errors, argument):
        """Validate grid is specified and 2D.

        Args:
            errors (dict): Dictionary of errors keyed by argument name.
            argument (GridArgument): The grid argument.
        """
        self._ug_name = argument.text_value
        self._input_cogrid = self.get_input_grid(self._ug_name)

        if not self._input_cogrid:
            errors[argument.name] = 'Could not read grid.'

    def run(self, arguments):
        """Override to run the tool.

        Args:
            arguments (list): The tool arguments.
        """
        self._args = arguments
        self._get_inputs()
        self._create_output_grid()
        self._create_datasets()
        self._set_outputs()

    def _set_outputs(self):
        """Set the outputs from the tool."""
        self.set_output_grid(self._out_grid, self._args[self.ARG_OUTPUT_GRID])
        if self._datasets:
            for ds in self._datasets:
                self.set_output_dataset(ds)

    def _get_inputs(self):
        """Get the inputs from the arguments."""
        user_input = self._args[self.ARG_OUTPUT_GRID].text_value
        self._args[self.ARG_OUTPUT_GRID].value = user_input if user_input else os.path.basename(self._ug_name)
        self._available_datasets = self.get_grid_datasets(self._args[self.ARG_INPUT_GRID].value)

    def _create_output_grid(self):
        """Creates the output grid."""
        self._out_grid = UnconstrainedGrid(ugrid=self._input_cogrid.ugrid)
        self._out_grid.uuid = str(uuid.uuid4())

    def _create_datasets(self):
        """Copies compatible datasets to the output mesh."""
        ug = self._out_grid.ugrid
        calculator = CellToPointActivityCalculator(ug)
        data = {'tool': self, 'old_geom_uuid': self._input_cogrid.uuid, 'new_geom_uuid': self._out_grid.uuid,
                'old_geom_name': self._ug_name, 'ug_points': ug.locations, 'calculator': calculator,
                'available_datasets': self._available_datasets, 'datasets': self._datasets}

        copy_datasets_to_new_geometry(data)
