"""Testing utility functions."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


def _get_base_test_path():
    """Returns the full path to the 'tests' directory.

    Returns:
        (str): See description.
    """
    file_dir = os.path.dirname(os.path.realpath(__file__))
    files_path = os.path.join(file_dir, '..', '..', '..', 'tests')
    return os.path.abspath(files_path)


def get_test_files_path():
    """Returns the full path to the 'tests/files' directory.

    Returns:
        (str): See description.
    """
    return os.path.join(_get_base_test_path(), 'files')


def current_test_folder(cur_folder=None):
    """Returns or sets the current test folder.

    Args:
        cur_folder (str): If this is passed, this function sets the current testing folder and returns the new folder.

    Returns:
        (str): The current testing folder.
    """
    if cur_folder is not None:
        current_test_folder.cur_folder = cur_folder
    if not hasattr(current_test_folder, 'cur_folder'):
        current_test_folder.cur_folder = ''
    return current_test_folder.cur_folder


def get_vector_filename(vec_name):
    """Returns the full path and filename for a vector shapefile."""
    xms_temp_folder = os.environ.get('XMS_PYTHON_APP_TEMP_DIRECTORY', 'unknown')
    basename = vec_name
    vector_file = os.path.join(xms_temp_folder, f'{basename}.shp')
    if not os.path.isfile(vector_file):
        # For testing only, on linux we only work with shapefiles in the "coverages" directory.
        dir_name = os.path.join(get_test_files_path(), current_test_folder(), 'coverages')
        if os.path.isdir(dir_name):  # pragma: no cover - reenable with test_watershed_from_raster_tool.py
            vector_file = os.path.join(dir_name, f'{basename}.shp')
    return vector_file


def get_csv_filename(name):
    """Returns the full path and filename for a CSV file."""
    xms_temp_folder = os.environ.get('XMS_PYTHON_APP_TEMP_DIRECTORY', 'unknown')
    basename = name
    csv_file = os.path.join(xms_temp_folder, f'{basename}.csv')
    if not os.path.isfile(csv_file):
        # For testing only, the CSV files get saved to the test's "rasters" directory.
        dir_name = os.path.join(get_test_files_path(), current_test_folder(), 'rasters')
        if os.path.isdir(dir_name):  # pragma: no cover - reenable with test_watershed_from_raster_tool.py
            csv_file = os.path.join(dir_name, f'{basename}.csv')
    return csv_file


def get_raster_filename(name: str, raster_extension: str = '.tif'):
    """Returns the full path and filename for a raster file."""
    xms_temp_folder = os.environ.get('XMS_PYTHON_APP_TEMP_DIRECTORY', 'unknown')
    basename = name
    raster_file = os.path.join(xms_temp_folder, f'{basename}{raster_extension}')
    if not os.path.isfile(raster_file):
        # For testing only, the temp files get saved to the test's "rasters" directory.
        dir_name = os.path.join(get_test_files_path(), current_test_folder(), 'rasters')
        if os.path.isdir(dir_name):
            raster_file = os.path.join(dir_name, f'{basename}{raster_extension}')
    return raster_file
