"""BcComponentDisplay class."""
# 1. Standard python modules
import os
import shutil

# 2. Third party modules
import orjson
from PySide2.QtGui import QIcon

# 3. Aquaveo modules
from xms.components.display.display_options_io import (read_display_option_ids, read_display_options_from_json,
                                                       write_display_option_ids, write_display_options_to_json)
from xms.components.display.xms_display_message import DrawType, XmsDisplayMessage
from xms.core.filesystem import filesystem as io_util
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.data.target_type import TargetType
from xms.guipy.dialogs.category_display_options_list import CategoryDisplayOptionsDialog
from xms.guipy.dialogs.xms_parent_dlg import get_xms_icon
from xms.guipy.settings import SettingsManager
from xms.guipy.time_format import XmsTimeFormatter

# 4. Local modules
from xms.tuflowfv.components.tuflowfv_component import change_display_options, UNINITIALIZED_COMP_ID
from xms.tuflowfv.data import bc_data as bcd
from xms.tuflowfv.gui import gui_util
from xms.tuflowfv.gui.assign_bc_dialog import AssignBcDialog


__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

BC_ARC_JSON = 'bc_display_options.json'
BC_POINT_JSON = 'bc_point_display_options.json'
BC_POLYGON_JSON = 'bc_polygon_display_options.json'

REG_KEY_BC_ARC = 'bc_coverage_arc_default_options'
REG_KEY_BC_POINT = 'bc_coverage_point_default_options'
REG_KEY_BC_POLY = 'bc_coverage_polygon_default_options'

BC_INITIAL_POINT_ATT_ID_FILE = 'initial_bc_point.attids'
BC_INITIAL_POINT_COMP_ID_FILE = 'initial_bc_point.compids'
BC_INITIAL_ARC_ATT_ID_FILE = 'initial_bc_arc.attids'
BC_INITIAL_ARC_COMP_ID_FILE = 'initial_bc_arc.compids'
BC_INITIAL_POLY_ATT_ID_FILE = 'initial_bc_poly.attids'
BC_INITIAL_POLY_COMP_ID_FILE = 'initial_bc_poly.compids'


class BcComponentDisplay:
    """Helper class for BC component display options."""

    def __init__(self, bc_comp, query=None):
        """Create a helper class.

        Args:
            bc_comp (BcComponent): The component this helper should help
            query (Query): The XMS inter-process communicator
        """
        self.comp = bc_comp
        self.selected_att_ids = []
        self.selected_comp_ids = []
        self.dlg_message = ''
        self._query = query

    def _unpack_xms_data(self, param_map, bc_location):
        """Unpack the selection info and component id maps sent by XMS.

        Args:
            param_map (dict): The ActionRequest parameter map
            bc_location (int): BC_LOCATION_* enum

        Returns:
            (int): component id of atts to display
        """
        # Get the XMS attribute ids of the selected features (if any)
        self.selected_att_ids = param_map.get('selection', [])
        if not self.selected_att_ids:
            return UNINITIALIZED_COMP_ID

        # Get the component id map of the selected features (if any).
        comp_id = UNINITIALIZED_COMP_ID
        id_files = param_map.get('id_files', [])
        if id_files and id_files[0]:
            if bc_location == bcd.BC_LOCATION_ARC:
                files_dict = {'ARC': (id_files[0], id_files[1])}
            elif bc_location == bcd.BC_LOCATION_POINT:
                files_dict = {'POINT': (id_files[0], id_files[1])}
            else:  # bc_location == bcd.BC_LOCATION_POLY:
                files_dict = {'POLYGON': (id_files[0], id_files[1])}
            self.comp.load_coverage_component_id_map(files_dict)
            comp_id = self._check_selected_types(bc_location=bc_location)
        # Clean up temp files dumped by SMS.
        shutil.rmtree(os.path.join(os.path.dirname(self.comp.main_file), 'temp'), ignore_errors=True)
        return comp_id

    def _check_selected_types(self, bc_location):
        """Determine which attributes to display in Assign BC dialog and any warning message that should be added.

        Args:
            bc_location (int): The BC_LOCATION_* enum

        Returns:
            (int): component id of attributes to display
        """
        num_features = len(self.selected_att_ids)
        if bc_location == bcd.BC_LOCATION_ARC:
            target_type = TargetType.arc
        elif bc_location == bcd.BC_LOCATION_POINT:
            target_type = TargetType.point
        else:  # bc_location == bcd.BC_LOCATION_POLY:
            target_type = TargetType.polygon
        if num_features == 1:  # 1 arc selected, use those atts
            comp_id = self.comp.get_comp_id(target_type, self.selected_att_ids[0])
            self.selected_comp_ids.append(comp_id)
            return comp_id
        else:  # More than one arc selected, check types
            try:  # Get the component ids of all the selected arcs
                self.selected_comp_ids = list(self.comp.comp_to_xms[self.comp.cov_uuid][target_type].keys())
            except KeyError:
                return UNINITIALIZED_COMP_ID  # No component ids assigned for any of the selected arcs

            if target_type == TargetType.arc:  # Add a warning message to the dialog if multiple features are selected
                self.dlg_message = 'Multiple arcs selected. Changes will apply to all selected arcs.'
            elif target_type == TargetType.point:
                self.dlg_message = 'Multiple points selected. Changes will apply to all selected points.'
            else:  # target_type == TargetType.polygon:
                self.dlg_message = 'Multiple polygons selected. Changes will apply to all selected polygons.'
            # If there are multiple entities selected with the same component id, display those attributes. Otherwise
            # display an empty, default dialog.
            return UNINITIALIZED_COMP_ID if len(self.selected_comp_ids) != 1 else self.selected_comp_ids[0]

    def _update_coverage_registry_defaults(self, reg_key):
        """Save the default display options to the registry.

        Args:
            reg_key (str): The name of the key in the registry
        """
        settings = SettingsManager()
        # Read the default display options for mapped BC components.
        options = settings.get_setting('xmstuflowfv', reg_key)
        if not options:
            return  # No defaults set in registry
        json_dict = orjson.loads(options)
        json_dict['is_ids'] = 1  # Change from draw at locations to draw at ids.
        # Copy from mapped component registry key to BC coverage registry key.
        json_text = orjson.dumps(json_dict)
        settings.save_setting('xmstuflowfv', reg_key, json_text)

    def _update_component_ids_from_files(self, target_type):
        """Read att and comp ids from files and update in XMS.

        Called in get_initial_display_options() after a model native import.

        Args:
            target_type (TargetType): Coverage entity type to update
        """
        comp_folder = os.path.dirname(self.comp.main_file)
        if target_type == TargetType.point:
            initial_att_file = os.path.join(comp_folder, BC_INITIAL_POINT_ATT_ID_FILE)
            initial_comp_file = os.path.join(comp_folder, BC_INITIAL_POINT_COMP_ID_FILE)
        elif target_type == TargetType.arc:
            initial_att_file = os.path.join(comp_folder, BC_INITIAL_ARC_ATT_ID_FILE)
            initial_comp_file = os.path.join(comp_folder, BC_INITIAL_ARC_COMP_ID_FILE)
        else:  # target_type == TargetType.polygon
            initial_att_file = os.path.join(comp_folder, BC_INITIAL_POLY_ATT_ID_FILE)
            initial_comp_file = os.path.join(comp_folder, BC_INITIAL_POLY_COMP_ID_FILE)

        att_ids = read_display_option_ids(initial_att_file)
        comp_ids = read_display_option_ids(initial_comp_file)
        io_util.removefile(initial_att_file)
        io_util.removefile(initial_comp_file)
        for att_id, comp_id in zip(att_ids, comp_ids):
            self.comp.update_component_id(target_type, att_id, comp_id)

    def update_display_options_file(self, new_main_file, new_path):
        """Generate new UUIDs for the component and display lists.

        Will commit data file in this method.

        Args:
            new_main_file (str): Path to the new component's main file
            new_path (str): The new component's directory.
        """
        new_data = bcd.BcData(new_main_file)
        new_data.info.attrs['cov_uuid'] = ''
        new_comp_uuid = os.path.basename(new_path)
        fname = os.path.join(new_path, os.path.basename(self.comp.disp_opts_files[0]))
        new_data.info.attrs['display_uuid'] = change_display_options(new_comp_uuid, fname)
        fname = os.path.join(new_path, os.path.basename(self.comp.disp_opts_files[1]))
        new_data.info.attrs['point_display_uuid'] = change_display_options(new_comp_uuid, fname)
        fname = os.path.join(new_path, os.path.basename(self.comp.disp_opts_files[1]))
        new_data.info.attrs['poly_display_uuid'] = change_display_options(new_comp_uuid, fname)
        new_data.commit()

    def refresh_component_ids(self, new_comp_id, bc_location):
        """Load all the currently used component ids and clean out the unused ones.

        Called on OK of the assign BC feature dialogs. Keeps the files fresh and the data has been small enough so far.

        Args:
            new_comp_id (int): Component id of the feature that was just assigned
            bc_location (int): The BC_LOCATION_* enum
        """
        # Load all currently used component ids
        self._query.load_component_ids(self.comp, points=True, arcs=True, polygons=True)
        # Drop unused component ids from the xarray datasets
        # BC arcs
        arc_comp_ids = list(self.comp.comp_to_xms.get(self.comp.cov_uuid, {}).get(TargetType.arc, {}).keys())
        if bc_location == bcd.BC_LOCATION_ARC and new_comp_id > UNINITIALIZED_COMP_ID:
            arc_comp_ids.append(new_comp_id)
        mask = self.comp.data.bcs.comp_id.isin(arc_comp_ids)
        self.comp.data.bcs = self.comp.data.bcs.where(mask, drop=True)
        # BC points
        point_comp_ids = list(self.comp.comp_to_xms.get(self.comp.cov_uuid, {}).get(TargetType.point, {}).keys())
        if bc_location == bcd.BC_LOCATION_POINT and new_comp_id > UNINITIALIZED_COMP_ID:
            point_comp_ids.append(new_comp_id)
        mask = self.comp.data.points.comp_id.isin(point_comp_ids)
        self.comp.data.points = self.comp.data.points.where(mask, drop=True)
        # BC polygons
        poly_comp_ids = list(self.comp.comp_to_xms.get(self.comp.cov_uuid, {}).get(TargetType.polygon, {}).keys())
        if bc_location == bcd.BC_LOCATION_POLY and new_comp_id > UNINITIALIZED_COMP_ID:
            poly_comp_ids.append(new_comp_id)
        mask = self.comp.data.polygons.comp_id.isin(poly_comp_ids)
        self.comp.data.polygons = self.comp.data.polygons.where(mask, drop=True)

        # Load all used BC curves into memory
        arc_comp_ids.extend(point_comp_ids)
        arc_comp_ids.extend(poly_comp_ids)
        self.comp.data.clear_and_load_curves(arc_comp_ids)
        # Rewrite the display option id files now to clear out non-existent component ids.
        self.update_id_files()
        self.comp.data.vacuum()
        # Send back updated display lists to XMS after ActionRequest
        self.comp.display_option_list = [
            XmsDisplayMessage(file=self.comp.disp_opts_files[0], edit_uuid=self.comp.cov_uuid),
            XmsDisplayMessage(file=self.comp.disp_opts_files[1], edit_uuid=self.comp.cov_uuid),
            XmsDisplayMessage(file=self.comp.disp_opts_files[2], edit_uuid=self.comp.cov_uuid)
        ]

    def update_id_files(self):
        """Writes the display id files."""
        comp_path = os.path.dirname(self.comp.main_file)
        # Update arc BC id files
        arc_types = list(bcd.ARC_BC_TYPES.keys())
        arc_types.append('Monitor')
        for arc_bc_type in arc_types:
            id_file = self.get_display_id_file(arc_bc_type, comp_path)
            vals = self.comp.data.bcs.where(self.comp.data.bcs.type == arc_bc_type, drop=True)
            if len(vals.comp_id) > 0:
                write_display_option_ids(id_file, vals.comp_id.data.astype(int))
            else:  # If we don't have any of this type, just delete the id file.
                io_util.removefile(id_file)
        # Update point BC id files
        point_types = list(bcd.POINT_BC_TYPES.keys())
        for point_bc_type in point_types:
            id_file = self.get_display_id_file(point_bc_type, comp_path)
            vals = self.comp.data.points.where(self.comp.data.points.type == point_bc_type, drop=True)
            if len(vals.comp_id) > 0:
                write_display_option_ids(id_file, vals.comp_id.data.astype(int))
            else:  # If we don't have any of this type, just delete the id file.
                io_util.removefile(id_file)
        # Update polygon BC id files
        polygon_types = list(bcd.POLYGON_BC_TYPES.keys())
        polygon_types.append('QC_POLY')
        for polygon_bc_type in polygon_types:
            id_file = self.get_display_id_file(f'{polygon_bc_type}', comp_path)
            vals = self.comp.data.polygons.where(self.comp.data.polygons.type == polygon_bc_type, drop=True)
            if len(vals.comp_id) > 0:
                write_display_option_ids(id_file, vals.comp_id.data.astype(int))
            else:  # If we don't have any of this type, just delete the id file.
                io_util.removefile(id_file)

    def assign_bc(self, params, parent, bc_location):
        """Display the Assign BC dialog and persist data if accepted.

        Args:
            params (dict): The ActionRequest parameter map
            parent (QWidget): The parent window
            bc_location (int): BC_LOCATION_* enum

        Returns:
            tuple(list, list):
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the message
                    level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        old_comp_id = self._unpack_xms_data(params[0], bc_location=bc_location)
        if not self.selected_att_ids:
            if bc_location == bcd.BC_LOCATION_POINT:
                feature = 'points'
            elif bc_location == bcd.BC_LOCATION_ARC:
                feature = 'arcs'
            else:  # bc_location == bcd.BC_LOCATION_POLY:
                feature = 'polygons'
            return [('INFO', f'No {feature} selected. Select one or more {feature} to assign boundary conditions.')], []

        increment = True
        comp_id = old_comp_id

        if comp_id == UNINITIALIZED_COMP_ID:  # Create some default atts if this is the initial assignment.
            comp_id = self.comp.data.add_bc_atts(bc_location=bc_location)
            increment = False
        if bc_location == bcd.BC_LOCATION_POINT:
            dset = self.comp.data.points
        elif bc_location == bcd.BC_LOCATION_ARC:
            dset = self.comp.data.bcs
        else:  # bc_location == bcd.BC_LOCATION_POLY:
            dset = self.comp.data.polygons
        bc_data = dset.where(dset.comp_id == comp_id, drop=True)
        if increment:
            # Increment the component id anytime dialog triggered (won't commit unless user accepts). We don't want to
            # just update the existing attributes because they may have been assigned with a multi-select.
            comp_id = self.comp.data.add_bc_atts(bc_location=bc_location, dset=bc_data)

        time_formatter = XmsTimeFormatter(self._query.global_time_settings)
        dlg = AssignBcDialog(parent=parent, bc_data=bc_data, all_data=self.comp.data, msg=self.dlg_message,
                             bc_location=bc_location, comp_id=comp_id,
                             time_formats=(time_formatter.abs_specifier, time_formatter.qt_abs_specifier))
        if dlg.exec():
            # Update the attribute datasets
            self.comp.data.update_bc(comp_id=comp_id, new_atts=dlg.bc_data, bc_location=bc_location)
            # Associate all selected arcs with the new component id.
            if bc_location == bcd.BC_LOCATION_POINT:
                target_type = TargetType.point
            elif bc_location == bcd.BC_LOCATION_ARC:
                target_type = TargetType.arc
            else:  # bc_location == bcd.BC_LOCATION_POLY:
                target_type = TargetType.polygon
            for feature_id in self.selected_att_ids:
                self.comp.update_component_id(target_type, feature_id, comp_id)
            self.refresh_component_ids(comp_id, bc_location=bc_location)
        return [], []

    def display_options(self, parent, cov_uuid):
        """Shows the display options dialog for a display list.

        Args:
            parent (QWidget): The parent window
            cov_uuid (str): UUID of the coverage geometry associated with the display list, if any.

        Returns:
            tuple(list, list):
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the message
                    level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        # Get the arc/line display options.
        arc_categories = CategoryDisplayOptionList()
        json_dict = read_display_options_from_json(self.comp.disp_opts_files[0])
        arc_categories.from_dict(json_dict)
        categories_list = [arc_categories]

        # Get the point display options.
        point_categories = CategoryDisplayOptionList()
        json_dict = read_display_options_from_json(self.comp.disp_opts_files[1])
        point_categories.from_dict(json_dict)
        categories_list.append(point_categories)

        # Get the polygon display options.
        polygon_categories = CategoryDisplayOptionList()
        json_dict = read_display_options_from_json(self.comp.disp_opts_files[2])
        polygon_categories.from_dict(json_dict)
        categories_list.append(polygon_categories)

        wkt = ''
        if 'wkt' in arc_categories.projection:
            wkt = arc_categories.projection['wkt']

        reg_keys = [REG_KEY_BC_ARC, REG_KEY_BC_POINT, REG_KEY_BC_POLY]
        dlg = CategoryDisplayOptionsDialog(category_lists=categories_list, parent=parent, package_name='xmstuflowfv',
                                           registry_keys=reg_keys)

        dlg.setWindowIcon(QIcon(get_xms_icon()))
        dlg.setModal(True)
        if dlg.exec() and cov_uuid:
            # write display options JSON files
            category_lists = dlg.get_category_lists()
            for idx, category_list in enumerate(category_lists):
                category_list.projection['wkt'] = wkt
                write_display_options_to_json(self.comp.disp_opts_files[idx], category_list)
                self.comp.display_option_list.append(
                    XmsDisplayMessage(file=self.comp.disp_opts_files[idx], edit_uuid=cov_uuid,
                                      draw_type=DrawType.draw_at_ids)
                )
            # Save the default display options to the registry.
            self._update_coverage_registry_defaults(REG_KEY_BC_POINT)
            self._update_coverage_registry_defaults(REG_KEY_BC_ARC)
            self._update_coverage_registry_defaults(REG_KEY_BC_POLY)
        return [], []

    def ensure_display_option_files_exist(self):
        """Copies default BC arc and point display option JSON files to the component directory if needed.

        Will create new random UUIDs for the display lists. Should only be called by the unmapped BC coverage on
        creation.
        """
        if not os.path.exists(self.comp.disp_opts_files[0]):  # First is arcs, second is points, third is poygons
            categories = gui_util.read_default_display_options_from_registry(REG_KEY_BC_ARC)
            if not categories:
                # Read the default arc display options, and save ourselves a copy with a randomized UUID.
                categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
                default_file = os.path.join(os.path.dirname(os.path.dirname(__file__)), 'gui', 'resources',
                                            'default_data', BC_ARC_JSON)
                json_dict = read_display_options_from_json(default_file)
                categories.from_dict(json_dict)
            categories.comp_uuid = self.comp.uuid
            write_display_options_to_json(self.comp.disp_opts_files[0], categories)
            # Save our display list UUID to the main file
            self.comp.data.info.attrs['display_uuid'] = categories.uuid

            categories = gui_util.read_default_display_options_from_registry(REG_KEY_BC_POINT)
            if not categories:
                # Read the default point display options, and save ourselves a copy with a randomized UUID.
                categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
                default_file = os.path.join(os.path.dirname(os.path.dirname(__file__)), 'gui', 'resources',
                                            'default_data', BC_POINT_JSON)
                json_dict = read_display_options_from_json(default_file)
                categories.from_dict(json_dict)
            categories.comp_uuid = self.comp.uuid
            write_display_options_to_json(self.comp.disp_opts_files[1], categories)
            # Save our display list UUID to the main file
            self.comp.data.info.attrs['point_display_uuid'] = categories.uuid

            categories = gui_util.read_default_display_options_from_registry(REG_KEY_BC_POLY)
            if not categories:
                # Read the default point display options, and save ourselves a copy with a randomized UUID.
                categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
                default_file = os.path.join(os.path.dirname(os.path.dirname(__file__)), 'gui', 'resources',
                                            'default_data', BC_POLYGON_JSON)
                json_dict = read_display_options_from_json(default_file)
                categories.from_dict(json_dict)
            categories.comp_uuid = self.comp.uuid
            write_display_options_to_json(self.comp.disp_opts_files[2], categories)
            # Save our display list UUID to the main file
            self.comp.data.info.attrs['poly_display_uuid'] = categories.uuid

            self.comp.data.commit()

    def get_initial_display_options(self, query, params):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query (Query): Object for communicating with XMS
            params (Optional[dict]): Generic map of parameters, unused

        Returns:
            Empty message and ActionRequest lists
        """
        # Query XMS for parent coverage's UUID if we don't know it yet.
        if not self.comp.cov_uuid:
            self.comp.cov_uuid = query.parent_item_uuid()
            self.comp.data.info.attrs['cov_uuid'] = self.comp.cov_uuid
            self.comp.data.commit()
        if not self.comp.cov_uuid:
            return [('ERROR', 'Could not get the TUFLOWFV BC coverage UUID.')], []

        initial_att_file = os.path.join(os.path.dirname(self.comp.main_file), BC_INITIAL_ARC_ATT_ID_FILE)
        if os.path.isfile(initial_att_file):  # Came from a model native read, initialize the component ids.
            self._update_component_ids_from_files(TargetType.point)
            self._update_component_ids_from_files(TargetType.arc)
            self._update_component_ids_from_files(TargetType.polygon)
        # Send the display options json files to XMS.
        self.comp.display_option_list = [
            XmsDisplayMessage(file=self.comp.disp_opts_files[0], edit_uuid=self.comp.cov_uuid),  # arcs
            XmsDisplayMessage(file=self.comp.disp_opts_files[1], edit_uuid=self.comp.cov_uuid),  # points
            XmsDisplayMessage(file=self.comp.disp_opts_files[2], edit_uuid=self.comp.cov_uuid),  # polygons
        ]

    @staticmethod
    def get_display_id_file(bc_type, path):
        """Get the display option id file for a BC arc/point type.

        Args:
            bc_type (str): BC type string
            path (str): Path to the location of the component id files

        Returns:
            (str): See description
        """
        bc_type = bc_type.lower()
        if bc_type == 'unassigned':  # Get the right filename for point unassigned category
            bc_type = 'qc'
        if bc_type == 'monitor':  # Get the right filename for arc unassigned category
            bc_type = 'arc_unassigned'
        if bc_type == 'qc_poly':  # Get the right filename for polygon unassigned category
            bc_type = 'qc_poly'
        filename = f'{bc_type}.display_ids'
        return os.path.join(path, filename)
