"""Avoid circular dependencies."""
# 1. Standard python modules
import datetime

# 2. Third party modules
import numpy as np
from PySide2.QtCore import QDateTime

# 3. Aquaveo modules
from xms.guipy.time_format import ISO_DATETIME_FORMAT, qdatetime_to_datetime

# 4. Local modules


def check_for_object_strings_dumb(dset, variables):
    """Need this stupid check because xarray.where() switches the dtype of empty string variables to object.

    object dtype fails when serializing to NetCDF.

    Args:
        dset (xr.Dataset): The Dataset to check for bad string variables
        variables (Iterable): The names of the string variables that are potentially bad
    """
    for variable in variables:
        if variable in dset and dset[variable].dtype == object:
            dset[variable] = dset[variable].astype(np.unicode_)


def fix_datetime_format(dt_str):
    """Ensure string datetimes are in YYYY-MM-DD HH:MM:SS format.

    Args:
        dt_str (str): The datetime string representation in either locale or ISO_DATETIME_FORMAT format

    Returns:
        str: The datetime in ISO_DATETIME_FORMAT format
    """
    try:  # Try to parse using current locale
        qreftime = QDateTime.fromString(dt_str)
        reftime = qdatetime_to_datetime(qreftime)
    except Exception:
        try:  # Try to parse using ISO format
            reftime = datetime.datetime.strptime(dt_str, ISO_DATETIME_FORMAT)
        except Exception:  # Set default to 01/01/1990 epoch
            reftime = datetime.datetime(1990, 1, 1)
    return reftime.strftime(ISO_DATETIME_FORMAT)  # Reformat to ISO datetime representation
