"""TUFLOWFV XMDF solution file importer."""
# 1. Standard python modules
import logging
import os.path

# 2. Third party modules
import h5py

# 3. Aquaveo modules
from xms.datasets.dataset_io import update_geom_uuid
from xms.datasets.dataset_reader import DatasetReader

# 4. Local modules
from xms.tuflowfv.file_io import io_util


class XmdfSolutionReader:
    """Class for reading TUFLOWFV XMDF solutions."""

    def __init__(self, filenames, sim_groups, geom_uuid):
        """Constructor.

        Args:
            filenames (list[str]): Absolute paths to the XMDF solution files
            sim_groups (list[str]): H5 groups of the simulations in the XMDF files. The sim name plus '_<suffix>' if the
                suffix command has been defined for the output block. Parallel with filenames.
            geom_uuid (str): UUID of the simulation's geometry
        """
        self._logger = logging.getLogger('xms.tuflowfv')
        self._filenames = filenames
        self._sim_groups = sim_groups
        self._readers = []  # [DatasetReader] - the imported solution datasets
        self._geom_uuid = geom_uuid

    def _read_dataset(self, filename, sim_group):
        """Read solution datasets from an XMDF file.

        Args:
            filename (str): Absolute filepath to the XMDF solution file
            sim_group (str): Group path in the file to the simulation group
        """
        self._logger.info(f'Reading XMDF file: "{io_util.logging_filename(filename)}"...')
        with h5py.File(filename, 'r') as f:
            temporal_datasets = f'{sim_group}/temporal'
            if temporal_datasets not in f:
                self._logger.error('Unable to find simulation datasets in XMDF file.')
                return

            # Read min/max if we ran stats
            minimum_datasets = f'{sim_group}/minimums'
            if minimum_datasets in f:
                dataset_group_paths = [temporal_datasets, minimum_datasets, f'{sim_group}/maximums']
            else:
                dataset_group_paths = [temporal_datasets]

            for dataset_group_path in dataset_group_paths:
                datasets_group = f[dataset_group_path]
                for subgroup in datasets_group:
                    dataset_path = f'{dataset_group_path}/{subgroup}'
                    self._logger.info(f'Reading XMDF dataset from file: "{dataset_path}"...')
                    if 'Values' not in datasets_group[subgroup]:
                        self._logger.error(f'Empty dataset found in XMDF file: {dataset_path}')
                        continue
                    self._readers.append(DatasetReader(h5_filename=filename, group_path=dataset_path))

    def read(self):
        """Import all the TUFLOWFV XMDF solution datasets.

        Returns:
            list[DatasetReader]: The imported datasets. We just send whatever TUFLOWFV wrote to the XMDF format file.
        """
        for filename, sim_group in zip(self._filenames, self._sim_groups):
            try:  # If reading one barfs, continue reading other files.
                if os.path.exists(filename):
                    try:
                        update_geom_uuid(filename, self._geom_uuid, sim_group)  # Make sure the geometry UUID is linked.
                    except Exception:
                        self._logger.error('Unable to set geometry identifier in XMDF file.')
                self._read_dataset(filename, sim_group)
            except Exception as e:
                self._logger.error('Errors reading XMDF file: {str(e)}')
        return self._readers
