"""Utility methods used by the GUI modules."""
# 1. Standard python modules
import datetime
import os

# 2. Third party modules
import netCDF4
import orjson
from PySide2.QtCore import QDate, QDateTime, Qt, QTime
from PySide2.QtWidgets import QDialogButtonBox

# 3. Aquaveo modules
from xms.api.tree import tree_util
from xms.core.filesystem import filesystem as io_util
from xms.guipy import settings
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.dialogs.file_selector_dialogs import get_open_filename, get_open_foldername
from xms.guipy.time_format import ISO_DATETIME_FORMAT, string_to_datetime

# 4. Local modules


__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"


NULL_SELECTION = '(none selected)'
CSV_FILE_FILTER = 'CSV files (*.csv);;All files (*.*)'
GRID_ZPTS_FILE_FILTER = 'ASCII GRID files (*.asc);;Raster files (*.flt);;All files (*.*)'
# Use this to tell TUFLOWFV to use a default value for a NetCDF variable.
DEFAULT_VALUE_VARIABLE = '-UseDefaultTUFLOWFV-'


def get_file_selector_start_dir(label_text, proj_dir):
    """Get the directory to open file browser in.

    Args:
        label_text (str): The GUI label text associated with a file selector
        proj_dir (str): Directory of the saved project if it exists. If the project has been saved, any files
            selected at the time were converted to relative paths from the project directory.

    Returns:
        str: Directory to open file browser in. If previously selected file, use that folder. Otherwise, use the
            last directory that was stored in the registry.
    """
    start_dir = proj_dir  # Default to the project directory if there is one.
    if label_text != NULL_SELECTION:  # Start in the directory of the last selected file, if there is one.
        if not os.path.isabs(label_text):  # Stored relative to the project directory.
            label_text = io_util.resolve_relative_path(proj_dir, label_text)
        start_dir = os.path.dirname(label_text)
    if not start_dir:  # If no project location and no file previously selected, look in registry
        start_dir = settings.get_file_browser_directory()
    return start_dir


def select_file(parent, label, caption, file_filter, proj_dir, is_directory):
    """Display a file selector dialog.

    Args:
        parent (QWidget): The parent dialog
        label (QLabel): The label widget associated with the file selector.
        caption (str): The dialog caption
        file_filter (str): File extension filter
        proj_dir (str): Directory of the saved project if it exists. If the project has been saved, any files
            selected at the time were converted to relative paths from the project directory.
        is_directory (bool): True if selecting a directory, False for files
    """
    start_dir = get_file_selector_start_dir(label.text(), proj_dir)
    if is_directory:
        filename = get_open_foldername(parent, caption, start_dir)
    else:
        filename = get_open_filename(parent, caption, file_filter, start_dir)
    if filename and os.path.exists(filename):
        label.setText(filename)


def datetime_to_qdatetime(dt_literal):
    """Convert a Python datetime object to a QDateTime.

    Args:
        dt_literal (datetime.datetime): Datetime to convert

    Returns:
        (QDateTime): See description
    """
    return QDateTime(
        QDate(dt_literal.year, dt_literal.month, dt_literal.day),
        QTime(dt_literal.hour, dt_literal.minute, dt_literal.second)
    )


def get_tuflow_zero_time(to_qt=None):
    """Returns the TUFLOW zero time of 1990/01/01 00:00:00 as a datetime, QDatetime, QDateTime string.

    Args:
        to_qt (Optional[bool]): If None returns a Python datetime object. If True, returns a QDateTime. If False
            returns the default string representation for a QDateTime.

    Returns:
        Union[datetime.datetime, QDateTime, str]: See argument description
    """
    zero_time = datetime.datetime(year=1990, month=1, day=1)
    # Since to_qt can be None, we need to check explicitly for True and False
    if to_qt is True:  # Convert to Qt
        zero_time = datetime_to_qdatetime(zero_time)
    elif to_qt is False:  # Convert to a string
        zero_time = zero_time.strftime(ISO_DATETIME_FORMAT)
    return zero_time


def intialize_datetime_widget(datetime_str, datetime_widget, time_format):
    """Setup a datetime edit widget and set to use current SMS time settings.

    Args:
        datetime_str (str): The datetime in default Qt format
        datetime_widget (QDateTimeEdit): The datetime edit widget
        time_format (str): The current SMS absolute time format (using Qt format specifiers)
    """
    if not datetime_str:
        timestamp = get_tuflow_zero_time(True)
    else:  # Current SMS zero time if undefined
        timestamp = string_to_datetime(datetime_str)
    datetime_widget.setDateTime(timestamp)
    datetime_widget.setDisplayFormat(time_format)


def set_combobox_from_data(cbx, value):
    """Set a combobox option index from user data value.

    Args:
        cbx (QCombobox): The combobox to search
        value (object): The user data value

    Returns:
        int: Index of the data's combobox option or -1 if not found
    """
    index = cbx.findData(value)
    cbx.setCurrentIndex(max(index, 0))
    return index


def add_combobox_options(option_map, cbx):
    """Add a combobox's option texts and user data values.

    Args:
        option_map (dict): Mapping of the display text to TUFLOWFV data value
        cbx (QCombobox): The combobox widget.
    """
    cbx_name = cbx.objectName()
    model_vals = option_map[cbx_name][1]
    for idx, opt in enumerate(option_map[cbx_name][0]):  # Loop through the list of display option texts
        cbx.addItem(opt, model_vals[idx])  # Add the TUFLOWFV value as UserData


def build_ok_cancel_buttons(dlg, cancel=True):
    """Create a simple Ok/Cancel (no Help) button box for a dialog.

    Args:
        dlg (QDialog): Dialog to build button box for
        cancel (bool): If False, no cancel button

    Returns:
        QDialogButtonBox: The Ok/Cancel button box for dlg
    """
    button_box = QDialogButtonBox(dlg)
    button_box.setOrientation(Qt.Horizontal)
    if cancel:
        button_box.setStandardButtons(QDialogButtonBox.Cancel | QDialogButtonBox.Ok)
    else:
        button_box.setStandardButtons(QDialogButtonBox.Ok)
    button_box.accepted.connect(dlg.accept)
    button_box.rejected.connect(dlg.reject)
    return button_box


def read_default_display_options_from_registry(registry_key):
    """Retrieve the default display options from the registry.

    Args:
        registry_key (str): The registry key of the display options to load

    Returns:
        CategoryDisplayOptionList: The default options from the registry, None if not saved yet
    """
    settings_manager = settings.SettingsManager()
    json_text = settings_manager.get_setting('xmstuflowfv', registry_key)
    if not json_text:
        return None

    json_dict = orjson.loads(json_text)
    categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
    categories.from_dict(json_dict)
    return categories


def populate_gridded_bc_netcdf_variables(abs_path, comboboxes):
    """Populate comboboxes with variables from a NetCDF file.

    Args:
        abs_path (str): Absolute path to the NetCDF file
        comboboxes (list[QComboBox]): The combobox to populate

    Returns:
        bool: True if comboboxes were successfully populated, False on error
    """
    try:
        with netCDF4.Dataset(abs_path, 'r') as f:
            for _, combobox in enumerate(comboboxes):
                combobox.addItem('<use default>', DEFAULT_VALUE_VARIABLE)
                for variable in f.variables:
                    combobox.addItem(variable, variable)
    except Exception:
        for combobox in comboboxes:  # Make sure we don't leave garbage in the comboboxes and report an error state.
            combobox.clear()
        return False
    return True


def set_widget_text_to_tree_path(tree_node, item_uuid, widget):
    """Set the text of a QWidget to be the tree path of a selected item.

    Args:
        tree_node (TreeNode): The project explorer tree
        item_uuid (str): UUID of the selected item
        widget (QWidget): The QWidget whose text attribute will be updated
    """
    item_name = tree_util.build_tree_path(tree_node, item_uuid)
    if not item_name:
        widget.setText(NULL_SELECTION)
        return

    name_split = item_name.split('/', 2)
    if len(name_split) > 2:  # Chop off project and module roots
        item_name = name_split[2]
    widget.setText(item_name)
