"""TableOptions Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calcdata import VariableGroup
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules


class TableOptions(VariableGroup):
    """Provides a class that will define the site data of a culvert barrel."""

    def __init__(self, name_of_items='points', app_data=None, model_name=None, project_uuid=None,
                 min_items=1, max_items=sys.maxsize, show_increment=False, show_num_items=True):
        """Initializes the Site Data.

        Args:
            name_of_items (string): The name of the items
            app_data (AppData): The application data
            model_name (string): The model name
            project_uuid (uuid): The project uuid
            min_items (int): The minimum number of items
            max_items (int): The maximum number of items
            show_increment (bool): Show the increment
            show_num_items (bool): Show the number of items
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = 'TableOptions'
        self.type = 'TableOptions'

        # Input
        self.input['Number of items'] = Variable(f'Number of {name_of_items}', 'int', min_items,
                                                 limits=(min_items, max_items))
        self.show_num_items = show_num_items

        self.input['Table direction'] = Variable('Table direction', 'list', 0, ['vertical', 'horizontal'],
                                                 complexity=1)
        if show_increment:
            self.input['Increment'] = Variable('Incremental data', 'bool', False)

        self.warnings = []
        self.results = {}
        # self.results['Results'] = Variable('Results', 'float_list', 0.0, [], precision=precision,
        #                                   unit_type=unit_type, native_unit=native_unit,
        #                                   us_units=us_units, si_units=si_units)

    def get_can_compute(self):
        """Determine whether we have enough data to compute.

        Returns:
            True, if we can compute; otherwise, False
        """
        return True, ''

    def get_input_group(self, unknown=None):
        """Get the input group (for user-input).

        Returns
            input_vars (list of variables): input group of variables
        """
        input_vars = copy.deepcopy(self.input)

        if not self.show_num_items:
            input_vars.pop('Number of items')

        return input_vars

    def compute_data(self):
        """Compute the data.

        Returns:
            True, if successful; otherwise, False
        """
        return True
