"""Perform Unit Conversion."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class ConversionMassCalc:
    """Perform mass unit conversion computations."""

    # Lists to determine units from strings
    # ----------------------------------------------------------------------
    # Mass
    # ----------------------------------------------------------------------
    # US Units
    ton = ['US ton', 'US Ton', 'US TON', 'US tons', 'US Tons', 'US TONS']
    lb = ['lb', 'pound', 'Pound', 'POUND', 'pounds', 'Pounds', 'POUNDS']
    oz = ['oz', 'ounce', 'Ounce', 'OUNCE', 'ounces', 'Ounces', 'OUNCES']
    # SI Unis
    metric_ton = ['metric ton', 'Metric Ton', 'METRIC TON', 'metric tons', 'Metric Tons', 'METRIC TONS']
    kg = ['kg', 'kilogram', 'Kilogram', 'KILOGRAM', 'kilograms', 'Kilograms', 'KILOGRAMS']
    g = ['g', 'gram', 'gram', 'GRAM', 'grams', 'grams', 'GRAMS']

    def __init__(self):
        """Initialize the ConversionCalc Class.

        Returns:
            True if converted, False if units not found
        """

    def get_si_complementary_unit(self, from_unit):
        """Get the complementary (similar) si unit as given US unit.

        Args:
            from_unit (string): US unit

        Returns:
            Successful (bool): If the functions suceeded
            SI Unit (str): The complementary unit
        """
        if from_unit in self.ton:
            return True, 'metric ton'
        elif from_unit in self.lb:
            return True, 'kg'
        elif from_unit in self.oz:
            return True, 'g'
        return False, ''

    # ----------------------------------------------------------------------
    # Mass
    # ----------------------------------------------------------------------
    # US Units
    def convert_units(self, from_unit, to_unit, value):
        """Convert mass.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Units
        if from_unit in self.ton:
            return self.convert_from_ton(to_unit, value)
        elif from_unit in self.lb:
            return self.convert_from_pound(to_unit, value)
        elif from_unit in self.oz:
            return self.convert_from_ounce(to_unit, value)
        # SI Units
        elif from_unit in self.metric_ton:
            return self.convert_from_metric_ton(to_unit, value)
        elif from_unit in self.kg:
            return self.convert_from_kg(to_unit, value)
        elif from_unit in self.g:
            return self.convert_from_g(to_unit, value)

        return False, value

    def convert_from_ton(self, to_unit, value):
        """Convert from the ton unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.ton:
            return True, value
        elif to_unit in self.lb:
            return True, value * 2000.0
        elif to_unit in self.oz:
            return True, value * 32000.0
        # SI
        elif to_unit in self.metric_ton:
            return True, value * 0.907185
        elif to_unit in self.kg:
            return True, value * 907.185
        elif to_unit in self.g:
            return True, value * 907185.0

        else:
            return False, value

    def convert_from_pound(self, to_unit, value):
        """Convert from the pound unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.ton:
            return True, value * 0.0005
        elif to_unit in self.lb:
            return True, value
        elif to_unit in self.oz:
            return True, value * 16.0
        # SI
        elif to_unit in self.metric_ton:
            return True, value * 0.000453592
        elif to_unit in self.kg:
            return True, value * 0.453592
        elif to_unit in self.g:
            return True, value * 453.592

        else:
            return False, value

    def convert_from_ounce(self, to_unit, value):
        """Convert from the ounce unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.ton:
            return True, value * 3.125e-5
        elif to_unit in self.lb:
            return True, value * 0.0625
        elif to_unit in self.oz:
            return True, value
        # SI
        elif to_unit in self.metric_ton:
            return True, value * 2.835e-5
        elif to_unit in self.kg:
            return True, value * 0.0283495
        elif to_unit in self.g:
            return True, value * 28.3495

        else:
            return False, value

    # SI Units
    def convert_from_metric_ton(self, to_unit, value):
        """Convert from the metric ton unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.ton:
            return True, value * 1.10231
        elif to_unit in self.lb:
            return True, value * 2204.62
        elif to_unit in self.oz:
            return True, value * 35274.0
        # SI
        elif to_unit in self.metric_ton:
            return True, value
        elif to_unit in self.kg:
            return True, value * 1000.0
        elif to_unit in self.g:
            return True, value * 1e+6

        else:
            return False, value

    def convert_from_kg(self, to_unit, value):
        """Convert from the kg unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.ton:
            return True, value * 0.00110231
        elif to_unit in self.lb:
            return True, value * 2.20462
        elif to_unit in self.oz:
            return True, value * 35.274
        # SI
        elif to_unit in self.metric_ton:
            return True, value * 0.001
        elif to_unit in self.kg:
            return True, value
        elif to_unit in self.g:
            return True, value * 1000

        else:
            return False, value

    def convert_from_g(self, to_unit, value):
        """Convert from the g unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # US Unit
        if to_unit in self.ton:
            return True, value * 1.1023e-6
        elif to_unit in self.lb:
            return True, value * 0.00220462
        elif to_unit in self.oz:
            return True, value * 0.035274
        # SI
        elif to_unit in self.metric_ton:
            return True, value * 1e-6
        elif to_unit in self.kg:
            return True, value * 0.001
        elif to_unit in self.g:
            return True, value

        else:
            return False, value
