"""Variable Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path
import re
import sys
import uuid

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.core_data.units.unit_conversion import ConversionCalc
from xms.FhwaVariable.interface_adapters.view_model.variable_data import VariableData

# self.length = {}
# self.length['all_US'] = ['mi', 'yd', 'ft', 'in']
# self.length['all_SI'] = ['km', 'm', 'cm', 'mm']
# self.length['long_US'] = ['mi', 'yd', 'ft']
# self.length['long_SI'] = ['km', 'm']
# self.length['med_US'] = ['yd', 'ft', 'in']
# self.length['med_SI'] = ['m', 'cm', 'mm']
# self.length['short_US'] = ['ft', 'in', 'mm']
# self.length['short_SI'] = ['cm', 'mm']

# self.flow_all_units_US = ['cfy', 'cfs', 'stream', 'gpm', 'gpd']
# self.flow_all_units_SI = ['cmy', 'cmh', 'cmm', 'cms', 'Mld', 'Mls', 'lm', 'ls']

# self.flow_units_US = ['cfs', 'stream', 'gpm']
# self.flow_units_SI = ['cms', 'lm', 'ls']


class Variable:
    """Provides a class that will store a variable, handle interface, and file IO."""
    # Reference class-level constants from VariableData
    string_types = VariableData.string_types
    supported_color_formats = VariableData.supported_color_formats
    list_types = VariableData.list_types

    def __init__(self, name: str = '', var_type: str = '', value=0.0, value_options=None,
                 limits=(0.0, sys.float_info.max),
                 precision: int = 2, unit_type=None, native_unit: str = "", us_units=None, si_units=None,
                 note: str = "", complexity: int = 0, help_url: str = None):
        """Initializes the variable values.

        Args:
            name (string): name of the variable to be displayed in the UI
            var_type (string): Data type 'bool', 'int', 'float', 'string', 'Zip', 'Phone', 'email', 'url', 'list',
                'float_list', 'int_list', 'string_list', 'bool_list', 'class', 'file', 'image', 'color'
            value (float): Value of the variable.  Type will change as needed
            value_options (list): provides the options for list
            limits (tuple): the limits the variable is allowed to range across
            precision (int): amount of precision to display for the variable
            unit_type (): specifies types of units to provide
            native_unit (string): the native and default unit - units the variable will convert to for internal use
            us_units (list of lists): the US units to provide to the user
            si_units (list of lists): the SI units to provide to the user
            note (string): The note to display to the user.
            complexity (int): Complexity level of the variable
            help_url (string): URL to help the user understand the variable
        """
        if value_options is None:
            value_options = []
        if unit_type is None:
            unit_type = []
        if us_units is None:
            us_units = [[]]
        if si_units is None:
            si_units = [[]]

        # Get SI complementary unit
        _, selected_si_unit = ConversionCalc(None).get_si_complementary_unit(native_unit)

        # Properly handle the default value and value_options for list types
        # If value_options is not a list for list types, convert it to a list
        if var_type in Variable.list_types and not isinstance(value_options, list):
            value_options = [value_options]

        # Determine default_value based on value_options
        if var_type in Variable.list_types and isinstance(value_options, list):
            if len(value_options) > 0:
                default_value = value_options[0]
            else:
                default_value = 0.0
        else:
            default_value = value

        # Initialize the data object
        self.data = VariableData(
            name=name,
            type=var_type,
            value=value,
            default_value=default_value,
            value_options=value_options,
            limits=limits,
            precision=precision,
            unit_type=unit_type,
            native_unit=native_unit,
            selected_us_unit=native_unit,
            selected_si_unit=selected_si_unit,
            available_us_units=us_units,
            available_si_units=si_units,
            note=note,
            complexity=complexity,
            help_url=help_url,
            read_only=False,
            file_mode='existing file',
            action=None,
            set_row_as_header=False,
            data_status='normal',
            uuid=uuid.uuid4()
        )

        # Set name on value if applicable
        if not isinstance(value, Path) and hasattr(value, 'name'):
            value.name = name

        # TODO: Add an option that will remember the last value the user entered for a variable
        # (when the variable is init)
        # TODO: Add an option that will remember the last units the user entered for a variable
        # (when the variable is init)

        # An example of these features would be, specifying an option to use priority raster or highest resolution
        # raster when # merging rasters. A user is likely to have a preference that we would want to persist to the
        # next time they run the tool.
        # Or if a user wants to specify channel depth in inches instead of feet, we would want to remember that
        # preference.
        # I imagine most of the time, we would not want these to persist, but there are cases where it would be useful.

    # Property accessors for backward compatibility
    @property
    def name(self):
        """Gets the name of the variable."""
        return self.data.name

    @name.setter
    def name(self, value):
        """Sets the name of the variable."""
        self.data.name = value

    @property
    def type(self):
        """Gets the type of the variable."""
        return self.data.type

    @type.setter
    def type(self, value):
        """Sets the type of the variable."""
        self.data.type = value

    @property
    def value(self):
        """Gets the value of the variable."""
        return self.data.value

    @value.setter
    def value(self, val):
        """Sets the value of the variable."""
        self.data.value = val

    @property
    def default_value(self):
        """Gets the default value of the variable."""
        return self.data.default_value

    @default_value.setter
    def default_value(self, value):
        """Sets the default value of the variable."""
        self.data.default_value = value

    @property
    def value_options(self):
        """Gets the value options of the variable."""
        return self.data.value_options

    @value_options.setter
    def value_options(self, value):
        """Sets the value options of the variable."""
        self.data.value_options = value

    @property
    def limits(self):
        """Gets the limits of the variable."""
        return self.data.limits

    @limits.setter
    def limits(self, value):
        """Sets the limits of the variable."""
        self.data.limits = value

    @property
    def precision(self):
        """Gets the precision of the variable."""
        return self.data.precision

    @precision.setter
    def precision(self, value):
        """Sets the precision of the variable."""
        self.data.precision = value

    @property
    def unit_type(self):
        """Gets the unit type of the variable."""
        return self.data.unit_type

    @unit_type.setter
    def unit_type(self, value):
        """Sets the unit type of the variable."""
        self.data.unit_type = value

    @property
    def native_unit(self):
        """Gets the native unit of the variable."""
        return self.data.native_unit

    @native_unit.setter
    def native_unit(self, value):
        """Sets the native unit of the variable."""
        self.data.native_unit = value

    @property
    def selected_us_unit(self):
        """Gets the selected US unit of the variable."""
        return self.data.selected_us_unit

    @selected_us_unit.setter
    def selected_us_unit(self, value):
        """Sets the selected US unit of the variable."""
        self.data.selected_us_unit = value

    @property
    def selected_si_unit(self):
        """Gets the selected SI unit of the variable."""
        return self.data.selected_si_unit

    @selected_si_unit.setter
    def selected_si_unit(self, value):
        """Sets the selected SI unit of the variable."""
        self.data.selected_si_unit = value

    @property
    def available_us_units(self):
        """Gets the available US units of the variable."""
        return self.data.available_us_units

    @available_us_units.setter
    def available_us_units(self, value):
        """Sets the available US units of the variable."""
        self.data.available_us_units = value

    @property
    def available_si_units(self):
        """Gets the available SI units of the variable."""
        return self.data.available_si_units

    @available_si_units.setter
    def available_si_units(self, value):
        """Sets the available SI units of the variable."""
        self.data.available_si_units = value

    @property
    def note(self):
        """Gets the note of the variable."""
        return self.data.note

    @note.setter
    def note(self, value):
        """Sets the note of the variable."""
        self.data.note = value

    @property
    def complexity(self):
        """Gets the complexity of the variable."""
        return self.data.complexity

    @complexity.setter
    def complexity(self, value):
        """Sets the complexity of the variable."""
        self.data.complexity = value

    @property
    def help_url(self):
        """Gets the help URL of the variable."""
        return self.data.help_url

    @help_url.setter
    def help_url(self, value):
        """Sets the help URL of the variable."""
        self.data.help_url = value

    @property
    def read_only(self):
        """Gets the read-only status of the variable."""
        return self.data.read_only

    @read_only.setter
    def read_only(self, value):
        """Sets the read-only status of the variable."""
        self.data.read_only = value

    @property
    def file_mode(self):
        """Gets the file mode of the variable."""
        return self.data.file_mode

    @file_mode.setter
    def file_mode(self, value):
        """Sets the file mode of the variable."""
        self.data.file_mode = value

    @property
    def action(self):
        """Gets the action of the variable."""
        return self.data.action

    @action.setter
    def action(self, value):
        """Sets the action of the variable."""
        self.data.action = value

    @property
    def set_row_as_header(self):
        """Gets the set_row_as_header status of the variable."""
        return self.data.set_row_as_header

    @set_row_as_header.setter
    def set_row_as_header(self, value):
        """Sets the set_row_as_header status of the variable."""
        self.data.set_row_as_header = value

    @property
    def data_status(self):
        """Gets the data status of the variable."""
        return self.data.data_status

    @data_status.setter
    def data_status(self, value):
        """Sets the data status of the variable."""
        self.data.data_status = value

    @property
    def uuid(self):
        """Gets the UUID of the variable."""
        return self.data.uuid

    @uuid.setter
    def uuid(self, value):
        """Sets the UUID of the variable."""
        self.data.uuid = value

    def get_data(self):
        """Returns the data object for serialization.

        Returns:
            VariableData: The data object containing all variable data
        """
        return self.data

    def set_data(self, data: VariableData):
        """Sets the data object from deserialization.

        Args:
            data (VariableData): The data object to set
        """
        self.data = data

    def get_val(self, app_data=None):
        """Returns the value of the variable.

        Returns:
            ? (?): The return type depends on the data type (specified in self.type)
        """
        selected_unit_system = 'U.S. Customary Units'
        if app_data is not None:
            _, selected_unit_system = app_data.get_setting('Selected unit system', selected_unit_system)
        if self.data.type in ['bool', 'calc', 'class', 'file', 'int', 'string', 'table', 'UserArray', ]:
            return self.data.value
        elif self.data.type in ['float']:
            _, val = self._prepare_double(self.data.value, app_data, selected_unit_system)
            return val
        elif self.data.type in ['float_list', 'int_list', 'bool_list', 'string_list']:
            new_list = []
            for val in self.data.value_options:
                if self.data.type == 'int_list':
                    prepare_func = self._prepare_int
                elif self.data.type == 'float_list':
                    prepare_func = self._prepare_double
                elif self.data.type == 'bool_list':
                    prepare_func = self._prepare_bool
                elif self.data.type == 'string_list':
                    prepare_func = self._prepare_string
                result, new_val = prepare_func(val, app_data, selected_unit_system, set_val=False)
                if result:
                    new_list.append(new_val)
            return new_list
        elif self.data.type == 'list':
            if len(self.data.value_options):
                if self.data.value < 0:
                    self.data.value = 0
                if self.data.value >= len(self.data.value_options):
                    self.data.value = len(self.data.value_options) - 1
                return self.data.value_options[self.data.value]
            return ''
        elif self.data.type in ['calc_list']:
            return self.data.value
        elif self.data.type in ['color']:
            return self.data.value_options
        elif self.data.type in Variable.list_types:  # list is already handled, because we select one item from the list
            return self.data.value_options
        elif self.data.type in ['uuid_dict']:
            if self.data.value in self.data.value_options:
                return self.data.value
        return self.data.value

    def set_val(self, new_value, app_data=None, index=None):
        """Sets the value of the variable.

        Args:
            new_value (?): The value to set to the variable. The type depends on the data type (specified in self.type)
            app_data (AppSettingsData): The settings CalcData to use for unit conversion
            index (int): The index of the list to set the value to (if applicable)

        Returns:
            succeeded (bool): True if the value was set successfully; otherwise, False
        """
        selected_unit_system = 'U.S. Customary Units'
        if app_data is not None:
            _, selected_unit_system = app_data.get_setting('Selected unit system', selected_unit_system)
        if self.data.type in ['UserArray', 'bool', 'calc', 'class', 'file', 'table'
                              ] or self.data.type in Variable.string_types:
            if self.data.type in ['file'] or self.data.type in Variable.string_types:
                result, val = self._prepare_string(new_value, app_data, selected_unit_system)
                if result:
                    self.data.value = val
            else:
                if self.data.type == 'bool':
                    result, val = self._prepare_bool(new_value, app_data, selected_unit_system)
                    if result:
                        self.data.value = val
                else:
                    self.data.value = new_value
        elif self.data.type in ['int']:
            result, val = self._prepare_int(new_value, app_data, selected_unit_system)
            if result:
                self.data.value = val
        elif self.data.type in ['float']:
            result, val = self._prepare_double(new_value, app_data, selected_unit_system)
            if result:
                self.data.value = val
        elif self.data.type == 'list':
            if isinstance(new_value, int) or isinstance(new_value, float):
                self.data.value = int(new_value)
            else:
                if new_value in self.data.value_options:
                    self.data.value = self.data.value_options.index(new_value)
                else:
                    return False
        elif self.data.type == 'uuid_dict':
            if new_value in self.data.value_options.values():
                for key, value in self.data.value_options.items():
                    if value == new_value:
                        self.data.value = key
            elif new_value in self.data.value_options.keys():
                self.data.value = new_value
        elif self.data.type in Variable.list_types:
            # set the list, if given a list
            prepare_func = None
            if self.data.type == 'string_list':
                prepare_func = self._prepare_string
            elif self.data.type == 'float_list':
                prepare_func = self._prepare_double
            elif self.data.type == 'int_list':
                prepare_func = self._prepare_int
            try:
                self.data.value = int(self.data.value)
            except ValueError:
                self.data.value = 0
            if isinstance(new_value, list) or isinstance(new_value, tuple):
                new_list = []
                for i in range(len(new_value)):
                    if prepare_func is None:
                        new_list.append(new_value[i])
                        continue
                    result, val = prepare_func(new_value[i], app_data, selected_unit_system)
                    if result:
                        new_list.append(val)
                self.data.value_options = new_list
                self.data.value = len(self.data.value_options)
            elif index is not None:  # Set a single value in the list
                if len(self.data.value_options) > index:
                    result, val = prepare_func(new_value, app_data, selected_unit_system)
                    if result:
                        self.data.value_options[index] = val
                elif len(self.data.value_options) == index:
                    self.data.value += 1
                    result, val = prepare_func(new_value, app_data, selected_unit_system)
                    if result:
                        self.data.value_options.append(val)
            elif index is None:
                if prepare_func is None:
                    self.data.value_options = [new_value]
                    self.data.value = int(1)
                else:
                    result, val = prepare_func(new_value, app_data, selected_unit_system)
                    if result:
                        self.data.value_options = [val]
                        self.data.value = int(1)
        elif self.data.type in ['image']:
            if isinstance(new_value, np.ndarray):
                self.data.value = new_value
        elif self.data.type in ['color']:
            if new_value in Variable.supported_color_formats:
                self.data.value = new_value
            else:
                if isinstance(new_value, (tuple, list)) and len(new_value) == 3:
                    self.data.value_options = new_value
        return True

    def _prepare_string(self, new_value, app_settings, selected_unit_system, set_val=None):
        """Prepare the string value for setting the value.

        Args:
            new_value (?): The value to set to the variable. The type depends on the data type (specified in self.type)
            app_settings (AppSettingsData): The settings CalcData to use for unit conversion
            selected_unit_system (string): The unit system selected by the user

        Returns:
            succeeded (bool): True if the value was set successfully; otherwise, False
            new_value (float): The new value to set to the variable
        """
        if not isinstance(new_value, str):
            new_value = str(new_value)

        self.data_status = 'normal'
        if self.data.type == 'zip' and new_value != '':
            # Remove any non-digit characters
            digits = ''.join(filter(str.isdigit, new_value))

            # Check if the number of digits is correct for a ZIP code
            if len(digits) == 5:
                # Short form ZIP code
                new_value = digits
            elif len(digits) == 9:
                # Long form ZIP code
                formatted_zipcode = f"{digits[:5]}-{digits[5:]}"
                new_value = formatted_zipcode
            else:
                # Invalid ZIP code
                self.data_status = 'warning'
        elif self.data.type == 'phone' and new_value != '':
            # Remove any non-digit characters
            digits = ''.join(filter(str.isdigit, new_value))

            # Check if the number of digits is correct for a phone number
            if len(digits) != 10:
                self.data.data_status = 'warning'
            else:
                # Format the phone number
                formatted_number = f"({digits[:3]}) {digits[3:6]}-{digits[6:]}"
                new_value = formatted_number
        elif self.data.type == 'email' and new_value != '':
            # Regular expression for validating an email
            email_regex = r'^[a-zA-Z0-9_.+-]+@[a-zA-Z0-9-]+\.[a-zA-Z0-9-.]+$'

            if re.match(email_regex, new_value):
                # Email is valid
                pass
            else:
                # Email is invalid
                self.data.data_status = 'warning'
        elif self.data.type == 'url' and new_value != '':
            # Regular expression for validating a URL
            url_regex = re.compile(
                r'^(?:http|ftp)s?://'  # http:// or https://
                r'(?:(?:[A-Z0-9](?:[A-Z0-9-]{0,61}[A-Z0-9])?\.)+(?:[A-Z]{2,6}\.?|[A-Z0-9-]{2,}\.?)|'  # domain...
                r'localhost|'  # localhost...
                r'\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}|'  # ...or ipv4
                r'\[?[A-F0-9]*:[A-F0-9:]+\]?)'  # ...or ipv6
                r'(?::\d+)?'  # optional port
                r'(?:/?|[/?]\S+)$', re.IGNORECASE)

            if re.match(url_regex, new_value):
                # URL is valid
                pass
            else:
                # URL is invalid
                self.data.data_status = 'warning'
        return True, new_value

    def _prepare_bool(self, new_value, app_settings, selected_unit_system, set_val=None):
        """Prepare the string value for setting the value.

        Args:
            new_value (?): The value to set to the variable. The type depends on the data type (specified in self.type)
            app_settings (AppSettingsData): The settings CalcData to use for unit conversion
            selected_unit_system (string): The unit system selected by the user

        Returns:
            succeeded (bool): True if the value was set successfully; otherwise, False
            new_value (float): The new value to set to the variable
        """
        if isinstance(new_value, str):
            if new_value.lower() in ['true', 't', 'yes', 'y', '1']:
                new_value = True
            elif new_value.lower() in ['false', 'f', 'no', 'n', '0', '-1']:
                new_value = False
            else:
                try:
                    new_value = float(new_value)
                except ValueError:
                    return False, new_value
        if isinstance(new_value, float):
            if new_value > 0:
                new_value = True
            else:
                new_value = False
        if not isinstance(new_value, bool):
            new_value = bool(new_value)
        return True, new_value

    def _prepare_int(self, new_value, app_settings, selected_unit_system, set_val=None):
        """Prepare the float value for setting the value.

        Args:
            new_value (?): The value to set to the variable. The type depends on the data type (specified in self.type)
            app_settings (AppSettingsData): The settings CalcData to use for unit conversion
            selected_unit_system (string): The unit system selected by the user

        Returns:
            new_value (float): The new value to set to the variable
        """
        # Clean up type if it was incorrectly set
        if not isinstance(new_value, int):
            try:
                new_value = int(new_value)
            except ValueError:
                try:
                    new_value = float(new_value)
                    new_value = int(new_value)
                except ValueError:
                    return False, new_value
        # Convert the value to the native unit
        # if app_settings is not None:  # Convert if we were given AppSettingsData
        #     selected_unit = self.get_selected_unit(selected_unit_system)
        #     if selected_unit != "" and self.native_unit != "":
        #         unit_converter = ConversionCalc(app_settings)
        #         _, new_value = unit_converter.convert_units(selected_unit, self.native_unit, new_value)
        # Check the limits
        (lower_limit, upper_limit) = self.data.limits
        if new_value < lower_limit:
            new_value = lower_limit
        if new_value > upper_limit:
            new_value = upper_limit

        return True, new_value

    def _prepare_double(self, new_value, app_settings, selected_unit_system, set_val=True):
        """Prepare the float value for setting the value.

        Args:
            new_value (?): The value to set to the variable. The type depends on the data type (specified in self.type)
            app_settings (AppSettingsData): The settings CalcData to use for unit conversion
            selected_unit_system (string): The unit system selected by the user
            set_val (bool): If called from set_val (as opposed to get_val), then True

        Returns:
            succeeded (bool): True if the value was set successfully; otherwise, False
            new_value (float): The new value to set to the variable
        """
        # Clean up type if it was incorrectly set
        if not isinstance(new_value, float):
            try:
                if new_value is None:
                    return False, new_value
                new_value = float(new_value)
            except ValueError:
                return False, new_value
        # Convert the value to the native unit
        if app_settings is not None:  # Convert if we were given AppSettingsData
            selected_unit = self.get_selected_unit(selected_unit_system)
            if selected_unit != "" and self.data.native_unit != "":
                unit_converter = ConversionCalc(app_settings)
                if set_val:
                    _, new_value = unit_converter.convert_units(selected_unit, self.data.native_unit, new_value)
                else:
                    _, new_value = unit_converter.convert_units(self.data.native_unit, selected_unit, new_value)
        # Check the limits
        (lower_limit, upper_limit) = self.data.limits
        if new_value < lower_limit:
            new_value = float(lower_limit)
        if new_value > upper_limit:
            new_value = float(upper_limit)

        return True, new_value

    def add_result(self, val):
        """Add result to a list.

        Args:
            val (?): The value to set to the variable. The type depends on the data type (specified in self.type)
        """
        if val is None:
            return
        if self.data.type in ['float_list', 'int_list', 'bool_list', 'string_list'
                              ] and isinstance(val, (float, str, int)):
            # If you passed a list, it would extend the existing variable; what if we want a list of lists?
            self.data.value_options.append(val)
            self.data.value = len(self.data.value_options)

    def set_index(self, var):
        """Sets the index of a list (by index or option).

        Args:
            var (?): The value to set to the variable. The type depends on the data type (specified in self.type)
        """
        if self.data.type in ['bool', 'int', 'float', 'float_list', 'int_list', 'bool_list', 'string_list', 'calc_list',
                              'calc', 'class', 'UserArray']:
            return
        elif self.data.type == 'list':
            if isinstance(var, int):
                # Make sure the variable is within limits; set tp closest limit
                if var < 0:
                    var = 0
                if var >= len(self.data.value_options):
                    var = len(self.data.value_options) - 1
                self.data.value = var
            else:
                self.data.value = self.data.value_options.index(var)

    def get_index(self, null_value, val=None):
        """Returns the index of a list.

        Args:
            null_value (float): what to return if it is not found
            val (?): The value to set to the variable. The type depends on the data type (specified in self.type)
        """
        if val is None or val == '':
            return self.data.value
        if val not in self.data.value_options:
            return null_value
        index = null_value
        if self.data.type in ['list', 'float_list', 'int_list', 'bool_list', 'string_list', 'calc_list', 'UserArray']:
            index = self.data.value_options.index(val)
        return index

    def append(self, val):
        """Append to the list.

        Args:
            val (?): The value to set to the variable. The type depends on the data type (specified in self.type)
        """
        if self.data.type in ['bool', 'int', 'float', 'class', 'UserArray']:
            return
        elif self.data.type in Variable.list_types:
            self.data.value_options.append(val)

    def get_list(self):
        """Returns the list (value options).

        Returns
            value_options (list): list of options available
        """
        return self.data.value_options

    def set_list(self, new_list):
        """Sets the list of items.

        Args:
            new_list (list): The new list
        """
        self.data.value_options = new_list

    def get_selected_unit(self, selected_unit_system):
        """Returns the selected units."""
        selected_unit = self.data.selected_si_unit
        if selected_unit_system in ['U.S. Customary Units', 'Both']:
            selected_unit = self.data.selected_us_unit
        return selected_unit

    def set_selected_unit(self, selected_unit_system, selected_unit):
        """Returns the selected units."""
        if selected_unit_system in ['', None]:
            return None
        if selected_unit in ['', None]:
            return None
        if selected_unit_system in ['U.S. Customary Units', 'Both']:
            self.data.selected_us_unit = selected_unit
        else:
            self.data.selected_si_unit = selected_unit

        return selected_unit

    def _remove_duplicates(self, units_list):
        """Removes duplicates from a list.

        Args:
            units_list (list): The list to remove duplicates from
        """
        new_units_list = []
        for unit in units_list:
            if unit not in new_units_list:
                new_units_list.append(unit)
        return new_units_list

    def get_units_list_and_index(self, selected_unit_system):
        """Returns the units list and index of the selected units.

        Args:
            selected_unit_system (string): The unit system selected by the user
        """
        units_list = []
        index = None
        selected_unit = self.data.selected_si_unit
        # if len(self.data.unit_type) == 0:
        #     return units_list, index

        if selected_unit_system in ['U.S. Customary Units', 'Both']:
            units_list.extend(self.data.available_us_units[0])
            selected_unit = self.data.selected_us_unit
        if selected_unit_system in ['SI Units (Metric)', 'Both']:
            units_list.extend(self.data.available_si_units[0])
            units_list = self._remove_duplicates(units_list)

        if len(units_list) <= 0:
            return units_list, None
        elif selected_unit == "":
            index = 0
            if selected_unit_system in ['U.S. Customary Units', 'Both']:
                self.data.selected_us_unit = units_list[index]
            else:
                self.data.selected_si_unit = units_list[index]
            return units_list, index

        index = None
        try:
            index = units_list.index(selected_unit)
        except ValueError:
            pass
        return units_list, index

    def check_limits(self, result, warnings, name=None):
        """Checks that the value (if applicable) is within limits.

        Args:
            result (bool): Set to False if anything fails a check; otherwise, UNMODIFIED
            warnings (list): list of strings of warnings to provide to the user
            name (string): Additional name for context to the user in the error message
        """
        if self.data.type in ['float', 'int']:
            if self.data.limits[0] > self.data.value or self.data.value > self.data.limits[1]:
                result = False
                txt_name = self.data.name
                if name is not None:
                    txt_name = name + ': ' + self.data.name
                warnings.append(f'Please select a value for {txt_name} that is within its limits.')
        if self.data.type in ['UserArray']:
            if not self.get_val().check_limits(result, warnings):
                result = False
        return result

    def find_item_by_uuid(self, uuid):
        """Finds a variable by its uuid.

        Args:
            uuid (string): uuid of the variable

        Returns:
            result: True if item found
            variable: the variable
        """
        if self.data.uuid == uuid:
            return True, self

        if self.data.type in ['class', 'calc', 'uuid_dict', 'UserArray', 'table']:
            item = self.get_val()
            if hasattr(item, 'find_item_by_uuid'):
                result, item = self.get_val().find_item_by_uuid(uuid)
                if result:
                    return result, item

        return False, None
