"""Classes to Define the model-view of a table."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from dataclasses import dataclass, field
from uuid import UUID, uuid4

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


@dataclass
class Cell:
    """A class to define the model-view of a cell.

    Args:
        cell_type: The type of cell ('string', 'float', 'int', 'bool', 'list', 'button', 'color')
        value: The value of the cell
        value_options: The options for the value of the cell
        is_read_only: If the cell is editable
        bg_color: The background color of the cell
        text_color: The text color of the cell
        variable_type: The type of variable (by variable definition-- used for validators)
    """
    cell_type: str
    value: str | float | int | bool | None = None
    value_options: list | None = None
    is_read_only: bool = True
    bg_color: str | None = None
    text_color: str | None = None
    variable_type: str | None = None
    is_visible: bool = True
    lower_limit: float | None = None
    upper_limit: float | None = None


@dataclass
class Row:
    """A class to define the model-view of a row.

    Args:
        name_cell: The name cell of the row
        value_cell: The value cell of the row
        unit_cell: The unit cell of the row
        note_cell: The note cell of the row
        type_cell: The type cell of the row
        key_cell: The key cell of the row
        uuid_cell: The uuid cell of the row
        variable_type: The type of variable
        name: The name of the row
        value: The value of the row
        value_options: The options for the value of the row
        text_color: The text color of the row
        bg_color: The background color of the row
        name_is_read_only: If the name is editable
        value_is_read_only: If the value is editable
        unit_options: The options for the unit of the row
        selected_unit: The selected unit of the row
        note: The note of the row
        key: The key of the row
        uuid: The uuid of the row
    """
    name_cell: Cell | None = None
    value_cell: Cell | None = None
    unit_cell: Cell | None = None
    note_cell: Cell | None = None
    type_cell: Cell | None = None
    key_cell: Cell | None = None
    uuid_cell: Cell | None = None
    variable_type: str | None = None
    name: str | None = None
    value: str | float | int | bool | None = None
    value_options: list | None = None
    text_color: str | None = None
    bg_color: str | None = None
    name_is_read_only: bool = True
    value_is_read_only: bool = False
    unit_options: list | None = None
    selected_unit: int | str = 0
    note: str = ''
    key: str = ''
    uuid: str | None = None
    value_cells: list = field(default_factory=list, init=False)

    def __post_init__(self):
        """Initialize cells after dataclass initialization."""
        if self.name_cell is None:
            self.name_cell = Cell(cell_type='string', value=self.name, is_read_only=self.name_is_read_only,
                                  text_color=self.text_color, bg_color=self.bg_color)
        if self.value_cell is None:
            value = self.value
            if value is None:
                value = ''
            cell_type = 'string'
            if isinstance(value, float):
                cell_type = 'float'
            elif isinstance(value, int):
                cell_type = 'int'
            elif isinstance(value, bool):
                cell_type = 'bool'
            self.value_cell = Cell(cell_type=cell_type, value=value, value_options=self.value_options,
                                   is_read_only=self.value_is_read_only, text_color=self.text_color,
                                   bg_color=self.bg_color, variable_type=self.variable_type)
        if self.unit_cell is None:
            cell_type = 'list'
            selected_unit = self.selected_unit
            if self.unit_options is None or selected_unit is None:
                cell_type = 'string'
                selected_unit = ''
                unit_options = None
            else:
                unit_options = self.unit_options
            self.unit_cell = Cell(cell_type=cell_type, value=selected_unit, value_options=unit_options,
                                  is_read_only=False, text_color=self.text_color, bg_color=self.bg_color)
        if self.note_cell is None:
            self.note_cell = Cell(cell_type='string', value=self.note, is_read_only=True,
                                  text_color=self.text_color, bg_color=self.bg_color)
        if self.type_cell is None:
            self.type_cell = Cell(cell_type='string', value=self.variable_type, is_read_only=True,
                                  text_color=self.text_color, bg_color=self.bg_color)
        if self.key_cell is None:
            self.key_cell = Cell(cell_type='string', value=self.key, is_read_only=True,
                                 text_color=self.text_color, bg_color=self.bg_color)
        if self.uuid_cell is None:
            self.uuid_cell = Cell(cell_type='string', value=self.uuid, is_read_only=True,
                                  text_color=self.text_color, bg_color=self.bg_color)


@dataclass
class Table:
    """A class to define the model-view of a table.

    Args:
        table_uuid: The uuid of the table
        rows: The rows of the table
        header_labels: The header labels of the table
    """
    table_uuid: UUID | None = None
    rows: list = field(default_factory=list)
    header_labels: list = field(default_factory=list)
    table_is_vertical: bool = False
    col_name: int = 0
    col_val: int = 1
    col_unit: int = 2
    col_note: int = 3
    col_type: int = 4
    col_key: int = 5
    col_uuid: int = 6
    text_color: str | None = None
    bg_color: str | None = None
    num_columns: int = 7
    num_rows: int = 0
    sys_complexity: int = 0
    max_cols: int = 7
    uuid: UUID = field(init=False)

    def __post_init__(self):
        """Initialize UUID after dataclass initialization."""
        self.uuid = self.table_uuid if self.table_uuid is not None else uuid4()

        # TODO: Consider adding the following:
        # self.has_focus
        # self.is_selected
        # self.selected_cell(s) (just had a is_selected to each cell)
        # self.horiz_scroll_pos
        # self.vert_scroll_pos

    def add_row(self, row=None, num_cols=None, row_index=None, name=None, value=None, value_options=None,
                text_color=None, bg_color=None, name_is_read_only=True, value_is_read_only=True,
                unit_options=None, selected_unit=0, note='', uuid=None):
        """Add a row to the table.

        Args:
            row (Row): The row to add
            num_cols (int): The number of columns in the table
            row_index (int): The index of the row to add
            name (string): The name of the row
            value (float): The value of the row
            value_options (list): The options for the value of the row
            text_color (string): The text color of the row
            bg_color (string): The background color of the row
            name_is_read_only (bool): If the name is editable
            value_is_read_only (bool): If the value is editable
            unit_options (list): The options for the unit of the row
            selected_unit (int): The selected unit of the row
            note (string): The note of the row
            uuid (string): The uuid of the row
        """
        if num_cols is None:
            if not isinstance(self.num_columns, int):
                try:
                    self.num_columns = int(self.num_columns)
                except Exception:
                    self.num_columns = 7
            num_cols = self.num_columns
        if row is None:
            row = Row(name=name, value=value, value_options=value_options,
                      text_color=text_color, bg_color=bg_color, name_is_read_only=name_is_read_only,
                      value_is_read_only=value_is_read_only, unit_options=unit_options,
                      selected_unit=selected_unit, note=note, uuid=uuid)
            for _ in range(7, num_cols):
                row.value_cells.append(Cell(cell_type='float', value=value, value_options=None, is_read_only=False,
                                            text_color=text_color, bg_color=bg_color))
        if row_index is None or row_index < 0:
            self.rows.append(row)
        else:
            if len(self.rows) <= row_index:
                while len(self.rows) <= row_index:
                    self.rows.append(row)
            elif 0 <= row_index < len(self.rows):
                self.rows[row_index] = row

    def set_item(self, row_index, col_index, mv_cell):
        """Set the value of a cell in the table.

        Args:
            row_index (int): The row index of the cell
            col_index (int): The column index of the cell
            mv_cell (Cell): The cell to set
        """
        while row_index > len(self.rows) - 1:
            self.add_row()
        if col_index == self.col_name:
            self.rows[row_index].name_cell = mv_cell
        elif col_index == self.col_val:
            self.rows[row_index].value_cell = mv_cell
        elif col_index == self.col_unit:
            self.rows[row_index].unit_cell = mv_cell
        elif col_index == self.col_note:
            self.rows[row_index].note_cell = mv_cell
        elif col_index == self.col_type:
            self.rows[row_index].type_cell = mv_cell
        elif col_index == self.col_key:
            self.rows[row_index].key_cell = mv_cell
        elif col_index == self.col_uuid:
            self.rows[row_index].uuid_cell = mv_cell
        elif 0 <= col_index < max(
            self.col_name,
            self.col_val,
            self.col_unit,
            self.col_note,
            self.col_type,
            self.col_key,
            self.col_uuid
        ):
            value_index = col_index - self.col_val
            if len(self.rows[row_index].value_cells) <= value_index - 1:
                self.rows[row_index].value_cells.append(mv_cell)
            else:
                self.rows[row_index].value_cells[value_index - 1] = mv_cell
        else:
            return False
        return True

    def get_item(self, row_index, col_index):
        """Get the value of a cell in the table.

        Args:
            row_index (int): The row index of the cell
            col_index (int): The column index of the cell

        Returns:
            string: The value of the cell
        """
        max_col = max(
            self.col_name,
            self.col_val,
            self.col_unit,
            self.col_note,
            self.col_type,
            self.col_key,
            self.col_uuid
        )
        if 0 <= row_index < len(self.rows):
            if col_index == self.col_name:
                return self.rows[row_index].name_cell
            elif col_index == self.col_val:
                return self.rows[row_index].value_cell
            elif col_index == self.col_unit:
                return self.rows[row_index].unit_cell
            elif col_index == self.col_note:
                return self.rows[row_index].note_cell
            elif col_index == self.col_type:
                return self.rows[row_index].type_cell
            elif col_index == self.col_key:
                return self.rows[row_index].key_cell
            elif col_index == self.col_uuid:
                return self.rows[row_index].uuid_cell
            elif 0 <= col_index < max_col:
                value_index = col_index - self.col_val
                return self.rows[row_index].value_cells[value_index]
            else:
                return None
        return None

    def set_max_cols(self, num_cols):
        """Set the maximum number of columns in the table.

        Args:
            num_cols (int): The maximum number of columns in the table
        """
        max_cols = max(self.max_cols, num_cols)
        self.max_cols = max_cols

    def set_num_rows(self, num_rows):
        """Set the number of rows in the table.

        Args:
            num_rows (int): The number of rows in the table
        """
        self.num_rows = num_rows

    def get_num_rows(self):
        """Get the number of rows in the table.

        Returns:
            int: The number of rows in the table
        """
        if not self.table_is_vertical:
            return len(self.rows)
        else:
            return max(self.num_columns, self.max_cols)

    def set_num_cols(self, num_cols):
        """Set the number of rows in the table.

        Args:
            num_cols (int): The number of columns in the table
        """
        self.num_columns = num_cols

    def get_num_cols(self):
        """Get the number of rows in the table.

        Returns:
            int: The number of rows in the table
        """
        if not self.table_is_vertical:
            return max(self.num_columns, self.max_cols)
        else:
            return len(self.rows)
