"""Abstract Class that defines the base functionality for a Graphics View plug-in."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from abc import ABC, abstractmethod
from typing import Tuple
import uuid

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.interface_adapters.view_model.main.window import GraphicsImage, Orientation


class GraphicsViewBase(ABC):
    """Provides a class that will take orientation, settings, and datahandler and returna rendered image."""

    @abstractmethod
    def render(self, view_uuid: uuid.UUID, orientation: Orientation, width: int, height: int) -> GraphicsImage:
        """Renders the image using the given orientation, settings, and data handler.

        Args:
            view_uuid (uuid.UUID): the uuid of the view
            orientation (Orientation): the orientation of the view
            width (px): the width of the view
            height (px): the height of the view

        Returns:
            image (GraphicsImage): the rendered image
        """
        pass

    @abstractmethod
    def set_vtk_scalars(self, view_uuid: uuid.UUID, grid_uuid: uuid.UUID, dataset, timestep):
        """Sets the data handler for the view.

        Args:
            view_uuid (uuid.UUID): the uuid of the view
            grid_uuid (uuid.UUID): the uuid of the grid
            dataset: the dataset to be set for the view
            timestep: the timestep to be set for the view
        """
        pass

    @abstractmethod
    def set_vtk_data_cache(self, view_uuid: uuid.UUID, view_data):
        """Sets the data handler for the view.

        Args:
            view_uuid (uuid.UUID): the uuid of the view
            view_data: the data to be set for the view
        """
        pass

    @abstractmethod
    def set_vtk_display_settings(self, view_uuid: uuid.UUID, grid_uuid: uuid.UUID, settings):
        """Sets the data handler for the view.

        Args:
            view_uuid (uuid.UUID): the uuid of the view
            grid_uuid (uuid.UUID): the uuid of the grid
            settings: the display settings for the view
        """
        pass

    # What we discussed
    @abstractmethod
    def gui_tool_change_old(self, view_uuid: uuid.UUID, orientation: Orientation, dh: int, df: int, tool='pan'
                            ) -> Orientation:
        """Handles the tool change event.

        Args:
            view_uuid (uuid.UUID): the uuid of the view
            orientation (Orientation): the current orientation of the view
            dh (DataHandler): the data handler for the view
            df (DataFrame): the data frame for the view
            tool (str): the tool to be used

        Returns:
            orientation (Orientation): the updated orientation of the view
        """
        pass

    # Proposed
    @abstractmethod
    def gui_tool_change(self, view_uuid: uuid.UUID, orientation: Orientation, point_1: Tuple[int, int],
                        point_2: Tuple[int, int],
                        tool='pan') -> Orientation:
        """Handles the tool change event.

        Args:
            view_uuid (uuid.UUID): the uuid of the view
            orientation (Orientation): the current orientation of the view
            dh (DataHandler): the data handler for the view
            df (DataFrame): the data frame for the view
            tool (str): the tool to be used

        Returns:
            orientation (Orientation): the updated orientation of the view
        """
        pass

    @abstractmethod
    def frame_view(self, view_uuid: uuid.UUID) -> Orientation:
        """Frames the view using the given orientation and uuid list.

        Args:
            view_uuid (uuid.UUID): the uuid of the view

        Returns:
            orientation (Orientation): the updated orientation of the view
        """
        pass

    @abstractmethod
    def set_orientation(self, view_uuid: uuid.UUID, orientation: Orientation, use_height=False) -> Orientation:
        """Sets the orientation of the view.

        Args:
            view_uuid (uuid.UUID): the uuid of the view
            orientation (Orientation): the orientation to be set for the view
            use_height (bool): whether to use height or not

        Returns:
            orientation (Orientation): the updated orientation of the view
        """
        pass

    # next/prev view ? tracked in vtk or in our code?
