"""Abstract Class that defines the base functionality for a File IO plug-in."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from abc import ABC, abstractmethod

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class ReadManagerBase(ABC):
    """Provides a base class that will define the functions needed for reading a file."""

    def __init__(self, app_name, app_version):
        """Initialize the ExportCalculator Class.

        Args:
            app_name (string): name of the application (checks that we are reading the correct file)
            app_version (string): current version of the software being used
        """
        self.app_name = app_name
        self.app_version_str = app_version
        # self.app_version = float('.'.join(self.app_version_str.split('.')[:2]))
        self.major_version = int(self.app_version_str.split('.')[0])
        self.minor_version = int(self.app_version_str.split('.')[1])
        self.patch_version = int(self.app_version_str.split('.')[2])
        self.file_version = ''

    def compare_file_version(self, file_version_str, ignore_patch=False):
        """Compares the file version with the current version of the software.

        Args:
            file_version (string): version of the file being read

        Returns:
            True if the file version is compatible with the current version of the software.
            False if the file version is not compatible.
        """
        self.file_major_version = int(file_version_str.split('.')[0])
        self.file_minor_version = int(file_version_str.split('.')[1])
        self.file_patch_version = int(file_version_str.split('.')[2])

        if self.file_major_version > self.major_version:
            return 'File is newer'
        elif self.file_major_version < self.major_version:
            return 'File is older'
        else:
            if self.file_minor_version > self.minor_version:
                return 'File is newer'
            elif self.file_minor_version < self.minor_version:
                return 'File is older'
            elif ignore_patch:
                return 'versions match'
            else:
                if self.file_patch_version > self.patch_version:
                    return 'File is newer'
                elif self.file_patch_version < self.patch_version:
                    return 'File is older'
                else:
                    return 'versions match'

    @abstractmethod
    def read(self, filename, app_data=None, model_name=None, is_setting: bool = False):
        """Exports the calcdatas to a given filename.

        Args:
            filename (string): filename to read
            app_data (AppData): the application data
            model_name (string): the name of the model
            is_setting (bool): True if the item is a setting

        Return:
            calcdatas (list): the list of calcdatas to save to a file
        """
        pass
