"""CalcData for performing Pier Scour calculations."""
__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calculator_list import CalcOrVarlist
from xms.FhwaVariable.core_data.calculator.plot.plot_options import PlotOptions
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.bridge_scour_calc import BridgeScourCalc
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.geometry.bridge_geometry import BridgeGeometry
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.gradations.gradation_manager_data import GradationManagerData
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.scenario_data import Scenario
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.scour_base_data import ScourBaseData
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.water_properties.water_properties import BridgeWaterProperties
from xms.HydraulicToolboxCalc.util.interpolation import Interpolation
from xms.HydraulicToolboxCalc.util.list_utils import remove_nans


class BridgeScourData(ScourBaseData):
    """A class that defines a pier scour at a bridge contraction."""

    def __init__(self, clear_my_own_results=True, app_data=None, model_name=None, project_uuid=None):
        """Initializes the Bridge scour calculator.

        Args:
            clear_my_own_results (bool): Should the class clean up its results?  If Standalone, yet. otherwise, no.
            app_data (AppData): The application data.
            model_name (str): The name of the model.
            project_uuid (str): The project UUID.
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = "Bridge Scour Project"
        self.type = "BridgeScourData"
        self.class_name = 'Bridge Scour Project'

        self.calculator = BridgeScourCalc()

        self.calc_support_dual_input = False
        self.min_y_size_for_plot_and_wiki = 1440

        # Input
        self.input = {}  # Clear the base class input

        # Bridge Geometry
        self.input['Bridge geometry'] = Variable('Bridge geometry', 'class', BridgeGeometry(
            app_data=app_data, model_name=model_name, project_uuid=project_uuid))

        # Water properties
        self.input['Water properties'] = Variable('Water properties', 'class', BridgeWaterProperties(
            app_data=app_data, model_name=model_name, project_uuid=project_uuid), complexity=1)

        # Gradations
        self.input['Gradations'] = Variable('Gradations', 'class', GradationManagerData(
            app_data=app_data, model_name=model_name, project_uuid=project_uuid))
        # self.input['Gradations'].set_row_as_header = True

        # Scenario
        #   Pier Scour
        self.input['Scenarios'] = Variable(
            'Scenarios', 'calc_list',
            CalcOrVarlist(Scenario, show_define_btn=False, default_name='Scenario', default_plural_name='Scenarios',
                          min_number_of_items=1, app_data=app_data, model_name=model_name, project_uuid=project_uuid))

        station_var = Variable('Station', 'float_list', 0, [0.0], precision=2,
                               limits=(-sys.float_info.max, sys.float_info.max), unit_type=['length'],
                               native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)
        elevation_var = Variable('Elevation', 'float_list', 0, [0.0], precision=2,
                                 limits=(-sys.float_info.max, sys.float_info.max), unit_type=['length'],
                                 native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

        station_vs_elevation = {}
        station_vs_elevation['Station'] = station_var
        station_vs_elevation['Elevation'] = elevation_var

        self.theme = self.get_theme()

        self.input_dict = {}

        self.min_y = sys.float_info.max

        self.input_dict['Long-term degradation'] = station_vs_elevation
        self.input_dict['Contraction scour'] = station_vs_elevation
        self.input_dict['Abutment scour'] = station_vs_elevation
        self.input_dict['Pier scour'] = station_vs_elevation
        self.input_dict['Total scour'] = station_vs_elevation

        self.input['Plot options'] = {}
        self.add_plot_options_if_needed()

        self.unknown = None

        # Intermediate
        self.compute_prep_functions.extend([self.update_bridge_geometry, self.update_plots,
                                            self.update_gradation_and_properties])
        self.intermediate_to_copy.extend(['d50', 'upstream_d50', 'contraction_scour_depth'])

        self.d50 = 0.0
        self.upstream_d50 = 0.0

        self.contraction_scour_depth = 0.0

        # Results
        self.clear_my_own_results = clear_my_own_results

        self.results = {}
        # Geometry data
        self.results['Geometry data'] = {}
        # ['Geometry data'] Cross-sectional data
        self.results['Geometry data']['Cross-sectional data'] = {}
        self.results['Geometry data']['Cross-sectional data']['Thalweg station'] = Variable(
            'Thalweg station',
            'float_list',
            0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Geometry data']['Cross-sectional data']['Thalweg elevation'] = Variable(
            'Thalweg elevation (prior to scour)',
            'float_list',
            0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.results['Geometry data']['Cross-sectional data']['Left bank max elevation'] = Variable(
            'Left bank max elevation (prior to scour)',
            'float_list',
            0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.results['Geometry data']['Cross-sectional data']['Right bank max elevation'] = Variable(
            'Right bank max elevation (prior to scour)',
            'float_list',
            0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        # ['Geometry data'] Bridge deck data
        self.results['Geometry data']['Bridge deck data'] = {}
        self.results['Geometry data']['Bridge deck data']['Bridge low-chord'] = Variable(
            'Bridge low-chord',
            'float_list',
            0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.results['Geometry data']['Bridge deck data']['Bridge high-chord'] = Variable(
            'Bridge high-chord',
            'float_list',
            0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        # ['Geometry data'] Pier geometry data
        # Gradation & water data

        # Scenario data
        self.results['Scenario data'] = {}
        # ['Scenario data'] WSE
        self.results['Scenario data']['WSE'] = Variable(
            'Average WSE',
            'float_list',
            0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        # ['Scenario data'] Long-term degradation
        self.results['Scenario data']['Long-term degradation'] = {}
        self.results['Scenario data']['Long-term degradation']['Long-term degradation'] = Variable(
            'Long-term degradation',
            'float_list',
            0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.results['Scenario data']['Long-term degradation']['Streambed thalweg elevation'] = Variable(
            'Streambed thalweg elevation',
            'float_list',
            0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.results['Scenario data']['Long-term degradation']['Minimum channel elevation with LTD'] = Variable(
            'Minimum channel elevation with LTD',
            'float_list',
            0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        # ['Scenario data'] Contraction scour
        self.results['Scenario data']['Contraction scour'] = {}
        self.results['Scenario data']['Contraction scour']['Contraction scour method'] = Variable(
            'Contraction scour method',
            'string_list',
            '', [], limits=(None, None),
            precision=0,
            unit_type=[''],
            native_unit='',
            us_units='',
            si_units='')
        self.results['Scenario data']['Contraction scour']['Applied contraction scour depth'] = Variable(
            'Applied contraction scour depth',
            'float_list',
            0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.results['Scenario data']['Contraction scour']['Clear-water scour depth'] = Variable(
            'Clear-water scour depth',
            'float_list',
            0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.results['Scenario data']['Contraction scour']['Live-bed scour depth'] = Variable(
            'Live-bed scour depth',
            'float_list',
            0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.results['Scenario data']['Contraction scour']['Pressure scour depth'] = Variable(
            'Pressure scour depth',
            'float_list',
            0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.results['Scenario data']['Contraction scour']['Cohesive scour depth'] = Variable(
            'Cohesive scour depth',
            'float_list',
            0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.results['Scenario data']['Contraction scour']['Streambed thalweg elevation'] = Variable(
            'Streambed thalweg elevation',
            'float_list',
            0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.results['Scenario data']['Contraction scour']['Minimum channel elevation with contraction scour and LTD'] \
            = Variable('Minimum channel elevation with contraction scour and LTD', 'float_list',
                       0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
                       precision=2,
                       unit_type=['length'],
                       native_unit='ft',
                       us_units=self.us_mid_length,
                       si_units=self.si_mid_length)

        # ['Scenario data'] Pier scour
        self.results['Scenario data']['Pier scour'] = {}
        self.results['Scenario data']['Pier scour']['Pier computation method'] = Variable(
            'Pier computation method',
            'string_list',
            '', [], limits=(None, None),
            precision=0,
            unit_type=[''],
            native_unit='',
            us_units='',
            si_units='')
        self.results['Scenario data']['Pier scour']['Pier scour depth'] = Variable(
            'Pier scour depth',
            'float_list',
            0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.results['Scenario data']['Pier scour']['Max flow depth including pier scour'] = Variable(
            'Max flow depth including pier scour',
            'float_list',
            0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.results['Scenario data']['Pier scour']['Total scour at pier'] = Variable(
            'Total scour at pier',
            'float_list',
            0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.results['Scenario data']['Pier scour']['Scour reference point for pier'] = Variable(
            'Scour reference point for pier',
            'string_list',
            '', [], limits=(None, None),
            precision=0,
            unit_type=[''],
            native_unit='',
            us_units='',
            si_units='')
        self.results['Scenario data']['Pier scour']['Streambed thalweg elevation'] = Variable(
            'Streambed thalweg elevation',
            'float_list',
            0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.results['Scenario data']['Pier scour']['Streambed elevation at pier'] = Variable(
            'Streambed elevation at pier',
            'float_list',
            0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.results['Scenario data']['Pier scour']['Total scour elevation at pier'] = Variable(
            'Total scour elevation at pier',
            'float_list',
            0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        # ['Scenario data'] Abutment scour
        self.results['Scenario data']['Abutment scour'] = {}
        self.results['Scenario data']['Abutment scour']['Abutment scour depth'] = Variable(
            'Abutment scour depth',
            'float_list',
            0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.results['Scenario data']['Abutment scour']['Max flow depth including abutment scour'] = Variable(
            'Max flow depth including abutment scour',
            'float_list',
            0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.results['Scenario data']['Abutment scour']['Total scour at abutment'] = Variable(
            'Total scour at abutment',
            'float_list',
            0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.results['Scenario data']['Abutment scour']['Streambed thalweg elevation'] = Variable(
            'Streambed thalweg elevation',
            'float_list',
            0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.results['Scenario data']['Abutment scour']['Streambed elevation at abutment'] = Variable(
            'Streambed elevation at abutment',
            'float_list',
            0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.results['Scenario data']['Abutment scour']['Total scour elevation at abutment'] = Variable(
            'Total scour elevation at abutment',
            'float_list',
            0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        # ['Scenario data'] Total scour
        self.results['Scenario data']['Total scour'] = {}
        self.results['Scenario data']['Total scour']['Minimum scour depth'] = Variable(
            'Minimum scour depth',
            'float_list',
            0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)
        self.results['Scenario data']['Total scour']['Minimum scour type'] = Variable(
            'Minimum scour type',
            'string_list',
            '', [], limits=(None, None),
            precision=0,
            unit_type=[''],
            native_unit='',
            us_units='',
            si_units='')
        self.warnings = []

        # Reference Manuals
        self.reference_pdfs['HEC-18'] = (
            'HEC-18, 5th Edition - Evaluating Scour at Bridges.pdf')

        # plot
        self.plots = {}

    def add_plot_options_if_needed(self):
        """Add plot options for scenarios if they do not exist."""
        for scenario in self.input['Scenarios'].get_val().item_list:
            if scenario.name not in self.input['Plot options']:
                self.add_plot_option_for_scenario(scenario.name)

    def add_plot_option_for_scenario(self, scenario_name):
        """Add plot options for a scenario.

        Args:
            scenario_name (string): scenario to add plot options for
        """
        channel_bank_color = self.theme['Plot channel bank line color']
        abutment_toe_color = self.theme['Plot abutment toe line color']
        # contrast_color = (255, 255, 255)  # White
        # if self.theme['Is theme dark']:
        #     contrast_color = (0, 0, 0)  # Black
        lateral_color = (52, 115, 14)  # Green

        ltd_color = self.theme['Plot long-term degradation color']
        ltd_fill_color = self.theme['Plot long-term degradation fill color']
        contraction_color = self.theme['Plot contraction scour color']
        contraction_fill_color = self.theme['Plot contraction scour fill color']
        pier_color = self.theme['Plot pier scour color']
        pier_fill_color = self.theme['Plot pier scour fill color']
        abut_color = self.theme['Plot abutment scour color']
        abut_fill_color = self.theme['Plot abutment scour fill color']

        soil_color = self.theme['Plot soil color']
        soil_fill_color = self.theme['Plot soil fill color']

        self.input['Plot options'][scenario_name] = Variable(
            'Plot options', 'class', PlotOptions('Bridge cross-section',
                                                 input_dict=self.input_dict,
                                                 show_series=True,
                                                 app_data=self.app_data,
                                                 model_name=self.model_name,
                                                 project_uuid=self.project_uuid),
            complexity=1)

        # Lines
        self.input['Plot options'][scenario_name].get_val().set_plot_line_options(
            0, 'Channel bank',
            line_intercept=None, line_alignment='vertical', line_color=channel_bank_color, linetype='dotted',
            line_width=1.5, text_color=channel_bank_color, text_alignment='top')
        self.input['Plot options'][scenario_name].get_val().set_plot_line_options(
            1, 'Abutment toe',
            line_intercept=None, line_alignment='vertical', line_color=abutment_toe_color, linetype='dashed',
            line_width=1.5, text_color=abutment_toe_color, text_alignment='top')
        self.input['Plot options'][scenario_name].get_val().set_plot_line_options(
            2, 'Borehole centerlines',
            line_intercept=None, line_alignment='vertical', line_color=soil_color, linetype='dash-dot',
            line_width=1.5, text_color=soil_color, text_alignment='bottom', multiple_labels=True)

        # Series
        # Bridge Geometry
        self.input['Plot options'][scenario_name].get_val().set_plot_series_options(
            related_index=0, index=0, x_axis='Station', y_axis='Elevation', name='Bridge cross-section')
        self.input['Plot options'][scenario_name].get_val().set_plot_series_options(
            related_index=0, index=1, x_axis='Station', y_axis='Elevation', name='Approach cross-section')
        self.input['Plot options'][scenario_name].get_val().set_plot_series_options(
            related_index=1, index=0, x_axis='Station', y_axis='Elevation', name='Bridge deck geometry')
        self.input['Plot options'][scenario_name].get_val().set_plot_series_options(
            related_index=2, index=0, x_axis='Station', y_axis='Elevation', name='Pier geometry')
        # Senarios
        self.input['Plot options'][scenario_name].get_val().set_plot_series_options(
            related_index=3, index=0, x_axis='Station', y_axis='Elevation', name='WSE')
        self.input['Plot options'][scenario_name].get_val().set_plot_series_options(
            related_index=4, index=0, x_axis='Station', y_axis='Elevation', name='Long-term degradation',
            line_color=ltd_color, linetype='solid', line_width=1.5, fill_below_line=False,
            fill_color=ltd_fill_color,
            pattern='reverse diagonal')
        self.input['Plot options'][scenario_name].get_val().set_plot_series_options(
            related_index=5, index=0, x_axis='Station', y_axis='Elevation', name='Contraction scour',
            line_color=contraction_color, linetype='solid', line_width=1.5, fill_below_line=False,
            fill_color=contraction_fill_color, pattern='diagonal')
        self.input['Plot options'][scenario_name].get_val().set_plot_series_options(
            related_index=6, index=0, x_axis='Station', y_axis='Elevation', name='Abutment scour',
            line_color=abut_color, linetype='dashed', line_width=1.5, fill_below_line=False,
            fill_color=abut_fill_color, pattern='horizontal')
        self.input['Plot options'][scenario_name].get_val().set_plot_series_options(
            related_index=7, index=0, x_axis='Station', y_axis='Elevation', name='Pier scour',
            line_color=pier_color, linetype='dashed', line_width=1.5, fill_below_line=False,
            fill_color=pier_fill_color, pattern='diamond')
        self.input['Plot options'][scenario_name].get_val().set_plot_series_options(
            related_index=8, index=0, x_axis='Station', y_axis='Elevation', name='Total scour',
            line_color=soil_color, linetype='solid', line_width=0.5, fill_below_line=True,
            fill_color=soil_fill_color, pattern='sand')
        self.input['Plot options'][scenario_name].get_val().set_plot_series_options(
            related_index=9, name='Lateral potential limits of contraction and abutment scour',
            index=0, x_axis='Station', y_axis='Elevation',
            line_color=lateral_color, linetype='solid', line_width=4.0, fill_below_line=False)

    def update_bridge_geometry(self):
        """Updates the bridge geometry."""
        # Check and move pier data between bridge geometry and scenarios
        scenario = self.input['Scenarios'].get_val().item_list[0]
        bcs_plot_dict = None
        pier_plot_dict = None
        bh_plot_dict = None
        abutment_dict = None
        _, null_data = self.get_setting('Null data')
        _, zero_tol = self.get_setting('Zero tolerance')
        if 'Bridge geometry' in self.plot_dict:
            if 'Bridge cross-section' in self.plot_dict['Bridge geometry'] and \
                    0 in self.plot_dict['Bridge geometry']['Bridge cross-section']['series']:
                bcs_plot_dict = self.plot_dict['Bridge geometry']['Bridge cross-section']['series'][0]
            if 'Piers' in self.plot_dict['Bridge geometry']:
                first_time = True
                for pier_index in self.plot_dict['Bridge geometry']['Piers']:
                    geom_dict = self.plot_dict['Bridge geometry']['Piers'][pier_index]['Pier geometry']['series'][0]
                    if first_time:
                        pier_plot_dict = copy.deepcopy(geom_dict)
                        first_time = False
                        if 'x var' in geom_dict and len(geom_dict['x var'].get_val()) <= 1:
                            pier_plot_dict['x var'].set_val([])
                            pier_plot_dict['y var'].set_val([])
                    else:
                        if 'x var' in geom_dict and len(geom_dict['x var'].get_val()) > 1:
                            pier_plot_dict['x var'].value_options.append(null_data)
                            pier_plot_dict['y var'].value_options.append(null_data)

                            pier_plot_dict['x var'].value_options.extend(geom_dict['x var'].get_val())
                            pier_plot_dict['y var'].value_options.extend(geom_dict['y var'].get_val())
                            pier_plot_dict['x var'].value += len(geom_dict['x var'].get_val()) + 1
                            pier_plot_dict['y var'].value += len(geom_dict['y var'].get_val()) + 1
            if 'Borehole centerlines' in self.plot_dict['Bridge geometry'] and \
                    0 in self.plot_dict['Bridge geometry']['Borehole centerlines']['series']:
                bh_plot_dict = self.plot_dict['Bridge geometry']['Borehole centerlines']['series'][0]
        if scenario.name in self.plot_dict and 'lines' in self.plot_dict[scenario.name]:
            for line_index in self.plot_dict[scenario.name]['lines']:
                if self.plot_dict[scenario.name]['lines'][line_index]['Label'] == 'Abutment toe':
                    abutment_dict = copy.deepcopy(self.plot_dict[scenario.name]['lines'][line_index])

        self.input['Gradations'].get_val().input['Boreholes'].value.bcs_plot_dict = copy.deepcopy(bcs_plot_dict)
        self.input['Gradations'].get_val().input['Boreholes'].value.pier_data = copy.deepcopy(pier_plot_dict)
        self.input['Gradations'].get_val().input['Boreholes'].value.abutment_data = copy.deepcopy(abutment_dict)
        self.input['Gradations'].get_val().set_bridge_cs_and_piers(bcs_plot_dict, pier_plot_dict, bh_plot_dict,
                                                                   abutment_dict)

        if bcs_plot_dict is None or 'x var' not in bcs_plot_dict or 'y var' not in bcs_plot_dict or \
                bcs_plot_dict['x var'].get_val() == [0.0, 0.0, 0.0]:
            x_data = self.input['Bridge geometry'].value.input['Bridge cross-section'].value.input[
                'Data input'].value.input['Station'].get_val()
            y_data = self.input['Bridge geometry'].value.input['Bridge cross-section'].value.input[
                'Data input'].value.input['Elevation'].get_val()

        else:
            x_data = bcs_plot_dict['x var'].get_val()
            y_data = bcs_plot_dict['y var'].get_val()

        self.update_structure_locations(x_data, y_data, null_data, zero_tol)

        # First, we check if scenario data clicked duplicate or delete
        num_piers = None
        pier_list = []
        btn_pressed = None
        btn_item_index = None

        for scenario in self.input['Scenarios'].get_val().item_list:
            if scenario.input['Pier scour'].get_val().btn_pressed:
                btn_pressed = scenario.input['Pier scour'].get_val().btn_pressed
                btn_item_index = scenario.input['Pier scour'].get_val().btn_item_index
                num_piers = scenario.input['Pier scour'].get_val().input['Number of items'].get_val()
                pier_list = scenario.input['Pier scour'].get_val().item_list
                break

        # Precheck to make sure that the names are unique
        pier_names = []
        if num_piers is not None:
            for pier in self.input['Bridge geometry'].get_val().input['Piers'].get_val().item_list:
                pier_names.append(pier.name)
            duplicate_names = self.calculator.check_unique_pier_names(pier_names)
            for dup_name in duplicate_names:
                count = 0
                for pier in self.input['Bridge geometry'].get_val().input['Piers'].get_val().item_list:
                    if pier.name == dup_name:
                        if count > 0:
                            pier.name = f'{pier.name} ({count})'
                        count += 1

        # If we changed scenario data, update bridge geometry data
        if num_piers is not None:
            self.input['Bridge geometry'].get_val().input['Piers'].get_val().input['Number of items'].set_val(
                num_piers)
            self.input['Bridge geometry'].get_val().input['Piers'].get_val().get_input_group()  # Updates the num
            list_len = min(len(self.input['Bridge geometry'].get_val().input['Piers'].get_val().item_list),
                           len(pier_list))
            for index in range(list_len):
                self.input['Bridge geometry'].get_val().input['Piers'].get_val().item_list[index].name = \
                    pier_list[index].name
                if index == list_len - 1 and btn_pressed == 'duplicate':
                    self.input['Bridge geometry'].get_val().input['Piers'].get_val().item_list[index] \
                        .input['Pier geometry'] = copy.deepcopy(
                            self.input['Bridge geometry'].get_val().input['Piers'].get_val().item_list[btn_item_index]
                            .input['Pier geometry'])

        # Finally, we update the scenario data from the bridge geometry data
        num_piers = self.input['Bridge geometry'].get_val().input['Piers'].get_val().input['Number of items'].get_val()
        self.input['Bridge geometry'].get_val().input['Piers'].get_val().get_input_group()  # Updates the num of items
        pier_list = self.input['Bridge geometry'].get_val().input['Piers'].get_val().item_list
        for scenario in self.input['Scenarios'].get_val().item_list:
            scenario.input['Pier scour'].get_val().input['Number of items'].set_val(num_piers)
            scenario.input['Pier scour'].get_val().get_input_group()  # Updates the num of pier items
            list_len = min(len(scenario.input['Pier scour'].get_val().item_list), len(pier_list))
            for index in range(list_len):
                scenario.input['Pier scour'].get_val().item_list[index].name = pier_list[index].name

    def update_structure_locations(self, x_data, y_data, null_data, zero_tol):
        """Update the structure locations based on the bridge geometry.

        Args:
            x_data (list): The x-coordinates of the bridge geometry.
            y_data (list): The y-coordinates of the bridge geometry.
            null_data (list): A list of null data points to be removed.
            zero_tol (float): The tolerance for zero values.
        """
        if x_data is None or y_data is None:
            return
        clean_x, clean_y, _ = remove_nans(x_data, y_data, null_data=null_data)
        thalweg_elevation = sys.float_info.max
        thalweg_station = None
        for index in range(len(clean_x)):
            if clean_y[index] < thalweg_elevation:
                thalweg_elevation = clean_y[index]
                thalweg_station = clean_x[index]
        left_bank = self.input['Bridge geometry'].value.input['Channel left bank station'].get_val()
        right_bank = self.input['Bridge geometry'].value.input['Channel right bank station'].get_val()
        streambed_interp = Interpolation([], [], null_data=null_data, zero_tol=zero_tol)
        left_elevation = 0.0
        right_elevation = 0.0
        if len(clean_x) >= 2 and len(clean_y) >= 2:
            streambed_interp.set_xy(clean_x, clean_y)
            left_elevation, _ = streambed_interp.interpolate_y(left_bank)
            right_elevation, _ = streambed_interp.interpolate_y(right_bank)

        # Update the 'Compute Shear Decay', centerlines, and centerline_streambed variables
        for scenario in self.input['Scenarios'].value.item_list:
            compute_shear_decay = scenario.input['Compute shear decay'].get_val()

            # Long-term degradation
            scenario.input['Long-term degradation'].value.centerline = thalweg_station
            scenario.input['Long-term degradation'].value.centerline_streambed = thalweg_elevation
            scenario.input['Long-term degradation'].value.channel_location = 'main'

            # Contraction scour
            con = scenario.input['Contraction scour']['Main channel contraction scour'].value
            con.centerline = thalweg_station
            con.centerline_streambed = thalweg_elevation
            con.channel_location = 'main channel'
            con.is_computing_shear_decay = compute_shear_decay

            con = scenario.input['Contraction scour']['Left overbank contraction scour'].value
            if con.channel_location == 'main' and con.centerline == 0.0 and con.centerline != left_bank:
                scenario.input['Contraction scour']['Left overbank contraction scour'].value.input[
                    'Selected approach location'].set_val(-1)
            con.channel_location = 'left overbank'
            con.centerline = left_bank
            con.centerline_streambed = left_elevation
            con.is_computing_shear_decay = compute_shear_decay

            con = scenario.input['Contraction scour']['Right overbank contraction scour'].value
            if con.channel_location == 'main' and con.centerline == 0.0 and con.centerline != right_bank:
                scenario.input['Contraction scour']['Right overbank contraction scour'].value.input[
                    'Selected approach location'].set_val(-1)
            con.channel_location = 'right overbank'
            con.centerline = right_bank
            con.centerline_streambed = right_elevation
            con.is_computing_shear_decay = compute_shear_decay

            # Pier scour
            for index, pier in enumerate(scenario.input['Pier scour'].value.item_list):
                # pier = scenario.input['Pier scour'][index]['calculator']
                if index < len(self.input['Bridge geometry'].value.input['Piers'].value.item_list):
                    cl = self.input['Bridge geometry'].value.input['Piers'].value.item_list[index].input[
                        'Centerline'].get_val()
                    scour_reference_point = self.input['Bridge geometry'].value.input['Piers'].value.item_list[
                        index].input['Scour reference point'].get_val()
                    if pier.centerline == 0.0 and pier.centerline != cl:
                        scenario.input['Pier scour'].value.item_list[index].input[
                            'Selected approach location'].set_val(-1)
                    pier.centerline = cl
                    if len(clean_x) >= 2 and len(clean_y) >= 2:
                        pier.centerline_streambed, _ = streambed_interp.interpolate_y(pier.centerline)
                    pier.channel_location = self.determine_channel_location(pier.centerline)
                    if scour_reference_point == 'Thalweg':
                        pier.channel_location = 'main channel'
                        if scour_reference_point != pier.scour_reference_point:
                            scenario.input['Pier scour'].value.item_list[index].input[
                                'Selected approach location'].set_val(-1)
                    pier.is_computing_shear_decay = compute_shear_decay

            # scenario.input['Abutment scour']['Left abutment scour'].value.is_computing_shear_decay = \
            #     compute_shear_decay
            # scenario.input['Abutment scour']['Right abutment scour'].value.is_computing_shear_decay = \
            #     compute_shear_decay

            # Abutment scour
            left_abut = scenario.input['Abutment scour']['Left abutment scour'].value
            left_abut_station = self.input['Bridge geometry'].value.input['Left abutment toe station'].value
            if left_abut.centerline == 0.0 and left_abut.centerline != left_abut_station:
                scenario.input['Abutment scour']['Left abutment scour'].value.input[
                    'Selected approach location'].set_val(-1)
            left_abut.centerline = left_abut_station
            if len(clean_x) >= 2 and len(clean_y) >= 2:
                left_abut.centerline_streambed, _ = streambed_interp.interpolate_y(left_abut_station)
            left_abut.channel_location = self.determine_channel_location(left_abut_station)
            left_abut.is_computing_shear_decay = compute_shear_decay

            right_abut = scenario.input['Abutment scour']['Right abutment scour'].value
            right_abut_station = self.input['Bridge geometry'].value.input['Right abutment toe station'].value
            if right_abut.centerline == 0.0 and right_abut.centerline != right_abut_station:
                scenario.input['Abutment scour']['Right abutment scour'].value.input[
                    'Selected approach location'].set_val(-1)
            right_abut.centerline = right_abut_station
            if len(clean_x) >= 2 and len(clean_y) >= 2:
                right_abut.centerline_streambed, _ = streambed_interp.interpolate_y(right_abut_station)
            right_abut.channel_location = self.determine_channel_location(right_abut_station)
            right_abut.is_computing_shear_decay = compute_shear_decay

    def get_plot_options(self):
        """Get the plot options.

        Returns:
            dict: The plot options.
        """
        input_vars = copy.deepcopy(self.input['Plot options'])

        # Remove these items, because we get those plot options from options defined with the geometry
        plot_hide_list = ['Approach cross-section data', 'Bridge cross-section data', 'Bridge deck geometry data',
                          'Pier geometry data', 'WSE data', 'Approach cross-section', 'Bridge cross-section',
                          'Bridge deck geometry', 'Pier geometry', 'WSE']
        for scenario in input_vars:
            for data_item in input_vars[scenario].get_val().input['Data series'].value.item_list:
                data_item.app_data = self.app_data
                if data_item.name in plot_hide_list:
                    data_item.complexity = 6  # Hide these items
                for series_item in data_item.input['Plot series'].get_val().item_list:
                    series_item.app_data = self.app_data
                    if series_item.name in plot_hide_list:
                        series_item.complexity = 6  # Hide these items

        return input_vars

    def get_input_group(self, unknown=None):
        """Returns a dictionary of input variables that are needed for current selections.

        Args:
            unknown (string): the variable that is unknown (and left out of the input dictionary)

        Returns:
              input_vars (dictionary of variables): the input variables
        """
        input_vars = {}

        # Get some system settings
        _, null_data = self.get_setting('Null data')
        _, complexity_var, _ = self.get_setting_var('Complexity')
        self.sys_complexity = complexity_var.get_index(null_data)
        if self.sys_complexity == null_data:
            self.sys_complexity = 0

        # Set Scenario with channel bank data
        bridge_geometry = self.input['Bridge geometry'].get_val()
        left_bank = bridge_geometry.input['Channel left bank station'].get_val()
        right_bank = bridge_geometry.input['Channel right bank station'].get_val()
        left_abut = bridge_geometry.input['Left abutment toe station'].get_val()
        right_abut = bridge_geometry.input['Right abutment toe station'].get_val()

        for index in range(len(self.input['Scenarios'].get_val().item_list)):
            self.input['Scenarios'].get_val().item_list[index].bridge_geometry = bridge_geometry
            self.input['Scenarios'].get_val().item_list[index].left_channel_bank_station = left_bank
            self.input['Scenarios'].get_val().item_list[index].right_channel_bank_station = right_bank
            self.input['Scenarios'].get_val().item_list[index].left_abutment_toe_station = left_abut
            self.input['Scenarios'].get_val().item_list[index].right_abutment_toe_station = right_abut

        # Provide Input
        input_vars['Data input'] = copy.deepcopy(self.input)

        return input_vars

    def determine_channel_location(self, centerline):
        """Determine the structure location.

        Args:
            centerline (list of float): the centerline

        Returns:
            channel_location (string): the structure location
        """
        if centerline < self.input['Bridge geometry'].get_val().input['Channel left bank station'].get_val():
            channel_location = 'left overbank'
        elif centerline > self.input['Bridge geometry'].get_val().input['Channel right bank station'].get_val():
            channel_location = 'right overbank'
        else:
            channel_location = 'main channel'
        return channel_location

    def update_gradation_and_properties(self):
        """Update the gradations and water/soil properties."""
        gradations = self.input['Gradations'].get_val()

        for scenario in self.input['Scenarios'].get_val().item_list:
            scenario.gradations = gradations

            # Update water and soil properties
            item_list = [scenario.input['Long-term degradation'].value,
                         scenario.input['Contraction scour']['Main channel contraction scour'].value,
                         scenario.input['Contraction scour']['Left overbank contraction scour'].value,
                         scenario.input['Contraction scour']['Right overbank contraction scour'].value,
                         scenario.input['Abutment scour']['Left abutment scour'].value,
                         scenario.input['Abutment scour']['Right abutment scour'].value]
            for pier in scenario.input['Pier scour'].get_val().item_list:
                item_list.append(pier)

            for item in item_list:
                item.gradations = gradations
                if 'Unit weight of water (γw)' in item.input:
                    item.input['Unit weight of water (γw)'] = \
                        self.input['Water properties'].get_val().input['Unit weight of water (γw)']
                if 'Water density' in item.input:
                    item.input['Water density'] = self.input['Water properties'].get_val().input['Water density']
                if 'Unit weight of soil (γs)' in item.input:
                    item.input['Unit weight of soil (γs)'] = gradations.input['Unit weight of soil (γs)']
                if 'Angle of repose (Θ)' in item.input:
                    item.input['Angle of repose (Θ)'] = gradations.input['Angle of repose (Θ)']
                if 'Scour hole geometry' in item.input:
                    item.input['Scour hole geometry'].value.input['Angle of repose (Θ)'] = \
                        gradations.input['Angle of repose (Θ)']

    def update_plots(self):
        """Update the plots."""
        self.plots = {}
        num_scenarios = self.input['Scenarios'].get_val().input['Number of items'].get_val()
        scenarios = self.input['Scenarios'].get_val().item_list[:num_scenarios]
        for scenario in scenarios:
            self.plots[scenario.name] = {}

    def get_results_tab_group(self, unknown=None):
        """Returns a dictionary of input variables that are needed for current selections.

        Args:
            unknown (string): the variable that is unknown (and included in the result dictionary)

        Returns:
              result_vars (dictionary of variables): the input variables
        """
        results_vars = {}

        # basic_results = {}
        # complete_results = {}
        all_scenarios = {}

        if not self.can_compute:
            return {'Results': results_vars}

        if 'Geometry data' in self.results['Results']:
            results_vars['Geometry data'] = copy.deepcopy(self.results['Results']['Geometry data'])

        count = 0
        for scenario in self.input['Scenarios'].get_val().item_list:
            results_vars[scenario.name] = copy.deepcopy(self.results['Results'][scenario.name])
            if count == 0:
                all_scenarios = copy.deepcopy(results_vars[scenario.name])
            else:
                for var in results_vars[scenario.name]:
                    if var in all_scenarios:
                        all_scenarios[var].add_result(results_vars[var].get_val())
                    else:
                        all_scenarios[var] = results_vars[scenario.name][var]
                        value = results_vars[scenario.name][var].get_val()
                        if isinstance(value, list):
                            if len(value) > 0:
                                value = value[0]
                            else:
                                value = '--'
                        all_scenarios[var].value_options = ['--' * count, value]
                for var in all_scenarios:
                    if var not in results_vars[scenario.name]:
                        all_scenarios[var].add_result('--')
            count += 1

        results_vars['All scenarios'] = all_scenarios

        return results_vars

    def get_the_approach_bridge_cross_section(self, bridge_geometry, plot_options):
        """Get the main bridge cross section.

        Args:
            bridge_geometry (BridgeGeometry): The bridge geometry.
            plot_options (PlotOptions): The plot options.

        Returns:
            True if the main bridge cross section is valid.
        """
        acs_plot_options = self.get_geometry(bridge_geometry, plot_options, 'Approach cross-section')

        if all(value == 0 for value in acs_plot_options.x_var.get_val()) and \
                all(value == 0 for value in acs_plot_options.y_var.get_val()):
            return False

        return True

    def get_the_main_bridge_cross_section(self, bridge_geometry, plot_options):
        """Get the main bridge cross section.

        Args:
            bridge_geometry (BridgeGeometry): The bridge geometry.
            plot_options (PlotOptions): The plot options.

        Returns:
            True if the main bridge cross section is valid.
        """
        bcs_plot_options = self.get_geometry(bridge_geometry, plot_options, 'Bridge cross-section')

        if all(value == 0 for value in bcs_plot_options.x_var.get_val()) and \
                all(value == 0 for value in bcs_plot_options.y_var.get_val()):
            return False

        return True

    def get_the_bridge_deck(self, bridge_geometry, plot_options):
        """Get the bridge deck.

        Args:
            bridge_geometry (BridgeGeometry): The bridge geometry.
            plot_options (PlotOptions): The plot options.

        Returns:
            True if the bridge deck is valid.
        """
        bd_plot_options = self.get_geometry(bridge_geometry, plot_options, 'Bridge deck geometry')

        if all(value == 0 for value in bd_plot_options.x_var.get_val()) and \
                all(value == 0 for value in bd_plot_options.y_var.get_val()):
            return None

        return True

    def get_geometry(self, bridge_geometry, original_plot_options, name):
        """Get the geometry.

        Args:
            bridge_geometry (BridgeGeometry): The bridge geometry.
            original_plot_options (PlotOptions): The original plot options.
            name (str): The name of the geometry.

        Returns:
            PlotOptions: The plot options.
        """
        # plot_options = original_plot_options.get_item_by_name(name)
        bridge_geometry.input[name].get_val().compute_data()
        new_plot_options = bridge_geometry.input[name].get_val().get_plot_options(name)  # Get the plot options
        plot_options = copy.deepcopy(new_plot_options.get_item_by_name(name))  # Copy the plot option series
        if plot_options:
            plot_options.x_var = bridge_geometry.input[name].get_val().results['Station']
            if 'Elevation' in bridge_geometry.input[name].get_val().results:
                plot_options.y_var = bridge_geometry.input[name].get_val().results['Elevation']
            else:
                plot_options.y_var = bridge_geometry.input[name].get_val().results['Bridge deck']
            original_plot_options.set_item_by_name(plot_options)

        return plot_options

    def get_the_pier_geometry(self, bridge_geometry, plot_options):
        """Get the pier geometry.

        Args:
            bridge_geometry (BridgeGeometry): The bridge geometry.

        Returns:
            dict: The plot data.
        """
        if bridge_geometry.input['Piers'].value.input['Number of items'].get_val() < 1:
            return None
        plot_data = None
        first_pier = True
        _, null_data = self.get_setting('Null data')

        for pier in bridge_geometry.input['Piers'].get_val().item_list:
            pier.centerline = pier.input['Centerline'].get_val()
            result, _ = pier.input['Pier geometry'].get_val().get_can_compute()
            if result:
                if plot_data is None:
                    plot_data = {}
                pier.input['Pier geometry'].get_val().compute_data()
                if first_pier:
                    plot_data['x_var'] = pier.input['Pier geometry'].get_val().results['Width']
                    plot_data['y_var'] = pier.input['Pier geometry'].get_val().results['Elevation']
                    plot_data['options'] = pier.input['Pier geometry'].get_val().get_plot_options('Pier geometry')
                    first_pier = False
                else:
                    plot_data['x_var'].value_options.append(null_data)
                    plot_data['x_var'].value_options.extend(pier.input['Pier geometry'].get_val().results['Width'].
                                                            get_val())
                    plot_data['y_var'].value_options.append(null_data)
                    plot_data['y_var'].value_options.extend(pier.input['Pier geometry'].get_val().results['Elevation'].
                                                            get_val())

        if plot_data is not None:
            name = 'Pier geometry'
            # Copy the plot option series
            pier_plot_options = copy.deepcopy(plot_data['options'].get_item_by_name(name))
            pier_plot_options.x_var = plot_data['x_var']
            pier_plot_options.y_var = plot_data['y_var']
            plot_options.set_item_by_name(pier_plot_options)

        return plot_data

    def handle_results(self, can_compute, compute_success, results_dict, warnings, plot_dict, calculator):
        """Handles the results of the computation.

        Args:
            can_compute (bool): whether the computation can be performed
            compute_success (bool): whether the computation was successful
            results_dict (dict): the results dictionary
            warnings (list): the list of warnings
            plot_dict (dict): the plot dictionary
            calculator (Calculator): the calculator instance
        """
        null_data = self.get_setting('Null data')[1]
        lowest_scour_elevation = None
        lowest_scour_type = None
        self.results['Results'] = {}
        if 'Geometry' in results_dict:
            self.results['Results']['Geometry data'] = {}
            if 'Thalweg elevation' in results_dict['Geometry']:
                self.results['Results']['Geometry data']['Cross-sectional data'] = {}
                self.results['Results']['Geometry data']['Cross-sectional data']['Thalweg elevation'] = \
                    copy.deepcopy(self.results['Geometry data']['Cross-sectional data']['Thalweg elevation'])
                self.results['Results']['Geometry data']['Cross-sectional data']['Thalweg elevation'].set_val(
                    results_dict['Geometry']['Thalweg elevation'])
                self.results['Results']['Geometry data']['Cross-sectional data']['Thalweg station'] = \
                    copy.deepcopy(self.results['Geometry data']['Cross-sectional data']['Thalweg station'])
                self.results['Results']['Geometry data']['Cross-sectional data']['Thalweg station'].set_val(
                    results_dict['Geometry']['Thalweg station'])
                self.results['Results']['Geometry data']['Cross-sectional data']['Left bank max elevation'] = \
                    copy.deepcopy(self.results['Geometry data']['Cross-sectional data']['Left bank max elevation'])
                self.results['Results']['Geometry data']['Cross-sectional data']['Left bank max elevation'].set_val(
                    results_dict['Geometry']['Left bank max elevation'])
                self.results['Results']['Geometry data']['Cross-sectional data']['Right bank max elevation'] = \
                    copy.deepcopy(self.results['Geometry data']['Cross-sectional data']['Right bank max elevation'])
                self.results['Results']['Geometry data']['Cross-sectional data']['Right bank max elevation'].set_val(
                    results_dict['Geometry']['Right bank max elevation'])
            if 'Bridge low-chord' in results_dict['Geometry']:
                self.results['Results']['Geometry data']['Bridge deck data'] = {}
                self.results['Results']['Geometry data']['Bridge deck data']['Bridge high-chord'] = \
                    copy.deepcopy(self.results['Geometry data']['Bridge deck data']['Bridge high-chord'])
                self.results['Results']['Geometry data']['Bridge deck data']['Bridge high-chord'].set_val(
                    results_dict['Geometry']['Bridge high-chord'])
                self.results['Results']['Geometry data']['Bridge deck data']['Bridge low-chord'] = \
                    copy.deepcopy(self.results['Geometry data']['Bridge deck data']['Bridge low-chord'])
                self.results['Results']['Geometry data']['Bridge deck data']['Bridge low-chord'].set_val(
                    results_dict['Geometry']['Bridge low-chord'])
        for scenario_name in results_dict:
            if 'Average WSE' in results_dict[scenario_name] and results_dict[scenario_name]['Average WSE'] != null_data:
                if 'Geometry data' not in self.results['Results']:
                    self.results['Results']['Geometry data'] = {}
                if 'WSE data' not in self.results['Results']['Geometry data']:
                    self.results['Results']['Geometry data']['WSE data'] = {}
                self.results['Results']['Geometry data']['WSE data'][scenario_name] = {}
                self.results['Results']['Geometry data']['WSE data'][scenario_name]['Average WSE'] = \
                    copy.deepcopy(self.results['Scenario data']['WSE'])
                self.results['Results']['Geometry data']['WSE data'][scenario_name]['Average WSE'].set_val(
                    results_dict[scenario_name]['Average WSE'])
            self.results['Results'][scenario_name] = {}

            if 'Long-term degradation' in results_dict[scenario_name]:
                self.results['Results'][scenario_name]['Long-term degradation'] = \
                    copy.deepcopy(self.results['Scenario data']['Long-term degradation'])
                self.results['Results'][scenario_name]['Long-term degradation']['Long-term degradation'].set_val(
                    results_dict[scenario_name]['Long-term degradation']['Long-term degradation'])
                self.results['Results'][scenario_name]['Long-term degradation']['Streambed thalweg elevation'].set_val(
                    results_dict[scenario_name]['Long-term degradation']['Streambed thalweg elevation'])
                self.results['Results'][scenario_name]['Long-term degradation'][
                    'Minimum channel elevation with LTD'].set_val(
                    results_dict[scenario_name]['Long-term degradation']['Minimum channel elevation with LTD'])
                lowest_scour_elevation = results_dict[scenario_name]['Long-term degradation'][
                    'Minimum channel elevation with LTD']
                lowest_scour_type = 'Long-term degradation'

            if 'Contraction scour' in results_dict[scenario_name]:
                self.results['Results'][scenario_name]['Contraction scour'] = {}
                con_types = ['Main channel',
                             'Left overbank',
                             'Right overbank']
                for con_type in con_types:
                    con_str = f'{con_type} contraction scour'
                    if con_str in results_dict[scenario_name]['Contraction scour']:
                        self.results['Results'][scenario_name]['Contraction scour'][con_str] = \
                            copy.deepcopy(self.results['Scenario data']['Contraction scour'])

                        self.results['Results'][scenario_name]['Contraction scour'][con_str][
                            'Contraction scour method'].set_val(
                            results_dict[scenario_name]['Contraction scour'][con_str]['Contraction scour method'])
                        self.results['Results'][scenario_name]['Contraction scour'][con_str][
                            'Applied contraction scour depth'].set_val(
                            results_dict[scenario_name]['Contraction scour'][con_str][
                                'Applied contraction scour depth'])
                        self.results['Results'][scenario_name]['Contraction scour'][con_str][
                            'Clear-water scour depth'].set_val(
                            results_dict[scenario_name]['Contraction scour'][con_str]['Clear-water scour depth'])
                        self.results['Results'][scenario_name]['Contraction scour'][con_str][
                            'Live-bed scour depth'].set_val(
                            results_dict[scenario_name]['Contraction scour'][con_str]['Live-bed scour depth'])
                        if 'Pressure scour depth' in results_dict[scenario_name]['Contraction scour'][con_str]:
                            self.results['Results'][scenario_name]['Contraction scour'][con_str][
                                'Pressure scour depth'].set_val(
                                results_dict[scenario_name]['Contraction scour'][con_str]['Pressure scour depth'])
                        else:
                            self.results['Results'][scenario_name]['Contraction scour'][con_str].pop(
                                'Pressure scour depth')
                        if 'Cohesive scour depth' in results_dict[scenario_name]['Contraction scour'][con_str]:
                            self.results['Results'][scenario_name]['Contraction scour'][con_str][
                                'Cohesive scour depth'].set_val(
                                results_dict[scenario_name]['Contraction scour'][con_str]['Cohesive scour depth'])
                        else:
                            self.results['Results'][scenario_name]['Contraction scour'][con_str].pop(
                                'Cohesive scour depth')
                        if 'Streambed thalweg elevation' in results_dict[scenario_name]['Contraction scour'][con_str]:
                            self.results['Results'][scenario_name]['Contraction scour'][con_str][
                                'Streambed thalweg elevation'].set_val(
                                results_dict[scenario_name]['Contraction scour'][con_str][
                                    'Streambed thalweg elevation'])
                        else:
                            self.results['Results'][scenario_name]['Contraction scour'][con_str].pop(
                                'Streambed thalweg elevation')
                        if 'Minimum channel elevation with contraction scour and LTD' in results_dict[scenario_name][
                                'Contraction scour'][con_str]:
                            self.results['Results'][scenario_name]['Contraction scour'][con_str][
                                'Minimum channel elevation with contraction scour and LTD'].set_val(
                                results_dict[scenario_name]['Contraction scour'][con_str][
                                    'Minimum channel elevation with contraction scour and LTD'])
                            if lowest_scour_elevation is None or \
                                    results_dict[scenario_name]['Contraction scour'][con_str][
                                        'Minimum channel elevation with contraction scour and LTD'] < \
                                    lowest_scour_elevation:
                                lowest_scour_elevation = results_dict[scenario_name]['Contraction scour'][con_str][
                                    'Minimum channel elevation with contraction scour and LTD']
                                lowest_scour_type = 'Contraction scour'
                        else:
                            self.results['Results'][scenario_name]['Contraction scour'][con_str].pop(
                                'Minimum channel elevation with contraction scour and LTD')

            if 'Pier scour' in results_dict[scenario_name]:
                self.results['Results'][scenario_name]['Pier scour'] = {}
                for pier in results_dict[scenario_name]['Pier scour']:
                    self.results['Results'][scenario_name]['Pier scour'][pier] = \
                        copy.deepcopy(self.results['Scenario data']['Pier scour'])
                    self.results['Results'][scenario_name]['Pier scour'][pier]['Pier computation method'].set_val(
                        results_dict[scenario_name]['Pier scour'][pier]['Pier computation method'])
                    self.results['Results'][scenario_name]['Pier scour'][pier]['Pier scour depth'].set_val(
                        results_dict[scenario_name]['Pier scour'][pier]['Pier scour depth'])
                    if 'Max flow depth including pier scour' in results_dict[scenario_name]['Pier scour'][pier]:
                        self.results['Results'][scenario_name]['Pier scour'][pier][
                            'Max flow depth including pier scour'].set_val(
                            results_dict[scenario_name]['Pier scour'][pier]['Max flow depth including pier scour'])
                    self.results['Results'][scenario_name]['Pier scour'][pier]['Total scour at pier'].set_val(
                        results_dict[scenario_name]['Pier scour'][pier]['Total scour at pier'])
                    self.results['Results'][scenario_name]['Pier scour'][pier][
                        'Scour reference point for pier'].set_val(
                        results_dict[scenario_name]['Pier scour'][pier]['Scour reference point for pier'])
                    if 'Streambed thalweg elevation' in results_dict[scenario_name]['Pier scour'][pier]:
                        self.results['Results'][scenario_name]['Pier scour'][pier][
                            'Streambed thalweg elevation'].set_val(
                            results_dict[scenario_name]['Pier scour'][pier]['Streambed thalweg elevation'])
                    self.results['Results'][scenario_name]['Pier scour'][pier]['Streambed elevation at pier'].set_val(
                        results_dict[scenario_name]['Pier scour'][pier]['Streambed elevation at pier'])
                    self.results['Results'][scenario_name]['Pier scour'][pier]['Total scour elevation at pier'].set_val(
                        results_dict[scenario_name]['Pier scour'][pier]['Total scour elevation at pier'])
                    if lowest_scour_elevation is None or \
                            results_dict[scenario_name]['Pier scour'][pier]['Total scour elevation at pier'] < \
                            lowest_scour_elevation:
                        lowest_scour_elevation = results_dict[scenario_name]['Pier scour'][pier][
                            'Total scour elevation at pier']
                        lowest_scour_type = f'Pier scour: {pier}'
                    elif lowest_scour_elevation is not None or \
                            results_dict[scenario_name]['Pier scour'][pier]['Total scour elevation at pier'] == \
                            lowest_scour_elevation and \
                            lowest_scour_type[0:12] != 'Pier scour: ':
                        lowest_scour_type += f', {pier}'

            if 'Abutment scour' in results_dict[scenario_name]:
                self.results['Results'][scenario_name]['Abutment scour'] = {}
                abut_types = ['Left abutment', 'Right abutment']
                for abut_type in abut_types:
                    abut_str = f'{abut_type} scour'
                    if abut_type in results_dict[scenario_name]['Abutment scour']:
                        self.results['Results'][scenario_name]['Abutment scour'][abut_str] = \
                            copy.deepcopy(self.results['Scenario data']['Abutment scour'])
                        self.results['Results'][scenario_name]['Abutment scour'][abut_str][
                            'Abutment scour depth'].set_val(
                            results_dict[scenario_name]['Abutment scour'][abut_type]['Abutment scour depth'])
                        self.results['Results'][scenario_name]['Abutment scour'][abut_str][
                            'Max flow depth including abutment scour'].set_val(
                            results_dict[scenario_name]['Abutment scour'][abut_type][
                                'Max flow depth including abutment scour'])
                        self.results['Results'][scenario_name]['Abutment scour'][abut_str][
                            'Total scour at abutment'].set_val(
                            results_dict[scenario_name]['Abutment scour'][abut_type]['Total scour at abutment'])
                        self.results['Results'][scenario_name]['Abutment scour'][abut_str][
                            'Streambed thalweg elevation'].set_val(
                            results_dict[scenario_name]['Abutment scour'][abut_type]['Streambed thalweg elevation'])
                        self.results['Results'][scenario_name]['Abutment scour'][abut_str][
                            'Streambed elevation at abutment'].set_val(
                            results_dict[scenario_name]['Abutment scour'][abut_type]['Streambed elevation at abutment'])
                        self.results['Results'][scenario_name]['Abutment scour'][abut_str][
                            'Total scour elevation at abutment'].set_val(
                            results_dict[scenario_name]['Abutment scour'][abut_type][
                                'Total scour elevation at abutment'])
                        if lowest_scour_elevation is None or \
                                results_dict[scenario_name]['Abutment scour'][abut_type][
                                    'Total scour elevation at abutment'] < lowest_scour_elevation:
                            lowest_scour_elevation = results_dict[scenario_name]['Abutment scour'][abut_type][
                                'Total scour elevation at abutment']
                            lowest_scour_type = f'Abutment scour: {abut_type}'
            if lowest_scour_elevation is not None:
                self.results['Results'][scenario_name]['Total scour'] = {}
                self.results['Results'][scenario_name]['Total scour']['Minimum scour depth'] = \
                    copy.deepcopy(self.results['Scenario data']['Total scour']['Minimum scour depth'])
                self.results['Results'][scenario_name]['Total scour']['Minimum scour depth'].set_val(
                    lowest_scour_elevation)
                self.results['Results'][scenario_name]['Total scour']['Minimum scour type'] = \
                    copy.deepcopy(self.results['Scenario data']['Total scour']['Minimum scour type'])
                self.results['Results'][scenario_name]['Total scour']['Minimum scour type'].set_val(
                    lowest_scour_type)
