"""Bridge Deck Table Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.table_calc import TableCalc
from xms.FhwaVariable.core_data.calculator.table_data import TableData
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.culvert.inlet_control.shape_db import ShapeDatabase


class SelectSizeCalc(TableCalc):
    """A class that defines the bridge deck table calculator."""
    def __init__(self):
        """Initializes the calculator."""
        super().__init__()
        self.name = 'Culvert cross-section'
        self.plot_names = ['Culvert cross-section']

    def _compute_data(self):
        """Compute the data.

        Returns:
            True, if successful; otherwise, False
        """
        super()._compute_data()

        # self.shape_db.update_lists(geometry=self.geometry, cur_shape=self.cur_shape, cur_mat=self.cur_mat)
        self.shape_db.update_lists(inlet_control=self.inlet_control)

        self.station_var.set_val(self.inlet_control.cs_x)
        self.elevation_var.set_val(self.inlet_control.cs_y)

        self.plot_dict['Culvert cross-section']['series'][0]['x var'].set_val(self.inlet_control.cs_x)
        self.plot_dict['Culvert cross-section']['series'][0]['y var'].set_val(self.inlet_control.cs_y)

        self.plot_dict['Culvert cross-section']['series'][0]['x var'].selected_us_unit = \
            self.station_var.selected_us_unit
        self.plot_dict['Culvert cross-section']['series'][0]['x var'].selected_si_unit = \
            self.station_var.selected_si_unit
        self.plot_dict['Culvert cross-section']['series'][0]['y var'].selected_us_unit = \
            self.elevation_var.selected_us_unit
        self.plot_dict['Culvert cross-section']['series'][0]['y var'].selected_si_unit = \
            self.elevation_var.selected_si_unit

        return True


class SelectSizeTable(TableData):
    """Override the table class, so we can draw plots based on widths."""
    def __init__(self, theme, name, plot_names, name_of_items='points', stand_alone_calc=True, input=None, min_items=1,
                 max_items=sys.maxsize, show_increment=False, app_data=None, model_name=None, project_uuid=None,
                 cur_shape=None, cur_mat=None, geometry=None):
        """Initializes the calculator.

        Args:
            theme (dict): the theme
            name (str): the name of the calculator
            plot_names (list of str): list of plot names
            name_of_items (str): name of the items
            stand_alone_calc (bool): whether the calculator is a stand-alone calculator
            input (dict): dictionary of input variables
            min_items (int): minimum number of items
            max_items (int): maximum number of items
            show_increment (bool): whether to show the increment
            app_data (AppData): the application data
            model_name (str): the name of the model
            project_uuid (str): the project UUID
            cur_shape (str): the current culvert shape
            cur_mat (str): the current culvert material
            geometry (Variable): the geometry variable
        """
        super().__init__(theme=theme, name=name, plot_names=plot_names, name_of_items=name_of_items,
                         stand_alone_calc=stand_alone_calc, input=input, min_items=min_items, max_items=max_items,
                         show_increment=show_increment, app_data=app_data, model_name=model_name,
                         project_uuid=project_uuid)

        # Set to False, if this is to display the cross-section only, but define the Manning's n values by station
        self.calculator = SelectSizeCalc()

        self.stack_input_tables = False  # Stack the input tables, False will put them side by side

        self.inlet_control = None
        self.cur_shape = cur_shape
        self.cur_mat = cur_mat
        self.geometry = geometry
        self.station_var = copy.copy(input['Station'])
        self.elevation_var = copy.copy(input['Elevation'])

        self.prev_sel_cat = -1
        self.prev_sel_sub_cat = -1
        self.prev_sel_item = -1

        # self.compute_prep_functions.extend([self.update_lists])
        self.compute_prep_functions.extend([self.update_indices])
        self.compute_finalize_functions.extend([])
        self.intermediate_to_copy.extend([
            'cur_shape', 'cur_mat', 'geometry', 'station_var', 'elevation_var',
            'shape_db', 'inlet_control'
        ])

        self.input['Table options'].value.input['Category'] = Variable('Category', "list", 0, [])
        self.input['Table options'].value.input['Sub Category'] = Variable('Sub Category', "list", 0, [])
        self.input['Table options'].value.input['Selected row'] = Variable('Selected row', "list", 0, [])

        self.input['Data input'].value.input['Geometry'] = copy.copy(self.geometry)
        self.input['Data input'].value.input['Geometry'].value.input['Area'] = Variable(
            'Flow area', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft^2',
            us_units=[['yd^2', 'ft^2', 'in^2']], si_units=[['m^2', 'mm^2']])
        self.input['Data input'].value.input['Geometry'].value.input["Manning's n"] = Variable(
            "Manning's n", 'float', 0.0, [], limits=[0.0, 1.0], precision=4, unit_type=['coefficient'],
            native_unit='coefficient', us_units=[['coefficient']], si_units=[['coefficient']])
        self.input['Data input'].value.input['Geometry'].value.input['Notes'] = Variable(
            'Notes', 'string', '', [], )

        plot_name = 'Culvert cross-section'
        self.input['Plot options'][plot_name].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['X axis'].set_val('Station')
        self.input['Plot options'][plot_name].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['Y axis'].set_val('Elevation')

        self.shape_db = ShapeDatabase(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        # Cross-section data
        culvert_color = (77, 77, 77)
        culvert_fill_color = (191, 191, 191)
        if self.theme is not None:
            culvert_color = self.theme['Plot structure color']
            culvert_fill_color = self.theme['Plot structure fill color']

        # Set cross-section plot options
        self.input['Plot options'][plot_name].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['Line color'].set_val(culvert_color)
        self.input['Plot options'][plot_name].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['Fill color'].set_val(culvert_fill_color)
        self.input['Plot options'][plot_name].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['Fill pattern'].set_val('solid')
        self.input['Plot options'][plot_name].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['Line type'].set_val('solid')
        self.input['Plot options'][plot_name].value.input['Data series'].value.item_list[0].input[
            'Plot series'].value.item_list[0].input['Line width'].set_val(2.0)

    def update_indices(self):
        """Update the selected indices based on the input table selections."""
        sel_cat = self.input['Table options'].value.input['Category'].value
        sel_sub_cat = self.input['Table options'].value.input['Sub Category'].value
        sel_item = self.input['Table options'].value.input['Selected row'].value

        self.shape_db.sel_cat = sel_cat
        self.shape_db.sel_sub_cat = sel_sub_cat
        self.shape_db.sel_item = sel_item

    def set_shape_mat(self, inlet_control):
        """Set the current shape, material, and geometry.

        Args:
            inlet_control (InletControlData): the inlet control data (we will update it)
        """
        self.inlet_control = inlet_control

        self.compute_data()

        cat_name = self.shape_db.cat_name
        sub_cat_name = self.shape_db.sub_cat_name

        self.input['Table options'].value.input['Category'].name = cat_name
        self.input['Table options'].value.input['Category'].value_options = self.shape_db.cats
        self.input['Table options'].value.input['Category'].set_val(self.shape_db.sel_cat)
        self.input['Table options'].value.input['Sub Category'].name = sub_cat_name
        self.input['Table options'].value.input['Sub Category'].value_options = self.shape_db.sub_cats
        self.input['Table options'].value.input['Sub Category'].set_val(self.shape_db.sel_sub_cat)

    def get_input_tab_group(self, unknown=None):
        """Returns the input group for the user interface.

        Args:
            unknown (string): variable that is unknown

        Returns:
            input_vars (list of variables): input group for the user interface's input table
        """
        if self.input['Table options'].value.input['Table direction'].get_val() == 'vertical':
            item_name = 'Selected row'
        else:
            item_name = 'Selected column'
        self.input['Table options'].value.input['Selected row'].name = item_name

        input_vars = super().get_input_tab_group(unknown=unknown)
        input_vars['Table options'].pop('Number of items')  # Number of items is set by the database
        if 'Category' in input_vars['Table options'] and self.input['Table options'].value.input[
                'Category'].value_options == []:
            input_vars['Table options'].pop('Category')
            if 'Sub Category' in input_vars['Table options']:
                input_vars['Table options'].pop('Sub Category')
        if 'Sub Category' in input_vars['Table options']:
            if self.input['Table options'].value.input['Sub Category'].value_options == [] or \
                    self.input['Table options'].value.input['Sub Category'].value_options == ['Sub Category 1']:
                input_vars['Table options'].pop('Sub Category')

        return input_vars

    def get_input_dual_tab_group(self, unknown=None):
        """Get the input dual tab group.

        Args:
            unknown (str): the unknown variable

        Returns:
            dict: the input tab group
        """
        input_dual = {}
        input_dual['Available Culvert Sizes'] = {}

        param_vars = {}

        if 'Parameter Names' in self.shape_db.parameters:
            for name in self.shape_db.parameters['Parameter Names']:
                chan_name = name
                if name in self.shape_db.culv_params_to_chan_params:
                    chan_name = self.shape_db.culv_params_to_chan_params[name]
                if chan_name in self.geometry.value.input and 'Data' in self.shape_db.parameters[name]:
                    param_vars[chan_name] = copy.copy(self.geometry.value.input[chan_name])
                    param_vars[chan_name].type = 'float_list'

                    orig_us_units = param_vars[chan_name].selected_us_unit
                    orig_si_units = param_vars[chan_name].selected_si_unit
                    us_units = self.shape_db.parameters[name]['English Units'][0]
                    if us_units in param_vars[chan_name].available_us_units[0]:
                        param_vars[chan_name].selected_us_unit = us_units
                    si_units = self.shape_db.parameters[name]['Metric Units'][0]
                    if si_units in param_vars[chan_name].available_si_units[0]:
                        param_vars[chan_name].selected_si_unit = si_units
                    param_vars[chan_name].set_val(self.shape_db.parameters[name]['Data'], app_data=self.app_data)
                    param_vars[chan_name].read_only = True

                    if self.prev_sel_cat == self.shape_db.sel_cat and \
                            self.prev_sel_sub_cat == self.shape_db.sel_sub_cat:
                        param_vars[chan_name].selected_us_unit = orig_us_units
                        param_vars[chan_name].selected_si_unit = orig_si_units

                    if chan_name == 'Rise':
                        self.elevation_var.selected_us_unit = param_vars[chan_name].selected_us_unit
                        self.elevation_var.selected_si_unit = param_vars[chan_name].selected_si_unit

                        self.plot_dict['Culvert cross-section']['series'][0]['y var'].selected_us_unit = \
                            self.elevation_var.selected_us_unit
                        self.plot_dict['Culvert cross-section']['series'][0]['y var'].selected_si_unit = \
                            self.elevation_var.selected_si_unit

                    elif chan_name == 'Span':
                        self.station_var.selected_us_unit = param_vars[chan_name].selected_us_unit
                        self.station_var.selected_si_unit = param_vars[chan_name].selected_si_unit

                        self.plot_dict['Culvert cross-section']['series'][0]['x var'].selected_us_unit = \
                            self.station_var.selected_us_unit
                        self.plot_dict['Culvert cross-section']['series'][0]['x var'].selected_si_unit = \
                            self.station_var.selected_si_unit

        input_dual['Available Culvert Sizes'].update(param_vars)

        self.prev_sel_cat = self.shape_db.sel_cat
        self.prev_sel_sub_cat = self.shape_db.sel_sub_cat
        self.prev_sel_item = self.shape_db.sel_item

        self.input['Data input'].value.input['Geometry'].value = self.geometry.value

        return input_dual

    def get_tables_with_vertical_direction(self):
        """Returns the tables with vertical direction.

        Returns:
            list of str: tables with vertical direction
        """
        if self.input['Table options'].get_val().input['Table direction'].get_val() == 'vertical':
            return ['Available Culvert Sizes']
        return []
