"""BcComponentDisplay class."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules
import gc
import os
import shutil

# 2. Third party modules
import numpy as np
import orjson
from PySide2.QtGui import QIcon

# 3. Aquaveo modules
from xms.api.tree import tree_util as tr_util
from xms.components.display.display_options_io import (
    read_display_option_ids, read_display_options_from_json, write_display_option_ids,
    write_display_option_line_locations, write_display_options_to_json
)
from xms.components.display.xms_display_message import DrawType, XmsDisplayMessage
from xms.constraint import read_grid_from_file
from xms.core.filesystem import filesystem as io_util
from xms.guipy.data.category_display_option import CategoryDisplayOption
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.data.line_style import LineOptions
from xms.guipy.data.target_type import TargetType
from xms.guipy.dialogs.category_display_options_list import CategoryDisplayOptionsDialog
from xms.guipy.dialogs.dataset_selector import DatasetSelector
from xms.guipy.dialogs.xms_parent_dlg import get_xms_icon
from xms.guipy.settings import SettingsManager

# 4. Local modules
from xms.adcirc.data import bc_data as bcd
from xms.adcirc.data.adcirc_data import UNINITIALIZED_COMP_ID
from xms.adcirc.data.bc_data import default_levee_flags
from xms.adcirc.data.xms_data import XmsData
from xms.adcirc.gui import assign_bc_dlg, assign_pipe_dlg, forcing_options_dlg, mapped_bc_dlg

DEFAULT_BC_JSON = 'default_bc_display_options.json'
DEFAULT_POINT_BC_JSON = 'default_point_bc_display_options.json'
BC_JSON = 'bc_display_options.json'
BC_POINT_JSON = 'bc_point_display_options.json'
BC_POINT_ID_FILE = 'bc_pipe_point.display_ids'

REG_KEY_BC_ARC = 'bc_coverage_arc_default_options'
REG_KEY_BC_POINT = 'bc_coverage_point_default_options'
REG_KEY_MAPPED_BC_ARC = 'mapped_bc_arc_default_options'
REG_KEY_MAPPED_BC_POINT = 'mapped_bc_point_default_options'  # Pipe points are actually arcs in mapped BC component


class BcComponentDisplay:
    """Helper class for BC component display options."""
    def __init__(self, bc_comp, query=None):
        """Create a helper class.

        Args:
            bc_comp (:obj:`BcComponent`): The component this helper should help
            query (:obj:`Query`): The XMS inter-process communicator
        """
        self.bc_comp = bc_comp
        self.selected_att_ids = []
        self.selected_comp_ids = []
        self.dlg_message = ''
        self._xd = XmsData(query)  # Only needed when populating levee heights
        self.ugrid = None
        self.xmgrid = None

    def assign_bc(self, params, parent):
        """Display the Assign BC dialog and persist data if accepted.

        Args:
            params (:obj:`dict`): The ActionRequest parameter map
            parent (:obj:`QWidget`): The parent window

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        old_comp_id = self._unpack_xms_data(params[0], True)
        if not self.selected_att_ids:
            return [('INFO', 'No arcs selected. Select one or more arcs to boundary conditions.')], []

        increment = True
        num_selected = len(self.selected_att_ids)
        comp_id = old_comp_id
        if comp_id == UNINITIALIZED_COMP_ID:  # Create some default atts if this is the initial assignment.
            comp_id = self.bc_comp.data.add_bc_atts()
            increment = False
        bc_data = self.bc_comp.data.arcs.where(self.bc_comp.data.arcs.comp_id == comp_id, drop=True)
        old_bc_type = int(bc_data['type'].data[0].item())

        # Find the att id of the other arc in the pair if single select of a levee.
        single_select_levee = num_selected == 1 and old_bc_type == bcd.LEVEE_INDEX
        selected_att_ids = self.selected_att_ids
        if single_select_levee:
            self._xd.load_component_ids(self.bc_comp, arcs=True)
            levee_pair_att_ids = self.bc_comp.get_xms_ids(TargetType.arc, comp_id)
            if levee_pair_att_ids != UNINITIALIZED_COMP_ID:
                selected_att_ids = list(set(levee_pair_att_ids))

        # Increment the component id anytime dialog triggered except from single-select levee pair
        if increment and not single_select_levee:
            comp_id = self.bc_comp.data.add_bc_atts(bc_data)

        # Make a copy of the BC's attributes, so we can tell if they change.
        original_ds = bc_data.copy(deep=True)
        # Save the attributes so we can tell
        dlg = assign_bc_dlg.AdcircBcDlg(
            parent, bc_data, self.bc_comp.data.levees, self.bc_comp.data.q, self.dlg_message, selected_att_ids,
            old_comp_id, self._xd, self.bc_comp, self
        )
        if dlg.exec():
            bc_type = int(dlg.data['type'].data.item())

            # Check if we changed any attributes. Need this for levee pairs.
            have_flags = self.bc_comp.data.levee_flags.sizes['comp_id'] > 0
            is_levee_pair = bc_type == bcd.LEVEE_INDEX
            if have_flags and not original_ds.identical(dlg.data) and is_levee_pair:
                # If we are updating the BC, then the attributes have changed.
                if comp_id in self.bc_comp.data.levee_flags.coords['comp_id']:
                    self.bc_comp.data.levee_flags['use_second_side'].loc[dict(comp_id=[comp_id])] = 0
                else:
                    # We may have incremented the comp id of the levee pair. Add a row with the second side disabled.
                    new_flags = default_levee_flags(fill_num=1, coords={'comp_id': [comp_id]})
                    self.bc_comp.data.add_levee_flags(new_flags)

            # Check for special case of one levee arc selected
            if num_selected == 1:
                if bc_type == bcd.LEVEE_INDEX:
                    # Only one arc selected and its type is still Levee. Store the updated attributes without
                    # incrementing the component id to preserve set relationship.
                    self.bc_comp.data.update_bc(comp_id, dlg.data)
                    self.bc_comp.data.levees = dlg.levee_data
                    self.bc_comp.data.commit()
                    shutil.rmtree(os.path.join(os.path.dirname(self.bc_comp.main_file), 'temp'), ignore_errors=True)
                    return [], []
                elif old_bc_type == bcd.LEVEE_INDEX:  # The user broke a levee set and only one arc was selected.
                    # Give the selected arc a new component id. The other levee arc will get moved to the unassigned
                    # category (via the old comp id).
                    comp_id = self.bc_comp.data.add_bc_atts(dlg.data)

            # Update the attribute datasets
            self.bc_comp.data.update_bc(comp_id, dlg.data)
            self.bc_comp.data.levees = dlg.levee_data

            # Associate all selected arcs with the new component id.
            for arc_id in self.selected_att_ids:
                self.bc_comp.update_component_id(TargetType.arc, arc_id, comp_id)

            # Write the updated comp id files for the necessary BC types.
            self._update_id_files(bc_type, comp_id)
            self.bc_comp.data.commit()  # Flush attribute dataset to disk
        else:
            # Hack to fix issue 0015519.
            gc.collect()

        # Delete the id dumped by xms files.
        shutil.rmtree(os.path.join(os.path.dirname(self.bc_comp.main_file), 'temp'), ignore_errors=True)
        return [], []

    def assign_pipe(self, params, parent):
        """Display the Assign Pipe dialog and persist data if accepted.

        Args:
            params (:obj:`dict`): The ActionRequest parameter map
            parent (:obj:`QWidget`): The parent window

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        old_comp_id = self._unpack_xms_data(params[0], False)
        if not self.selected_att_ids:
            return [('INFO', 'No points selected. Select one or more pipe points.')], []

        num_selected = len(self.selected_att_ids)
        increment = True
        comp_id = old_comp_id
        if comp_id == UNINITIALIZED_COMP_ID:  # Create some default atts if this is the initial assignment.
            comp_id = self.bc_comp.data.add_pipe_atts()
            increment = False
        pipe_data = self.bc_comp.data.pipes.where(self.bc_comp.data.pipes.comp_id == comp_id, drop=True)
        # Increment the component id anytime dialog triggered.
        if increment:
            comp_id = self.bc_comp.data.add_pipe_atts(pipe_data)

        dlg = assign_pipe_dlg.AdcircPipeDlg(parent, pipe_data, self.dlg_message, num_selected)
        if dlg.exec():
            # Update the attribute dataset
            self.bc_comp.data.update_pipe(comp_id, dlg.data)

            # Associate all selected points with the new component id.
            for point_id in self.selected_att_ids:
                self.bc_comp.update_component_id(TargetType.point, point_id, comp_id)

            # Write the updated comp id files for the necessary BC types.
            self._update_point_id_files(comp_id)
            self.bc_comp.data.commit()
        else:
            # Hack to fix issue 0015519.
            gc.collect()

        # Delete the id dumped by xms files.
        shutil.rmtree(os.path.join(os.path.dirname(self.bc_comp.main_file), 'temp'), ignore_errors=True)
        return [], []

    def display_options(self, parent, xms_draw_type, cov_uuid):
        """Shows the display options dialog for a display list.

        Args:
            parent (:obj:`QWidget`): The parent window
            xms_draw_type (:obj:`DrawType`): Locations or ids
            cov_uuid (:obj:`str`): UUID of the coverage geometry associated with the display list, if any.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        # Get the arc/line display options.
        arc_categories = CategoryDisplayOptionList()
        json_dict = read_display_options_from_json(self.bc_comp.disp_opts_files[0])
        arc_categories.from_dict(json_dict)
        categories_list = [arc_categories]

        # Get the pipe point display options if a BC coverage.
        bc_coverage = xms_draw_type == DrawType.draw_at_ids
        if bc_coverage:
            point_categories = CategoryDisplayOptionList()
            json_dict = read_display_options_from_json(self.bc_comp.disp_opts_files[1])
            point_categories.from_dict(json_dict)
            categories_list.append(point_categories)

        wkt = ''
        if 'wkt' in arc_categories.projection:
            wkt = arc_categories.projection['wkt']

        if bc_coverage:
            reg_keys = [REG_KEY_BC_ARC, REG_KEY_BC_POINT]
        else:
            reg_keys = [REG_KEY_MAPPED_BC_ARC]
        dlg = CategoryDisplayOptionsDialog(categories_list, parent, package_name='xmsadcirc', registry_keys=reg_keys)

        dlg.setWindowIcon(QIcon(get_xms_icon()))
        dlg.setModal(True)
        if dlg.exec():
            # write files
            category_lists = dlg.get_category_lists()
            for idx, category_list in enumerate(category_lists):
                category_list.projection['wkt'] = wkt
                write_display_options_to_json(self.bc_comp.disp_opts_files[idx], category_list)
                if cov_uuid:
                    self.bc_comp.display_option_list.append(
                        XmsDisplayMessage(
                            file=self.bc_comp.disp_opts_files[idx], edit_uuid=cov_uuid, draw_type=xms_draw_type
                        )
                    )
                else:
                    self.bc_comp.display_option_list.append(
                        XmsDisplayMessage(file=self.bc_comp.disp_opts_files[idx], draw_type=xms_draw_type)
                    )
        if bc_coverage:
            self._update_mapped_registry_defaults()
        else:
            self._update_coverage_registry_defaults()
        return [], []

    def forcing_options(self, parent, query):
        """Shows the display options dialog for a display list.

        Args:
            parent (:obj:`QWidget`): The parent window
            query (:obj:`xmsdmi.dmi.Query`): Query for communicating with XMS.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        pe_tree = query.project_tree
        geom_children = []
        if pe_tree:
            for child in pe_tree.children:  # Only allow datasets on 2D mesh, Quadtree, and UGrid.
                if child.name == 'Mesh Data':
                    geom_children.append(child)
                elif child.name == 'UGrid Data':
                    geom_children.append(child)
                elif child.name == 'Quadtree Data':
                    geom_children.append(child)
            pe_tree.children = geom_children
        tr_util.filter_project_explorer(pe_tree, DatasetSelector.is_scalar_if_dset)

        dlg = forcing_options_dlg.ForcingOptionsDlg(self.bc_comp.data, parent, pe_tree)
        dlg.setModal(True)
        if dlg.exec():
            self.bc_comp.data.commit()
        return [], []

    def view_mapped_bc(self, query, parent):
        """Shows the display options dialog for a display list.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            parent (:obj:`QWidget`): The parent window

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        comp_item = tr_util.find_tree_node_by_uuid(query.project_tree, self.bc_comp.uuid)
        mesh_item = tr_util.descendants_of_type(
            comp_item.parent, xms_types=['TI_MESH2D_PTR'], allow_pointers=True, only_first=True, recurse=False
        )
        if mesh_item:
            # Get the geometry dump
            do_ugrid = query.item_with_uuid(mesh_item.uuid)
            if do_ugrid:
                self.xmgrid = read_grid_from_file(do_ugrid.cogrid_file)
                if self.xmgrid:
                    self.ugrid = self.xmgrid.ugrid

        xd = XmsData(query=query)
        dlg = mapped_bc_dlg.MappedBcDlg(parent, self.bc_comp.data.source_data, self.bc_comp.data, self.ugrid, xd)
        dlg.setModal(True)
        if dlg.exec():
            self.bc_comp.data.source_data.commit()
            self.bc_comp.data.commit()

            # Rewrite the the pipe line files.
            pipe_loc_file = os.path.join(os.path.dirname(self.bc_comp.main_file), BC_POINT_ID_FILE)
            pipe_levees = self.bc_comp.data.levees.where(self.bc_comp.data.levees.Pipe == 1, drop=True)
            if pipe_levees.sizes['comp_id'] > 0:
                if self.ugrid:
                    node1_ids = pipe_levees['Node1 Id'].data.astype('i4').tolist()
                    node2_ids = pipe_levees['Node2 Id'].data.astype('i4').tolist()
                    pipe_lines = [
                        self._build_pipe_line(self.xmgrid, node1, node2) for node1, node2 in zip(node1_ids, node2_ids)
                    ]
                    write_display_option_line_locations(pipe_loc_file, pipe_lines)
                    self.bc_comp.display_option_list.append(
                        XmsDisplayMessage(
                            file=self.bc_comp.disp_opts_files[0], draw_type=DrawType.draw_at_locations
                        )
                    )
            else:  # Clear the pipe display list.
                write_display_option_line_locations(pipe_loc_file, [])
                self.bc_comp.display_option_list.append(
                    XmsDisplayMessage(file=self.bc_comp.disp_opts_files[0], draw_type=DrawType.draw_at_locations),
                )
        return [], []

    def ensure_display_option_files_exist(self):
        """Copies default BC arc and point display option JSON files to the component directory if needed.

        Will create new random UUIDs for the display lists. Should only be called by the unmapped BC coverage on
        creation.
        """
        pipe_categories = CategoryDisplayOptionList()
        if os.path.exists(self.bc_comp.disp_opts_files[1]):  # Repair bad point display option files
            json_dict = read_display_options_from_json(self.bc_comp.disp_opts_files[1])
            pipe_categories = CategoryDisplayOptionList()
            pipe_categories.from_dict(json_dict)
        if len(pipe_categories.categories) < 1:
            default_file = os.path.join(
                os.path.dirname(os.path.dirname(__file__)), 'gui', 'resources', 'default_data', DEFAULT_POINT_BC_JSON
            )
            json_dict = read_display_options_from_json(default_file)
            pipe_categories.from_dict(json_dict)
            write_display_options_to_json(self.bc_comp.disp_opts_files[1], pipe_categories)

        if not os.path.exists(self.bc_comp.disp_opts_files[0]):  # First is arcs, second is points.
            categories = self.read_default_display_options_from_registry(REG_KEY_BC_ARC)
            if not categories:  # No default line display options in registry, use default from resource file
                # Read the default arc display options, and save ourselves a copy with a randomized UUID.
                categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
                default_file = os.path.join(
                    os.path.dirname(os.path.dirname(__file__)), 'gui', 'resources', 'default_data', DEFAULT_BC_JSON
                )
                json_dict = read_display_options_from_json(default_file)
                categories.from_dict(json_dict)
            categories.comp_uuid = self.bc_comp.uuid
            write_display_options_to_json(self.bc_comp.disp_opts_files[0], categories)
            # Save our display list UUID to the main file
            self.bc_comp.data.info.attrs['display_uuid'] = categories.uuid

            if len(self.bc_comp.disp_opts_files) > 1:  # This a BC coverage, not a mapped BC component
                categories = self.read_default_display_options_from_registry(REG_KEY_BC_POINT)
                if not categories:  # No default point display options in registry, use default from resource file
                    # Read the default point display options, and save ourselves a copy with a randomized UUID.
                    categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
                    default_file = os.path.join(
                        os.path.dirname(os.path.dirname(__file__)), 'gui', 'resources', 'default_data',
                        DEFAULT_POINT_BC_JSON
                    )
                    json_dict = read_display_options_from_json(default_file)
                    categories.from_dict(json_dict)
                categories.comp_uuid = self.bc_comp.uuid
                write_display_options_to_json(self.bc_comp.disp_opts_files[1], categories)
                # Save our display list UUID to the main file
                self.bc_comp.data.info.attrs['point_display_uuid'] = categories.uuid
            self.bc_comp.data.commit()

    def get_initial_display_options(self, query, params):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Should contain coverage component id maps as dumped by XMS.

        Returns:
            Empty message and ActionRequest lists
        """
        # Query XMS for parent coverage's UUID if we don't know it yet.
        if not self.bc_comp.cov_uuid:
            self.bc_comp.cov_uuid = query.parent_item_uuid()
            self.bc_comp.data.info.attrs['cov_uuid'] = self.bc_comp.cov_uuid
            self.bc_comp.data.commit()

        if self.bc_comp.cov_uuid:
            # Initialize component coverage ids in XMS if temp files have been passed in.
            if params and 'ARC_ATT_IDS' in params[0]:
                # Update the arc component ids in XMS.
                arc_att_file = params[0]['ARC_ATT_IDS']
                arc_comp_file = params[0]['ARC_COMP_IDS']
                arc_att_ids = read_display_option_ids(arc_att_file)
                arc_comp_ids = read_display_option_ids(arc_comp_file)
                for att_id, comp_id in zip(arc_att_ids, arc_comp_ids):
                    self.bc_comp.update_component_id(TargetType.arc, att_id, comp_id)
                io_util.removefile(arc_att_file)
                io_util.removefile(arc_comp_file)
                # Update the point component ids in XMS.
                point_att_file = params[0]['POINT_ATT_IDS']
                point_comp_file = params[0]['POINT_COMP_IDS']
                point_att_ids = read_display_option_ids(point_att_file)
                point_comp_ids = read_display_option_ids(point_comp_file)
                for att_id, comp_id in zip(point_att_ids, point_comp_ids):
                    self.bc_comp.update_component_id(TargetType.point, att_id, comp_id)
                io_util.removefile(point_att_file)
                io_util.removefile(point_comp_file)
            # Send the display options json files to XMS.
            self.bc_comp.display_option_list = [
                XmsDisplayMessage(file=self.bc_comp.disp_opts_files[0], edit_uuid=self.bc_comp.cov_uuid),  # arcs
                XmsDisplayMessage(file=self.bc_comp.disp_opts_files[1], edit_uuid=self.bc_comp.cov_uuid),  # points
            ]

    @staticmethod
    def get_display_id_file(bc_type, path):
        """Get the display option id file for a BC arc type.

        Args:
            bc_type (:obj:`int`): Index of the BC arc type
            path (:obj:`str`): Path to the location of the component id files

        Returns:
            (:obj:`str`): See description

        """
        if bc_type == bcd.UNASSIGNED_INDEX:
            filename = 'bc_unassigned.display_ids'
        elif bc_type == bcd.OCEAN_INDEX:
            filename = 'bc_ocean.display_ids'
        elif bc_type == bcd.MAINLAND_INDEX:
            filename = 'bc_mainland.display_ids'
        elif bc_type == bcd.ISLAND_INDEX:
            filename = 'bc_island.display_ids'
        elif bc_type == bcd.RIVER_INDEX:
            filename = 'bc_river.display_ids'
        elif bc_type == bcd.LEVEE_OUTFLOW_INDEX:
            filename = 'bc_levee_outflow.display_ids'
        elif bc_type == bcd.LEVEE_INDEX:
            filename = 'bc_levee.display_ids'
        elif bc_type == bcd.RADIATION_INDEX:
            filename = 'bc_radiation.display_ids'
        elif bc_type == bcd.ZERO_NORMAL_INDEX:
            filename = 'bc_zero_normal.display_ids'
        elif bc_type == bcd.FLOW_AND_RADIATION_INDEX:
            filename = 'bc_flow_and_radiation.display_ids'
        else:
            return ''
        return os.path.join(path, filename)

    @staticmethod
    def read_default_display_options_from_registry(registry_key):
        """Retrieve the default display options from the registry.

        Args:
            registry_key (:obj:`str`): The registry key of the display options to load

        Returns:
            (:obj:`CategoryDisplayOptionList`): The default options from the registry, None if not saved yet
        """
        settings = SettingsManager()
        json_text = settings.get_setting('xmsadcirc', registry_key)
        if not json_text:
            return None

        json_dict = orjson.loads(json_text)
        categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
        categories.from_dict(json_dict)
        return categories

    @staticmethod
    def default_pipe_line_atts(category_id):
        """Add default pipe line attributes to a display category list.

        Args:
            category_id (:obj:`int`): The id to assign the category

        Returns:
            (:obj:`CategoryDisplayOption`): The default pipe line options
        """
        cat_opts = CategoryDisplayOption()
        cat_opts.file = BC_POINT_ID_FILE
        cat_opts.description = 'Pipe'
        cat_opts.id = category_id
        line_opts = LineOptions()
        cat_opts.options = line_opts
        return cat_opts

    @staticmethod
    def default_pipe_line_atts_registry(category_id):
        """Add default pipe line attributes to a display category list, but check registry for existing default first.

        Args:
            category_id (:obj:`int`): The id to assign the category

        Returns:
            (:obj:`CategoryDisplayOption`): The default pipe line options
        """
        settings = SettingsManager()
        mapped_bc_arc_options = settings.get_setting('xmsadcirc', REG_KEY_MAPPED_BC_ARC)
        pipe_category = None
        if mapped_bc_arc_options:
            mapped_json_dict = orjson.loads(mapped_bc_arc_options)
            for category in mapped_json_dict['categories']:
                if category['description'] == 'Pipe':
                    pipe_category = category
                    break
        if pipe_category is None:
            pipe_category = BcComponentDisplay.default_pipe_line_atts(category_id)
        return pipe_category

    def _update_id_files(self, bc_type, new_comp_id):
        """Update the XMS component ids for BC categories whose lists have changed.

        Args:
            bc_type (:obj:`int`): The new target BC type index
            new_comp_id (:obj:`int`): The new component id

        """
        path = os.path.dirname(self.bc_comp.main_file)
        # Read the levee component id file.
        levee_file = self.get_display_id_file(bcd.LEVEE_INDEX, path)
        levee_comps = read_display_option_ids(levee_file)
        np_levee_comps = np.array(levee_comps)
        # Find any selected arcs whose old component id belonged to a levee pair
        np_selected_comps = np.array(self.selected_comp_ids)
        mask = np.isin(np_levee_comps, np_selected_comps)
        selected_levees = np_levee_comps[mask]
        unassigned_file = ''
        unassigned_comp_ids = []
        valid_levees = None
        if selected_levees.size > 0:
            # Append the old levee comp id to the unassigned list as the relationship is now broken. May
            # be other unselected arcs with this same component id, and they need to be invalidated.
            unassigned_file = self.get_display_id_file(bcd.UNASSIGNED_INDEX, path)
            unassigned_comp_ids = read_display_option_ids(unassigned_file)
            selected_levees = selected_levees.tolist()
            unassigned_comp_ids.extend(selected_levees)
            write_display_option_ids(unassigned_file, unassigned_comp_ids)
            # Inverse the mask to get the new list of levee component ids that are still valid.
            valid_levees = np_levee_comps[np.logical_not(mask)].tolist()

            # Reset the type of the levee datasets
            for selected_levee in selected_levees:
                self.bc_comp.data.arcs['type'].loc[dict(comp_id=[selected_levee])] = bcd.UNASSIGNED_INDEX
        # Add the new component id to the target category.
        if bc_type == bcd.LEVEE_INDEX:
            # We already read the target category id file if it is levee. We may have also broken some
            # levee relationships and removed some old comp ids from the list.
            updated_comp_ids = levee_comps if valid_levees is None else valid_levees
            updated_display_file = levee_file
        elif bc_type == bcd.UNASSIGNED_INDEX and unassigned_file:
            # We may have already read the target category id file if it is unassigned.
            updated_comp_ids = unassigned_comp_ids
            updated_display_file = unassigned_file
        else:  # Target category is not one we have read yet.
            updated_display_file = self.get_display_id_file(bc_type, path)
            updated_comp_ids = read_display_option_ids(updated_display_file)
        updated_comp_ids.append(new_comp_id)
        write_display_option_ids(updated_display_file, updated_comp_ids)
        # Make sure the levee id file has been updated if it needs to be.
        if bc_type != bcd.LEVEE_INDEX and valid_levees is not None:
            # The target category was not levee, but some of the selected arcs used to be.
            # Rewrite the levee comp id file so they get removed from the list.
            write_display_option_ids(levee_file, valid_levees)

        # Send back updated display lists to XMS after ActionRequest
        self.bc_comp.display_option_list.append(
            # Update the arc display list only.
            XmsDisplayMessage(file=self.bc_comp.disp_opts_files[0], edit_uuid=self.bc_comp.cov_uuid)
        )

    def _update_point_id_files(self, new_comp_id):
        """Update the XMS component ids for pipe point categories whose lists have changed.

        Args:
            new_comp_id (:obj:`int`): The new component id

        """
        path = os.path.dirname(self.bc_comp.main_file)
        # Add the new component id to the target category.
        updated_display_file = os.path.join(path, BC_POINT_ID_FILE)
        updated_comp_ids = read_display_option_ids(updated_display_file)
        updated_comp_ids.append(new_comp_id)
        write_display_option_ids(updated_display_file, updated_comp_ids)

        # Send back updated display lists to XMS after ActionRequest
        self.bc_comp.display_option_list.append(
            # Update the point display list only.
            XmsDisplayMessage(file=self.bc_comp.disp_opts_files[1], edit_uuid=self.bc_comp.cov_uuid)
        )

    def _unpack_xms_data(self, param_map, for_arcs):
        """Unpack the selection info and component id maps sent by XMS.

        Args:
            param_map (:obj:`dict`): The ActionRequest parameter map
            for_arcs (:obj:`bool`): True if looking for arc data, False for points

        Returns:
            (:obj:`int`): component id of atts to display

        """
        # Get the XMS attribute ids of the selected for_arcs (if any)
        self.selected_att_ids = param_map.get('selection', [])
        if not self.selected_att_ids:
            return UNINITIALIZED_COMP_ID

        # Get the component id map of the selected for_arcs (if any).
        comp_id = UNINITIALIZED_COMP_ID
        if 'id_files' in param_map and param_map['id_files'] and param_map['id_files'][0]:
            if for_arcs:
                files_dict = {'ARC': (param_map['id_files'][0], param_map['id_files'][1])}
            else:
                files_dict = {'POINT': (param_map['id_files'][0], param_map['id_files'][1])}
            self.bc_comp.load_coverage_component_id_map(files_dict)
            comp_id = self._check_selected_types(for_arcs)

        return comp_id

    def _check_selected_types(self, for_arcs):
        """Determine which attributes to display in Assign BC dialog and any warning message that should be added.

        Returns:
            (:obj:`int`): component id of attributes to display

        """
        num_features = len(self.selected_att_ids)
        # 1 arc selected, use those atts
        target_type = TargetType.arc if for_arcs else TargetType.point
        if num_features == 1:
            comp_id = self.bc_comp.get_comp_id(target_type, self.selected_att_ids[0])
            self.selected_comp_ids.append(comp_id)
            return comp_id
        else:  # More than one arc selected, check types
            # Get the component ids of all the selected arcs
            try:
                self.selected_comp_ids = list(self.bc_comp.comp_to_xms[self.bc_comp.cov_uuid][target_type].keys())
            except KeyError:
                return UNINITIALIZED_COMP_ID  # No component ids assigned for any of the selected arcs

            # If there are two arcs that share the same component id, display their attributes
            if num_features == 2:
                arc1_comp = self.bc_comp.get_comp_id(target_type, self.selected_att_ids[0])
                arc2_comp = self.bc_comp.get_comp_id(target_type, self.selected_att_ids[1])
                if arc1_comp == arc2_comp:
                    self.is_component_pair = True
                    return arc1_comp
            if target_type == TargetType.arc:
                self.dlg_message = 'Multiple arcs selected. Changes will apply to all selected arcs. Any selected ' \
                                   'levee will be cleared.'
            else:
                self.dlg_message = 'Multiple pipe points selected. Changes will apply to all selected points.'
            return UNINITIALIZED_COMP_ID

    def _build_pipe_line(self, xmgrid, node1, node2):
        """Helper method for building up the list of pipe line locations for free-draw display.

        Args:
            xmgrid (:obj:`UGrid2d`): The XmUGrid containing the node locations
            node1 (:obj:`int`): 1-based id of the first node in the pipe pair
            node2 (:obj:`int`): 1-based id of the second node in the pipe pair

        Returns:
            (:obj:`list`): list containing the x,y,z coordinates of the two pipe nodes

        """
        ugrid = xmgrid.ugrid
        pipe_line = list(ugrid.get_point_location(node1 - 1))
        pipe_line.extend(ugrid.get_point_location(node2 - 1))
        return pipe_line

    def _update_mapped_registry_defaults(self):
        """Keep the BC coverage and mapped BC component default registry options in sync."""
        settings = SettingsManager()
        # Read the default display options for BC coverages.
        bc_arc_options = settings.get_setting('xmsadcirc', REG_KEY_BC_ARC)
        if not bc_arc_options:
            return  # No defaults set in registry
        json_dict = orjson.loads(bc_arc_options)
        json_dict['is_ids'] = 0  # Change from draw at ids to draw at locations.

        # Append a category for pipe lines. They are not represented as points in the mapped component. See if there
        # is existing defaults for mapped components in registry, otherwise create a default.
        pipe_category = self.default_pipe_line_atts_registry(json_dict['categories'][-1]['id'] + 1)
        json_dict['categories'].append(dict(pipe_category))

        # Copy from BC coverage registry key to mapped component registry key.
        json_text = orjson.dumps(json_dict)
        settings.save_setting('xmsadcirc', REG_KEY_MAPPED_BC_ARC, json_text)

    def _update_coverage_registry_defaults(self):
        """Keep the BC coverage and mapped BC component default registry options in sync."""
        settings = SettingsManager()
        # Read the default display options for mapped BC components.
        bc_arc_options = settings.get_setting('xmsadcirc', REG_KEY_MAPPED_BC_ARC)
        if not bc_arc_options:
            return  # No defaults set in registry
        json_dict = orjson.loads(bc_arc_options)
        json_dict['is_ids'] = 1  # Change from draw at locations to draw at ids.

        # Remove the pipe line category. They are represented as points in the BC coverage.
        pipe_idx = None
        for idx, category in enumerate(json_dict['categories']):
            if category['description'] == 'Pipe':
                pipe_idx = idx
                break
        if pipe_idx is not None:
            json_dict['categories'].pop(pipe_idx)

        # Copy from mapped component registry key to BC coverage registry key.
        json_text = orjson.dumps(json_dict)
        settings.save_setting('xmsadcirc', REG_KEY_BC_ARC, json_text)
