"""Module to perform xms.api Query operations for the ADCIRC simulation component."""

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.tree import tree_util

# 4. Local modules
from xms.adcirc.data.sim_data import SimData
from xms.adcirc.feedback.xmlog import XmLog


def unlink_tidal_comp_on_error(tidal_uuid, query):
    """Unlink the source tidal component link from the sim.

    Args:
        tidal_uuid (str): UUID of the tidal simulation that was being mapped
        query (Query): Interprocess communicator
    """
    query.clear()
    tidal_item_ptrs = []
    tree_util.find_linked_pointers_by_uuid(query.project_tree, tidal_uuid, tidal_item_ptrs)
    for linked_pointer in tidal_item_ptrs:
        linked_pointer_parent = linked_pointer.parent
        if linked_pointer_parent and linked_pointer_parent.model_name == 'ADCIRC':
            sim_uuid = linked_pointer_parent.uuid  # Should be one and only one in the entire tree
            query.unlink_item(sim_uuid, tidal_uuid)
            return


class SimComponentQueries:
    """Performs xms.api calls for the sim component."""
    def __init__(self, sim_comp, query):
        """Initialize the API helper.

        Args:
            sim_comp (:obj:`SimComponent`): The component instance to perform API call for
            query (:obj:`xms.api.dmi.ActionRequest`): Object for communicating with XMS
        """
        self.sim_comp = sim_comp
        self.query = query
        self._pe_tree = None
        self._sim_uuid = None
        self._sim_node = None

    @property
    def pe_tree(self):
        """Property for retrieving the project explorer."""
        if self._pe_tree is None:
            self._pe_tree = self.query.project_tree
        return self._pe_tree

    @property
    def sim_uuid(self):
        """Property for retrieving the simulation UUID."""
        if self._sim_uuid is None:
            self._sim_uuid = self.query.parent_item_uuid()
            if not self._sim_uuid or self._sim_uuid == '00000000-0000-0000-0000-000000000000':
                self._sim_uuid = self.query.current_item_uuid()
        return self._sim_uuid

    @property
    def sim_node(self):
        """Property for retrieving the simulation tree node."""
        if self._sim_node is None:
            self._sim_node = tree_util.find_tree_node_by_uuid(self.pe_tree, self.sim_uuid)
        return self._sim_node

    def get_bc_main_file(self):
        """Query for a BC coverage component mainfile.

        Will only be taken temporarily while the mapped component is created.

        Returns:
            (:obj:`str`): Path to the BC component's mainfile, empty string on error.
        """
        # Make sure the simulation has a mesh and get the hidden BC component's main file
        try:
            if not self.find_sim_link('TI_MESH2D_PTR'):
                XmLog(
                ).instance.error('Unable to apply boundary conditions because no mesh was found in the simulation.')
            else:
                bc_node = self.find_sim_link('TI_COVER_PTR', coverage_type='Boundary Conditions')
                if not bc_node:
                    raise RuntimeError()
                bc_comp = self.query.item_with_uuid(bc_node.uuid, model_name='ADCIRC', unique_name='Bc_Component')
                return bc_comp.main_file
        except Exception:
            XmLog().instance.exception('Unable to retrieve boundary condition attributes from SMS.')
        return ''

    def get_tidal_mapping_data(self):
        """Query for the XMS data required for mapping a tidal constituent component to the simulation.

        Data dictionary contains:
            ::
                {
                    mapped_bc_main_file = str
                    tidal_main_file = str
                    tidal_uuid = str
                    tidal_name = str
                    sim_uuid = str
                    display_projection = str
                    error = bool
                {

        Returns:
            (:obj:`dict`): A data dictionary containing the tidal constituent mapping data from XMS
        """
        tidal_data = {
            'mapped_bc_main_file': '',
            'tidal_main_file': '',
            'tidal_uuid': '',
            'tidal_name': '',
            'sim_uuid': '',
            'sim_reftime': '',
            'run_duration': 0.0,  # RUNDAY in hours
            'display_projection': '',
            'error': '',
        }
        # Get the mainfile and UUID of the source tidal component. Get the parent simulation's UUID and the Context at
        # that level. Make sure the simulation has a mapped BC component and get its mainfile.
        try:
            # Find the parent ADCIRC simulation and store off the current display projection.
            tidal_data['sim_uuid'] = self.query.parent_item_uuid()
            sim_item = tree_util.find_tree_node_by_uuid(self.query.project_tree, tidal_data['sim_uuid'])
            tidal_data['display_projection'] = self.query.display_projection.coordinate_system

            # Get the run duration in hours
            sim_data = SimData(sim_item.main_file)
            tidal_data['run_duration'] = sim_data.timing.attrs['RUNDAY'] * 24.0
            tidal_data['sim_reftime'] = sim_data.timing.attrs['ref_date']

            # Find the taken tidal constituent simulation (should be one and only one).
            XmLog().instance.info('Exporting source tidal constituent component.')
            tidal_item = self.find_sim_link('TI_DYN_SIM_PTR')
            tidal_data['tidal_uuid'] = tidal_item.uuid
            tidal_data['tidal_name'] = tidal_item.name
            # Get the tidal simulation's component.
            tidal_comp = self.query.item_with_uuid(
                tidal_item.uuid, model_name='Tidal Constituents', unique_name='Tidal_Component'
            )
            tidal_data['tidal_main_file'] = tidal_comp.main_file

            # Get the mapped boundary conditions.
            XmLog().instance.info('Exporting applied boundary conditions component.')
            mapped_bcs = tree_util.descendants_of_type(sim_item, unique_name='Mapped_Bc_Component', recurse=False)
            if not mapped_bcs:
                XmLog().instance.error(
                    'Boundary conditions must be applied to the ADCIRC simulation before tidal constituents.'
                )
                tidal_data['error'] = True
            else:
                # find the first mapped bc that has ocean nodes
                from xms.adcirc.components.mapped_bc_component import MappedBcComponent
                found = False
                for mapped_bc in mapped_bcs:
                    mapped_bc_comp = MappedBcComponent(mapped_bc.main_file)
                    ocean_nodes, _ = mapped_bc_comp.data.get_ocean_node_ids()
                    if len(ocean_nodes) > 0:
                        tidal_data['mapped_bc_main_file'] = mapped_bc.main_file
                        found = True
                        break
                if not found:
                    XmLog().instance.exception('There must be ocean nodes applied to the simulation for tidal '
                                               'constituents.')
                    tidal_data['error'] = True
        except Exception:
            XmLog().instance.exception('Unable to apply tidal constituents to ADCIRC simulation.')
            tidal_data['error'] = True
        return tidal_data

    def unlink_item(self, tree_enum, coverage_type=None):
        """Unlink an item from the simulation.

        Args:
            tree_enum (:obj:`str`): The stringified enum from XMS.
            coverage_type (:obj:`str`): The coverage type name. Used to differentiate between linked coverages.
        """
        if self.sim_node:
            child = tree_util.descendants_of_type(
                self.sim_node,
                xms_types=[tree_enum],
                allow_pointers=True,
                only_first=True,
                coverage_type=coverage_type,
                model_name='ADCIRC'
            )
            if child:
                self.query.unlink_item(self.sim_uuid, child.uuid)
                self.sim_comp.data.commit()

    def find_sim_link(self, tree_enum, coverage_type=None):
        """Find child links of the simulation. For ADCIRC only ever one of each link type.

        Args:
            tree_enum (:obj:`str`): The stringified enum from XMS.
            coverage_type (:obj:`str`): Coverage type if requesting a linked coverage

        Returns:
            (:obj:`TreeNode`) See description
        """
        child = None
        if self.sim_node:
            child = tree_util.descendants_of_type(
                self.sim_node,
                xms_types=[tree_enum],
                allow_pointers=True,
                only_first=True,
                recurse=True,
                coverage_type=coverage_type
            )
        return child
