"""Worker thread for the fort.14 importer."""

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.components.display import windows_gui as win_gui
from xms.guipy.dialogs import xms_parent_dlg as xms_dlg
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackThread
from xms.guipy.dialogs.xms_parent_dlg import ensure_qapplication_exists

# 4. Local modules
from xms.adcirc.feedback.xmlog import XmLog
from xms.adcirc.file_io.fort14_reader import Fort14Reader
from xms.adcirc.gui.import_process_feedback_dlg import ImportProcessFeedbackDlg


class Fort14ImportWorkerThread(ProcessFeedbackThread):
    """Worker thread for importing an ADCIRC fort.14 file."""
    def __init__(self, parent):
        """Construct the worker.

        Args:
            parent (:obj:`QWidget`): Parent of the QThread, probably the hidden parent dialog created by XMS.
        """
        super().__init__(parent=parent, do_work=self._do_work)
        self.reader = None

    def _do_work(self):
        """Thread runner that imports the fort.14."""
        try:
            self.reader = Fort14Reader("")  # Read the fort.14 as a control file. A new simulation will be created.
            self.reader.read()
        except Exception:
            XmLog().instance.exception('Error(s) encountered while reading fort.14 file.')


def import_fort14_with_feedback():
    """Driver for the fort.14 import script with a feedback dialog."""
    ensure_qapplication_exists()

    # Parse parent HWNDs and icon path from commandline arguments.
    parent_hwnd, main_hwnd, _ = xms_dlg.parse_parent_window_command_args()
    win_cont = xms_dlg.get_parent_window_container(parent_hwnd)

    # Create the timer that keeps our Python dialog in the foreground of XMS.
    _ = win_gui.create_and_connect_raise_timer(main_hwnd, win_cont)  # Keep the timer in scope

    worker = Fort14ImportWorkerThread(win_cont)
    display_text = {
        'title': 'Reading ADCIRC Domain and Boundary Conditions',
        'working_prompt': 'Reading fort.14 ADCIRC geometry and boundary conditions file. Please wait...',
        'error_prompt': 'Error(s) encountered reading fort.14. Review log output for more details.',
        'warning_prompt': 'Warning(s) encountered reading fort.14. Review log output for more details.',
        'success_prompt': 'Successfully read ADCIRC fort.14 file.',
        'note': '',
        'auto_load': 'Auto load data into SMS when operation is complete',
    }
    feedback_dlg = ImportProcessFeedbackDlg(display_text, 'xms.adcirc', worker, win_cont)
    feedback_dlg.testing = XmEnv.xms_environ_running_tests() == 'TRUE'
    if feedback_dlg.exec() and not LogEchoQSignalStream.logged_error:  # Only send if no error.
        worker.reader.send()  # Send data back to XMS
    win_gui.raise_main_xms_window(main_hwnd)  # Bring top-level Python window to foreground
