"""Testing utilities."""

# 1. Standard Python modules

# 2. Third party modules
import h5py

# 3. Aquaveo modules

# 4. Local modules


def compare_h5_files(output_file, base_file, to_ignore):
    """Compare two different H5 files.

    Args:
        output_file (:obj:`str`): The path to the output file that needs to be compared.
        base_file (:obj:`str`): The path to the base file against which the output file should be compared.
        to_ignore (:obj:`list`): A list of keys to ignore during the comparison.

    Returns:
        (:obj:`tuple`): A tuple containing the result of the comparison, and a message providing additional information
        about the result.
    """
    # update_baselines = True  # Remember to set back to False after baselines are updated.
    # if update_baselines:
    #     filesystem.copyfile(output_file, base_file)
    #     return True, 'Updated baselines'

    with h5py.File(output_file, 'r') as f1:
        with h5py.File(base_file, 'r') as f2:
            try:
                implicit_ignore = [
                    'dim_0', 'DIMENSION_LIST', 'REFERENCE_LIST', '_Netcdf4Dimid', '_NCProperties',
                    '_Netcdf4Coordinates', '_nc3_strict', 'CLASS', 'NAME', '_FillValue'
                ]
                ignore = to_ignore
                ignore.extend(implicit_ignore)
                result = _compare_h5_groups('/', f1, f2, ignore)
                message = f'{result[1]}\n{output_file}\n{base_file}\n'
                return result[0], message
            except Exception as e:
                return False, f'compare_h5_files exception: {str(e)}'


def _compare_h5_groups(path, group_out, group_base, to_ignore):
    """Compares two h5 groups and their datasets, including attributes and values.

    Args:
        path (:obj:`str`): the path of the current group.
        group_out (:obj:`h5py.Group`): the output group to compare.
        group_base (:obj:`h5py.Group`): the base group to compare against.
        to_ignore (:obj:`list`): a list of attributes or sub-groups to ignore during comparison.

    Returns:
        (:obj:`tuple(bool,str)`): a tuple indicating whether the comparison is successful and an error message if
        applicable. The first element of the tuple represents the success of the comparison, with True indicating
        success. The second element of the tuple is an empty string if the comparison is successful, or an error
        message if there are any mismatches or missing datasets/attributes.
    """
    g_base = set(group_base.keys())  # & g_out

    for sub_group in g_base:
        sub_path = f'{path}{sub_group}/'

        # pandas uses 'index' while xarray uses 'dim_0', both work equally well,
        # and the values in them are not relevant to our tests.
        # Check to make sure at least one of them is present.
        if sub_group == 'index' or sub_group == 'dim_0':
            if 'index' in group_out:
                sub_group_out = group_out['index']
            elif 'dim_0' in group_out:
                sub_group_out = group_out['dim_0']
            else:
                sub_group_out = None
                error = f'{sub_path}: Dataset missing index.'
                return False, error
            continue
        if sub_group not in group_out:
            error = f'{sub_path}: Dataset missing.'
            return False, error
        is_dset_out = isinstance(group_out[sub_group], h5py.Dataset)
        is_dset_base = isinstance(group_base[sub_group], h5py.Dataset)
        sub_group_out = group_out[sub_group]
        sub_group_base = group_base[sub_group]

        if sub_group in to_ignore:
            continue

        # check the attributes
        a_set_base = set(sub_group_base.attrs.keys())
        for att in a_set_base:
            # print(f'att: {att}')
            if att in to_ignore:
                continue
            attr_out = sub_group_out.attrs[att]
            attr_base = sub_group_base.attrs[att]
            if attr_out != attr_base:
                error = f'{sub_path}: Attribute value mismatch ({att}))'
                return False, error
            if sub_group_out.attrs[att]:
                type_out = str(type(sub_group_out.attrs[att][0]))
            else:
                type_out = None
            if sub_group_base.attrs[att]:
                type_base = str(type(sub_group_base.attrs[att][0]))
            else:
                type_base = None
            if type_out != type_base:
                error = f'{sub_path}: Attribute type mismatch ({type_out} != {type_base})'
                return False, error

        if is_dset_out and is_dset_base:
            if sub_group_out.shape != sub_group_base.shape:
                error = f'{sub_path}: Dataset shape mismatch.'
                return False, error
            if sub_group_out.dtype != sub_group_base.dtype:
                error = f'{sub_path}: Dataset type mismatch.'
                return False, error
            array_out = sub_group_out[()]
            array_base = sub_group_base[()]
            equal = (array_out == array_base).all()
            if not equal:
                error = f'{sub_path}: Dataset value mismatch.'
                return False, error
        elif not is_dset_out and not is_dset_base:
            # recurse
            result = _compare_h5_groups(sub_path, sub_group_out, sub_group_base, to_ignore)
            if not result[0]:
                return result
        else:
            error = f'{sub_path}: Dataset/group mismatch.'
            return False, error
    return True, ''
