"""Qt dialog for assigning attributes of a Boundary Conditions coverage pipe point."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtWidgets import (QDialogButtonBox, QGroupBox, QLabel, QLineEdit, QSizePolicy, QSpacerItem, QVBoxLayout)

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules


class AdcircPipeDlg(XmsDlg):
    """A dialog showing the Assign Pipe dialog."""
    def __init__(self, win_cont, pipe_data, msg, num_selected):
        """Initializes the sediment material list and properties dialog.

        Args:
            win_cont (:obj:`QWidget`): Parent dialog
            pipe_data (:obj:`xarray.Dataset`): Dataset containing the pipe parameters
            msg (:obj:`str`): Warning/multi-select message, if any
            num_selected (:obj:`int`): The number of points currently selected
        """
        super().__init__(win_cont, 'xmsadcirc.gui.assign_pipe_dlg')
        self._help_url = 'https://www.xmswiki.com/wiki/SMS:ADCIRC_Coverages#Pipes'
        self._data = pipe_data
        self._num_selected = num_selected
        self._msg = msg
        self._widgets = {}
        self._dbl_noneg_validator = None
        self._dbl_validator = None

        # Setup the dialog
        self.setWindowTitle('ADCIRC Pipe')
        self._setup_ui()

    @property
    def data(self):
        """Read-only accessor for the pipe attribute dataset."""
        return self._data

    def _setup_ui(self):
        """Setup widgets in the dialog."""
        # Add the message label if not empty
        self._widgets['group_box'] = QGroupBox('Pipe options')
        self._widgets['lbl_height'] = QLabel('Height:')
        self._widgets['lbl_coeff'] = QLabel('Coefficient:')
        self._widgets['lbl_diameter'] = QLabel('Diameter:')
        self._widgets['edt_height'] = QLineEdit()
        self._widgets['edt_coeff'] = QLineEdit()
        self._widgets['edt_diameter'] = QLineEdit()
        self._widgets['vert_layout'] = QVBoxLayout()
        if self._msg:
            label = QLabel(self._msg)
            label.setStyleSheet('QLabel{color: rgb(255, 0, 0);}')
            self._widgets['vert_layout'].addWidget(label)
        self._widgets['vert_layout'].addWidget(self._widgets['lbl_height'])
        self._widgets['vert_layout'].addWidget(self._widgets['edt_height'])
        self._widgets['vert_layout'].addWidget(self._widgets['lbl_coeff'])
        self._widgets['vert_layout'].addWidget(self._widgets['edt_coeff'])
        self._widgets['vert_layout'].addWidget(self._widgets['lbl_diameter'])
        self._widgets['vert_layout'].addWidget(self._widgets['edt_diameter'])
        self._widgets['group_box'].setLayout(self._widgets['vert_layout'])
        self._widgets['vert_layout_main'] = QVBoxLayout()
        self._widgets['vert_layout_main'].addWidget(self._widgets['group_box'])

        # Add a vertical spacer
        self._widgets['vert_spacer'] = QSpacerItem(40, 20, QSizePolicy.Minimum, QSizePolicy.Expanding)
        self._widgets['vert_layout_main'].addSpacerItem(self._widgets['vert_spacer'])

        # Add the buttonbox
        self._widgets['btn_box'] = QDialogButtonBox()
        self._widgets['btn_box'].setStandardButtons(
            QDialogButtonBox.Ok | QDialogButtonBox.Cancel | QDialogButtonBox.Help
        )
        self._widgets['btn_box'].accepted.connect(self.accept)
        self._widgets['btn_box'].rejected.connect(self.reject)
        self._widgets['btn_box'].helpRequested.connect(self.help_requested)
        self._widgets['vert_layout_main'].addWidget(self._widgets['btn_box'])
        self.setLayout(self._widgets['vert_layout_main'])

        # Add validators to the edit fields.
        self._dbl_noneg_validator = QxDoubleValidator(bottom=0.0, parent=self)
        self._dbl_validator = QxDoubleValidator(parent=self)
        self._widgets['edt_height'].setValidator(self._dbl_validator)
        self._widgets['edt_coeff'].setValidator(self._dbl_noneg_validator)
        self._widgets['edt_diameter'].setValidator(self._dbl_noneg_validator)

        # Populate widgets from data
        self._load_data()

    def accept(self):
        """Override default accept slot to update persistent dataset."""
        self._save_data()
        super().accept()

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self._help_url)

    def _load_data(self):
        """Populate widgets from persistent data."""
        self._widgets['edt_height'].setText(str(self._data['Height'].data[0].item()))
        self._widgets['edt_coeff'].setText(str(self._data['Coefficient'].data[0].item()))
        self._widgets['edt_diameter'].setText(str(self._data['Diameter'].data[0].item()))

    def _save_data(self):
        """Store widget values in the persistent dataset on 'OK'."""
        # Will reassign a new component id outside the dialog
        self._data['Height'] = float(self._widgets['edt_height'].text())
        self._data['Coefficient'] = float(self._widgets['edt_coeff'].text())
        self._data['Diameter'] = float(self._widgets['edt_diameter'].text())
