"""Qt table view for extracted tidal constituent properties."""

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtGui import QFontMetrics
from PySide2.QtWidgets import QAbstractButton, QAbstractItemView, QLabel, QVBoxLayout

# 3. Aquaveo modules
from xms.guipy.delegates.edit_field_validator import EditFieldValidator
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.adcirc.gui.adcirc_table import AdcircTableWidget

TBL_COL_AMPLITUDE = 0
TBL_COL_FREQUENCY = 1
TBL_COL_NODAL_FACTOR = 2
TBL_COL_EQUILIBRIUM_ARG = 3
TBL_COL_ETRF = 4


class ConstituentPropertiesTableWidget(AdcircTableWidget):
    """The harmonic analysis table widget."""
    def __init__(self, data_frame, tidal_cons, parent=None):
        """Construct the widget.

        Args:
            data_frame (:obj:`pandas.DataFrame`): The model data.
            tidal_cons (:obj:`bool`): True if this dialog is for tidal constituents, False for flow constituents
            parent (Something derived from :obj:`QObject`): The parent object.
        """
        # Rename and order columns to look good in table header.
        self.tidal_cons = tidal_cons
        if tidal_cons:
            data_frame = self._rename_columns_for_gui(
                data_frame[[
                    'amplitude',
                    'frequency',
                    'nodal_factor',
                    'equilibrium_argument',
                    'earth_tide_reduction_factor',
                ]]
            )
            super().__init__(
                parent, data_frame, TBL_COL_FREQUENCY, {
                    'Tidal\nPotential\nAmplitude': 0.0,
                    'Frequency': 0.0,
                    'Nodal Factor': 0.0,
                    'Equilibrium\nArgument': 0.0,
                    'Earth Tide\nReduction\nFactor': 0.0,
                }
            )
        else:  # No tidal potential amplitude or ETRF columns for flow constituents
            data_frame = self._rename_columns_for_gui(
                data_frame[[
                    'frequency',
                    'nodal_factor',
                    'equilibrium_argument',
                ]]
            )
            super().__init__(
                parent, data_frame, TBL_COL_FREQUENCY, {
                    'Frequency': 0.0,
                    'Nodal Factor': 0.0,
                    'Equilibrium\nArgument': 0.0,
                }
            )
        self.dbl_delegate = None
        self.setup_ui()

    def setup_ui(self):
        """Add the table widget and initialize the model."""
        dbl_validator = QxDoubleValidator(parent=self)
        self.dbl_delegate = EditFieldValidator(dbl_validator, self)
        if self.tidal_cons:
            delegates = {
                TBL_COL_AMPLITUDE: self.dbl_delegate,
                TBL_COL_FREQUENCY: self.dbl_delegate,
                TBL_COL_NODAL_FACTOR: self.dbl_delegate,
                TBL_COL_EQUILIBRIUM_ARG: self.dbl_delegate,
                TBL_COL_ETRF: self.dbl_delegate,
            }
        else:  # Flow constituents. No tidal potential amplitude or ETRF columns
            delegates = {
                TBL_COL_FREQUENCY - 1: self.dbl_delegate,
                TBL_COL_NODAL_FACTOR - 1: self.dbl_delegate,
                TBL_COL_EQUILIBRIUM_ARG - 1: self.dbl_delegate,
            }
        super()._setup_ui(delegates, False, True)
        self.table_view.setEditTriggers(QAbstractItemView.AllEditTriggers)
        # Set text for the vertical header column
        corner_btn = self.table_view.findChild(QAbstractButton)
        if corner_btn is not None:
            # Create a text label for the vertical header. This could just be a non-editable column in the table
            # I suppose, but this is just how the data gets represented using default base class behavior.
            layout = QVBoxLayout(corner_btn)
            layout.setContentsMargins(0, 0, 0, 0)
            header_lbl = QLabel('Name')
            header_lbl.setContentsMargins(0, 0, 0, 0)
            header_lbl.setAlignment(Qt.AlignCenter)
            layout.addWidget(header_lbl, alignment=Qt.AlignCenter)
            # Expand the vertical column so header text is visible.
            font_metrics = QFontMetrics(header_lbl.font())
            self.table_view.verticalHeader().setMinimumWidth(font_metrics.horizontalAdvance(header_lbl.text()) + 5)
            self.table_view.verticalHeader().setDefaultAlignment(Qt.AlignCenter)

    def _rename_columns_for_gui(self, data_frame):
        """Rename the DataFrame columns to user friendly text.

        Args:
            data_frame (:obj:`pandas.DataFrame`): The constituent properties data

        Returns:
            (:obj:`pandas.DataFrame`): The input DataFrame with renamed columns
        """
        if self.tidal_cons:
            data_frame = data_frame.rename(
                columns={
                    'amplitude': 'Tidal\nPotential\nAmplitude',
                    'frequency': 'Frequency',
                    'nodal_factor': 'Nodal Factor',
                    'equilibrium_argument': 'Equilibrium\nArgument',
                    'earth_tide_reduction_factor': 'Earth Tide\nReduction\nFactor',
                }
            )
        else:  # No tidal potential amplitude or ETRF columns for flow constituents
            data_frame = data_frame.rename(
                columns={
                    'frequency': 'Frequency',
                    'nodal_factor': 'Nodal Factor',
                    'equilibrium_argument': 'Equilibrium\nArgument',
                }
            )
        return data_frame
