"""This provides a way to store hydrodynamic materials."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from adhparam.material_properties import MaterialProperties
import pandas as pd

# 3. Aquaveo modules
from xms.guipy.data.polygon_texture import PolygonOptions

# 4. Local modules
from xms.adh.data.json_export import include_dataframe_if_not_empty, make_json_serializable, parameterized_to_dict


# 4. Local modules


class Materials:
    """This is a class for storing AdH materials."""
    def __init__(self):
        """Constructor."""
        self.material_properties = {0: MaterialProperties()}
        self.material_properties[0].material_name = 'OFF'
        self.material_display = {0: PolygonOptions()}
        self.friction = pd.DataFrame(
            data=[], columns=["CARD", "CARD_2", "STRING_ID", "REAL_01", "REAL_02", "REAL_03", "REAL_04", "REAL_05"]
        )
        self.material_use_meteorological = {0: False}
        self.material_meteorological_curve = {0: -1}
        self.friction_use_seasonal = {0: False}
        self.friction_seasonal_curve = {0: -1}
        self.time_series = {}

    def add_material(self, material, number=None):
        """Set a new material or replace an existing one.

        Args:
            material (MaterialProperty): The adhparam material.
            number (int): The material id.
        """
        if number is None:
            number = max(self.material_properties, key=int) + 1

        material.material_name = f'Material {number}'
        self.material_properties[number] = material
        self.material_display[number] = PolygonOptions()
        self.material_use_meteorological[number] = False
        self.material_meteorological_curve[number] = 0
        self.friction_use_seasonal[number] = False
        self.friction_seasonal_curve[number] = 1
        return number

    def delete_material(self, number):
        """Delete an existing material.

        Args:
            number (int): The material id.
        """
        self.material_properties.pop(number, None)
        self.material_display.pop(number, None)
        self.friction_use_seasonal.pop(number, None)
        self.friction_seasonal_curve.pop(number, None)
        self.material_use_meteorological.pop(number, None)
        self.material_meteorological_curve.pop(number, None)

    def as_dict(self) -> dict:
        """
        Converts the object's attributes and related data into a JSON serializable dictionary format.

        Returns:
            dict: A dictionary containing the serialized and filtered attributes of
            the object.

        Raises:
            TypeError: If processing the attributes results in an invalid data type.
        """
        material_properties = {
            key: parameterized_to_dict(value) if hasattr(value, 'param') else value.__dict__
            for key, value in self.material_properties.items()
        }

        # Convert time_series using parameterized_to_dict if they are Parameterized objects
        time_series = {
            series_id: parameterized_to_dict(series) if hasattr(series, 'param') else series
            for series_id, series in self.time_series.items()
        }

        # Use the helper functions to simplify the logic for DataFrames
        output = {
            "material_properties": material_properties,
            "friction": include_dataframe_if_not_empty(self.friction),
            "material_use_meteorological": self.material_use_meteorological,
            "material_meteorological_curve": self.material_meteorological_curve,
            "friction_use_seasonal": self.friction_use_seasonal,
            "friction_seasonal_curve": self.friction_seasonal_curve,
            "time_series": time_series,
        }

        output = make_json_serializable(output)
        return output
