"""A dialog for viewing all specified weirs, flap gates, and sluice gates."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
import pandas as pd
from PySide2.QtCore import QSortFilterProxyModel, Qt
from PySide2.QtWidgets import (
    QAbstractItemView, QApplication, QDialog, QPushButton, QStyle, QStyledItemDelegate, QStyleOptionButton
)

# 3. Aquaveo modules
from xms.guipy.data.target_type import TargetType
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel

# 4. Local modules
from xms.adh.data.structures_io import StructuresIO
import xms.adh.gui.resources.adhqrc  # noqa F401
from xms.adh.gui.view_all_dialog_ui import Ui_ViewAllDialog


class PushButtonDelegate(QStyledItemDelegate):
    """A delegate for putting push buttons on the table view."""
    def __init__(self, parent):
        """Constructs the delegate.

        Args:
            parent (QObject): The parent object for the delegate.
        """
        super().__init__(parent)
        self.button = QPushButton()
        self.button.setFlat(False)

    def paint(self, painter, option, index):
        """Draws a push button on the table.

        Args:
            painter (:obj:`PySide2.QPainter`): The object that does the painting.
            option (:obj:`PySide2.QStyleOption`): The object that holds style options for drawing and current state.
            index (:obj:`PySide2.QModelIndex`): The model index to be drawn.
        """
        if not index.isValid():
            return

        btn_opt = QStyleOptionButton()
        btn_opt.rect = option.rect
        btn_opt.text = 'Delete'
        btn_opt.state = option.state
        btn_opt.editable = False
        QApplication.style().drawControl(QStyle.CE_PushButton, btn_opt, painter, self.button)

    def editorEvent(self, event, model, option, index):  # noqa: N802
        """Does nothing. Here to prevent the base class from doing anything."""
        return False

    def createEditor(self, parent, option, index):  # noqa: N802
        """Does nothing. Here to prevent the base class from doing anything."""
        return None


class ViewAllDialog(QDialog):
    """A dialog for assigning materials to polygons."""
    def __init__(self, win_cont, icon, title, data, id_map):
        """Allows the user to edit arc boundary conditions.

        Args:
            win_cont (QWidget): Parent window
            icon (QIcon): Window icon
            title (str): Window title
            data (StructuresIO): Structures object
            id_map (:obj:`dict` of int): A dictionary with component id keys and attribute id values.
        """
        super().__init__(win_cont)
        self.ui = Ui_ViewAllDialog()
        self.ui.setupUi(self)
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:ADH_Boundary_Condition_Assignment'

        flags = self.windowFlags()
        self.setWindowFlags(flags & ~Qt.WindowContextHelpButtonHint)
        self.setWindowIcon(icon)
        self.setWindowTitle(title)
        self.type_filter_model = QSortFilterProxyModel()
        self.valid_filter_model = QSortFilterProxyModel()
        self.name_filter_model = QSortFilterProxyModel()
        self.button_delegate = PushButtonDelegate(self)
        # type dict of weir...id dict of list of index values into name and bc tables
        self.indexes_dict = {
            'Weir': {},  # {0: [0, 0]},
            'Flap Gate': {},
            'Sluice Gate': {}
        }
        self.data_model = QxPandasTableModel(self._set_structures(data, id_map))
        self.type_filter = ['Any', 'Weir', 'Flap Gate', 'Sluice Gate']
        self.valid_filter = ['Any', 'Valid', 'Invalid']
        self.DELETE_COL = 0
        self.VALID_COL = 1
        self.TYPE_COL = 2
        self.NAME_COL = 3
        self._setup()
        self.data = data

    def _setup(self):
        # QDialogButtonBox with Ok and Cancel buttons
        self.ui.buttonBox.helpRequested.connect(self.help_requested)
        self.type_filter_model.setSourceModel(self.data_model)
        self.valid_filter_model.setSourceModel(self.type_filter_model)
        self.name_filter_model.setSourceModel(self.valid_filter_model)
        self.type_filter_model.setFilterKeyColumn(self.TYPE_COL)
        self.valid_filter_model.setFilterKeyColumn(self.VALID_COL)
        self.name_filter_model.setFilterKeyColumn(self.NAME_COL)
        self.ui.view_all_table.setModel(self.name_filter_model)
        self.ui.type_combo.addItems(self.type_filter)
        self.ui.valid_combo.addItems(self.valid_filter)
        self.ui.type_combo.currentIndexChanged[int].connect(self.filter_type_changed)
        self.ui.valid_combo.currentIndexChanged[int].connect(self.filter_valid_changed)
        self.ui.delete_all_invalid.clicked.connect(self.delete_all_invalid)
        self.ui.name_edit.textEdited[str].connect(self.filter_name_changed)
        self.ui.view_all_table.clicked.connect(self.table_clicked)
        self.ui.view_all_table.setSortingEnabled(True)
        self.ui.view_all_table.setItemDelegateForColumn(self.DELETE_COL, self.button_delegate)
        self.ui.view_all_table.verticalHeader().hide()
        self.ui.view_all_table.horizontalHeader().setSortIndicator(self.NAME_COL, Qt.AscendingOrder)
        self.ui.view_all_table.horizontalHeader().setSortIndicator(self.TYPE_COL, Qt.AscendingOrder)
        self.ui.view_all_table.setEditTriggers(QAbstractItemView.NoEditTriggers)

        total_length = 0
        self.ui.view_all_table.resizeColumnsToContents()
        for i in range(self.ui.view_all_table.horizontalHeader().count()):
            total_length += self.ui.view_all_table.horizontalHeader().sectionSize(i)
        self.ui.view_all_table.setMinimumWidth(total_length)

        self.adjustSize()
        self.resize(self.size().width() + 6, self.size().height())

    def _set_structures(self, data: StructuresIO, id_map: dict):
        """Builds rows in the table for the given structure type.

        Args:
            data (:obj:`StructuresIO`): A structures IO class to hold the data.
            id_map (:obj:`dict` of int): A dictionary with component id keys and attribute id values.

        Returns:
            A list of values. The rows of data to add to the table.
        """
        # self.comp_to_xms[self.cov_uuid][entity_type_enum][comp_id].append(xms_id) dict is entity and to the right
        # data has adhparam and names, id_map has component_id keys and attribute id values
        # match adhparam weir... numbers to names
        # match adhparam weir... component IDs to id_map attribute IDs
        # build descriptions and determine validity
        # build rows

        rows = []
        rows.extend(
            self.build_table_rows(
                'Weir', 'WRS_NUMBER', 'WS_UPSTREAM', 'WS_DOWNSTREAM', data.weir_names, data.bc.weirs, id_map
            )
        )
        rows.extend(
            self.build_table_rows(
                'Flap Gate', 'FGT_NUMBER', 'FS_UPSTREAM', 'FS_DOWNSTREAM', data.flap_names, data.bc.flap_gates, id_map
            )
        )
        rows.extend(
            self.build_table_rows(
                'Sluice Gate', 'SLS_NUMBER', 'SS_UPSTREAM', 'SS_DOWNSTREAM', data.sluice_names, data.bc.sluice_gates,
                id_map
            )
        )

        df = pd.DataFrame(
            rows,
            columns=[
                'Delete', 'Valid', 'Type', 'Name', 'Point ID\nUpstream', 'Point ID\nDownstream', 'Arc ID\nUpstream',
                'Arc ID\nDownstream'
            ]
        )
        df.index += 1
        return df

    def build_table_rows(self, type, type_number, arc_up_str, arc_down_str, name_df, bc_df, id_map):
        """Builds rows in the table for the given structure type.

        Args:
            type (str): A string of the structure type. Example 'Weir'
            type_number (str): A string for the structure id number. Example 'WRS_NUMBER'
            arc_up_str (str): A string for the structure arc upstream. Example 'WS_UPSTREAM'
            arc_down_str (str): A string for the structure arc downstream. Example 'WS_DOWNSTREAM'
            name_df (:obj:`pd.DataFrame`): A dataframe containing the structure names.
            bc_df (:obj:`pd.DataFrame`): A dataframe containing the structure values.
            id_map (:obj:`dict` of int): A dictionary with component id keys and attribute id values.

        Returns:
            A list of values. The rows of data to add to the table.
        """
        rows = []
        for idx, row_data in name_df.iterrows():
            wid = int(row_data.ID)
            self.indexes_dict[type][wid] = [int(idx), 0]  # get second index later
        for idx, row_data in bc_df.iterrows():
            wid = int(getattr(row_data, type_number))
            self.indexes_dict[type][wid][1] = int(idx)  # set second index
        for wid in self.indexes_dict[type].keys():
            name = name_df.at[self.indexes_dict[type][wid][0], 'NAME']
            bc_index = self.indexes_dict[type][wid][1]
            pt_up_id = int(bc_df.at[bc_index, 'S_UPSTREAM'])
            pt_down_id = int(bc_df.at[bc_index, 'S_DOWNSTREAM'])
            arc_up_id = int(bc_df.at[bc_index, arc_up_str])
            arc_down_id = int(bc_df.at[bc_index, arc_down_str])
            pt_up, valid_1 = self.get_attribute_id_string(id_map[TargetType.point], pt_up_id)
            pt_down, valid_2 = self.get_attribute_id_string(id_map[TargetType.point], pt_down_id)
            arc_up, valid_3 = self.get_attribute_id_string(id_map[TargetType.arc], arc_up_id)
            arc_down, valid_4 = self.get_attribute_id_string(id_map[TargetType.arc], arc_down_id)
            valid = 'Invalid'
            if valid_1 and valid_2 and valid_3 and valid_4:
                valid = 'Valid'
            row = [wid, valid, type, name, pt_up, pt_down, arc_up, arc_down]
            rows.append(row)
        return rows

    @staticmethod
    def get_attribute_id_string(id_map, comp_id):
        """Gets a string representing attribute ids to use in the table.

        Args:
            id_map (:obj:`dict` of int): A dictionary with component id keys and attribute id values.
            comp_id (int): The component id.

        Returns:
            A tuple of str, bool. The string shows the attribute ids of the structure. True if valid for the structure.
        """
        att_str = ''
        is_valid = False
        if comp_id in id_map:
            is_valid = len(id_map[comp_id]) == 1
            for att_id in id_map[comp_id]:
                att_str += f'{att_id} '
        else:
            att_str = '<deleted>'
        return att_str, is_valid

    def filter_type_changed(self, idx):
        """Happens when the structure type index changes.

        Args:
            idx (int): The index into the combobox for structure type.
        """
        if idx == 0:
            self.type_filter_model.setFilterFixedString('')
        else:
            self.type_filter_model.setFilterFixedString(self.type_filter[idx])

    def filter_valid_changed(self, idx):
        """Happens when the valid state index changes.

        Args:
            idx (int): The index into the combobox for valid state.
        """
        if idx == 0:
            self.valid_filter_model.setFilterFixedString('')
        else:
            self.valid_filter_model.setFilterFixedString(self.valid_filter[idx])

    def filter_name_changed(self, name):
        """Happens when the name search text changes.

        Args:
            name (str): The text of the name search.
        """
        self.name_filter_model.setFilterWildcard(name)

    def delete_all_invalid(self):
        """Called when the Delete all invalid button is clicked. Deletes all of the invalid structures."""
        for row in range(self.data_model.rowCount() - 1, -1, -1):
            val = self.data_model.data(self.data_model.index(row, self.VALID_COL))
            if val == 'Invalid':
                self.data_model.removeRows(row, 1)

    def table_clicked(self, model_index):
        """Handles click events for the table. Deletes a structure if the click was in the delete column.

        Args:
            model_index (:obj:`PySide2.QModelIndex`): The model index of the click event.
        """
        if model_index.isValid() and model_index.column() == self.DELETE_COL:
            # translate the model_index to the data_model.
            model_index = self.name_filter_model.mapToSource(model_index)
            model_index = self.valid_filter_model.mapToSource(model_index)
            model_index = self.type_filter_model.mapToSource(model_index)
            row = model_index.row()
            self.data_model.removeRows(row, 1)

    def _remove_deleted(self, type_number, name_df, bc_df, keep_ids):
        """Removes the deleted structures from the the dataframes.

        Args:
            type_number (str): The structure number id. Example 'WRS_NUMBER'.
            name_df (:obj:`pd.DataFrame`): The structure name dataframe.
            bc_df (:obj:`pd.DataFrame`): The structure boundary condition dataframe.
            keep_ids (:obj:`list` of int): The list if valid structure ids.
        """
        name_drop_list = []
        bc_drop_list = []
        for idx, row_data in name_df.iterrows():
            wid = int(row_data.ID)
            if wid not in keep_ids:
                name_drop_list.append(idx)
        if name_drop_list:
            name_df.drop(name_drop_list, axis=0, inplace=True)
        for idx, row_data in bc_df.iterrows():
            wid = int(getattr(row_data, type_number))
            if wid not in keep_ids:
                bc_drop_list.append(idx)
        if bc_drop_list:
            bc_df.drop(bc_drop_list, axis=0, inplace=True)

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def accept(self):
        """Called when the Ok button is clicked. Removes deleted structures.

        Returns:
            QDialog.DialogCode
        """
        found_dict = {'Weir': [], 'Flap Gate': [], 'Sluice Gate': []}
        for row in range(self.data_model.rowCount()):
            structure_id = int(self.data_model.data(self.data_model.index(row, self.DELETE_COL)))
            structure_type = self.data_model.data(self.data_model.index(row, self.TYPE_COL))
            found_dict[structure_type].append(structure_id)
        self._remove_deleted('WRS_NUMBER', self.data.weir_names, self.data.bc.weirs, found_dict['Weir'])
        self._remove_deleted('FGT_NUMBER', self.data.flap_names, self.data.bc.flap_gates, found_dict['Flap Gate'])
        self._remove_deleted('SLS_NUMBER', self.data.sluice_names, self.data.bc.sluice_gates, found_dict['Sluice Gate'])
        return QDialog.accept(self)

    def reject(self):
        """Called when the Cancel button is clicked."""
        return QDialog.reject(self)
