"""Module for the SimComponent class."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"
__all__ = ['AeolisSimComponent']

# 1. Standard Python modules
from functools import cached_property
from pathlib import Path
from typing import Callable

# 2. Third party modules
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.components.bases.component_with_menus_base import MessagesAndRequests
from xms.constraint import GridType
from xms.gmi.component_bases.sim_component_base import SimComponentBase
from xms.gmi.data.generic_model import Section
from xms.gmi.gui.section_dialog import SectionDialog

# 4. Local modules
from xms.aeolis.data.model import get_model
from xms.aeolis.data.sim_base_data import SimData
from xms.aeolis.dmi.xms_data import XmsData


class AeolisSimComponent(SimComponentBase):
    """A hidden Dynamic Model Interface (DMI) component for the AeoLiS model simulation."""
    def __init__(self, main_file: str | Path):
        """
        Initialize the component.

        Args:
             main_file: Path to the component's data file.
        """
        super().__init__(main_file)
        self.tree_commands.insert(-1, ('Processes...', self.open_processes_dialog))
        self._section_dialog_keyword_args['enable_unchecked_groups'] = True
        self._section_dialog_keyword_args['hide_checkboxes'] = True

    def _get_global_parameter_section(self) -> Section:
        return get_model().global_parameters

    def item_linked(self, query: Query, linked_uuid: str, unique_name: str, lock_state: bool, parent: QWidget):
        """
        Check whether an item being linked is compatible with the simulation.

        Args:
            query: Interprocess communication object.
            linked_uuid: UUID of the item that was just linked.
            unique_name: Unique name of the item that was just linked.
            lock_state: Whether the component is locked for editing.
            parent: Parent widget to use for displaying dialogs.
        """
        data = XmsData(query)
        handle_item_linked(data, unique_name, lambda message: self.messages.append(('ERROR', message)))

    @cached_property
    def data(self) -> SimData:
        """The component's data manager."""
        return SimData(self.main_file)

    def open_processes_dialog(self, query: Query, _params: list[dict], parent: QWidget) -> MessagesAndRequests:
        """
        Run the model control dialog.

        Args:
            query: Interprocess communication object.
            _params: Ignored.
            parent: Parent widget for the dialog.
        """
        self._query = query
        section = get_model().model_parameters
        values = self.data.process_values
        section.restore_values(values)

        dlg = SectionDialog(
            parent=parent,
            section=section,
            dlg_name=f'{self.module_name}.process_control',
            window_title='Process control',
        )
        if dlg.exec():
            values = dlg.section.extract_values()
            self.data.process_values = values
            self.data.commit()

        return [], []


def handle_item_linked(data: XmsData, unique_name, report_error: Callable[[str], None]):
    if unique_name != 'UGrid':
        return  # Only need to check UGrids.

    ugrid = data.linked_grid
    if ugrid.grid_type == GridType.rectilinear_2d:
        return  # This is the type we want.

    data.unlink_grid()
    report_error('Linked UGrid must have the rectilinear 2D constraint')
