"""The help pane in the Structure properties dialog."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtWidgets import (
    QCheckBox, QComboBox, QGroupBox, QHBoxLayout, QLabel, QLineEdit, QPushButton, QVBoxLayout
)

# 3. Aquaveo modules
from xms.guipy.validators.qx_int_validator import QxIntValidator

# 4. Local modules


class StructureCulvertTab:
    """The profiles tab in the structure dialog."""
    def __init__(self, parent, widgets):
        """Initializes the class.

        Args:
            parent (:obj:`QWidget`): Parent dialog
            widgets (:obj:`dict`): Dictionary of widgets
        """
        self._parent = parent
        self._widgets = widgets
        self._add_widgets()

    def _add_widgets(self):
        """Set up the UI."""
        vlayout = self._widgets['tab3_scroll_area_vert_layout']
        dlg = self._parent

        # option to assign properties to individual arcs
        self._widgets['tog_culvert_assign_arcs'] = QCheckBox('Assign properties to individual arcs')
        vlayout.addWidget(self._widgets['tog_culvert_assign_arcs'])
        if dlg.data.data_dict['specify_culvert_arc_properties']:
            self._widgets['tog_culvert_assign_arcs'].setChecked(True)
        self._widgets['tog_culvert_assign_arcs'].stateChanged.connect(self.on_assign_arcs)
        self._widgets['grp_culvert_arc_props'] = QGroupBox('Arc properties')
        vlayout.addWidget(self._widgets['grp_culvert_arc_props'])
        self._widgets['vlayout_culvert_arc_props'] = QVBoxLayout()
        vlayout_grp = self._widgets['vlayout_culvert_arc_props']
        self._widgets['grp_culvert_arc_props'].setLayout(vlayout_grp)
        self._parent.culvert_arc_props.add_widgets(vlayout_grp)

        vlayout = self._widgets['tab3_scroll_area_vert_layout']
        # culvert width
        dlg.add_floating_point_widget(
            'culvert_width', 'Culvert length (upstream to downstream):', dlg.data.data_dict['bridge_width'], vlayout
        )
        # manning's n for underside of culvert
        dlg.add_floating_point_widget(
            'culvert_manning', 'Culvert ceiling manning roughness coefficient:',
            dlg.data.data_dict["bridge_ceiling_roughness"], vlayout
        )

        # toggle to add 3d ugrid bridge to SMS on OK
        self._widgets['tog_culvert_add_ugrid'] = QCheckBox('Add 3D Culvert UGrid to SMS on OK')
        vlayout.addWidget(self._widgets['tog_culvert_add_ugrid'])
        # culvert type
        self._widgets['txt_culvert_type'] = QLabel('Culvert type:')
        vlayout.addWidget(self._widgets['txt_culvert_type'])
        culvert_types = ['Circular', 'Box']
        self._widgets['cbx_culvert_type'] = QComboBox()
        self._widgets['cbx_culvert_type'].addItems(culvert_types)
        self._widgets['cbx_culvert_type'].setCurrentIndex(0)
        if dlg.data.data_dict['culvert_type'] in culvert_types:
            self._widgets['cbx_culvert_type'].setCurrentText(dlg.data.data_dict['culvert_type'])
        vlayout.addWidget(self._widgets['cbx_culvert_type'])
        self._widgets['cbx_culvert_type'].currentIndexChanged.connect(self._on_change_culvert_type)
        # culvert diameter for circular
        dlg.add_floating_point_widget(
            'culvert_diameter', 'Circular culvert diameter:', dlg.data.data_dict['culvert_diameter'], vlayout
        )
        # box rise, span
        dlg.add_floating_point_widget('culvert_rise', 'Box culvert rise:', dlg.data.data_dict['culvert_rise'], vlayout)
        dlg.add_floating_point_widget('culvert_span', 'Box culvert span:', dlg.data.data_dict['culvert_span'], vlayout)
        # upstream/downstream invert elevations
        positive_validator = False
        dlg.add_floating_point_widget(
            'culvert_up_invert',
            'Upstream culvert invert elevation:',
            dlg.data.data_dict['culvert_up_invert'],
            vlayout,
            set_positive_validator=positive_validator
        )
        dlg.add_floating_point_widget(
            'culvert_dn_invert',
            'Downstream culvert invert elevation:',
            dlg.data.data_dict['culvert_dn_invert'],
            vlayout,
            set_positive_validator=positive_validator
        )
        # Embedment depths
        dlg.add_floating_point_widget(
            'culvert_embed_depth', 'Culvert embedment depth:', dlg.data.data_dict['culvert_embed_depth'], vlayout
        )
        # number of barrels
        self._widgets['txt_culvert_num_barrels'] = QLabel("Number of culvert barrels:")
        vlayout.addWidget(self._widgets['txt_culvert_num_barrels'])
        self._widgets['edt_culvert_num_barrels'] = QLineEdit(f'{int(dlg.data.data_dict["culvert_num_barrels"])}')
        validator = QxIntValidator(parent=dlg)
        validator.setBottom(1)
        self._widgets['edt_culvert_num_barrels'].setValidator(validator)
        vlayout.addWidget(self._widgets['edt_culvert_num_barrels'])
        label = 'Wall width between culvert barrels:'
        dlg.add_floating_point_widget('culvert_wall_width', label, dlg.data.data_dict['culvert_wall_width'], vlayout)

        # 2d mesh generation
        self._widgets['grp_culvert_mesh'] = QGroupBox('Culvert 2D mesh generation')
        vlayout.addWidget(self._widgets['grp_culvert_mesh'])
        self._widgets['vlayout_culvert_mesh'] = QVBoxLayout()
        vlayout_grp = self._widgets['vlayout_culvert_mesh']
        self._widgets['grp_culvert_mesh'].setLayout(vlayout_grp)
        # bridge wrapping width
        dlg.add_floating_point_widget(
            'culvert_wrap_width', 'Culvert wrapping width:', dlg.data.data_dict['bridge_wrapping_width'], vlayout_grp
        )
        # toggle for "has abutments"
        self._widgets['tog_culvert_abutments'] = QCheckBox('Has abutments')
        checked = True if dlg.data.data_dict['has_abutments'] else False
        self._widgets['tog_culvert_abutments'].setChecked(checked)
        vlayout_grp.addWidget(self._widgets['tog_culvert_abutments'])
        # wall number of side elements
        self._widgets['txt_culvert_wall_num_side'] = QLabel("Culvert number of side elements:")
        vlayout_grp.addWidget(self._widgets['txt_culvert_wall_num_side'])
        self._widgets['edt_culvert_num_side'] = QLineEdit(f'{int(dlg.data.data_dict["pier_wall_num_side_elem"])}')
        validator = QxIntValidator(parent=dlg)
        validator.setBottom(1)
        self._widgets['edt_culvert_num_side'].setValidator(validator)
        vlayout_grp.addWidget(self._widgets['edt_culvert_num_side'])
        # number of elements across culvert barrel
        self._widgets['txt_culvert_num_elem_barrel'] = QLabel("Culvert number of elements across barrel:")
        vlayout_grp.addWidget(self._widgets['txt_culvert_num_elem_barrel'])
        val = f'{int(dlg.data.data_dict["culvert_num_seg_barrel"])}'
        self._widgets['edt_culvert_num_elem_barrel'] = QLineEdit(val)
        validator = QxIntValidator(parent=dlg)
        validator.setBottom(3)
        self._widgets['edt_culvert_num_elem_barrel'].setValidator(validator)
        vlayout_grp.addWidget(self._widgets['edt_culvert_num_elem_barrel'])
        # button elevation raster
        self._widgets['culvert_elev_raster_layout'] = QHBoxLayout()
        vlayout_grp.addLayout(self._widgets['culvert_elev_raster_layout'])
        self._widgets['btn_culvert_elev_raster'] = QPushButton('Select elevation raster...')
        self._widgets['btn_culvert_elev_raster'].clicked.connect(dlg.on_btn_elev_raster)
        self._widgets['culvert_elev_raster_layout'].addWidget(self._widgets['btn_culvert_elev_raster'])
        self._widgets['txt_culvert_elev_raster'] = QLabel('Not selected')
        self._widgets['culvert_elev_raster_layout'].addWidget(self._widgets['txt_culvert_elev_raster'])
        self._widgets['culvert_elev_raster_layout'].addStretch()
        # preview mesh button
        self._widgets['culvert_preview_layout'] = QHBoxLayout()
        vlayout_grp.addLayout(self._widgets['culvert_preview_layout'])
        self._widgets['btn_preview_culvert_mesh'] = QPushButton('Preview culvert mesh...')
        self._widgets['btn_preview_culvert_mesh'].clicked.connect(dlg.on_btn_preview_mesh)
        self._widgets['culvert_preview_layout'].addWidget(self._widgets['btn_preview_culvert_mesh'])
        # preview mesh browser button
        self._widgets['btn_preview_culvert_mesh_browser'] = QPushButton('Preview culvert mesh (compatibility mode)...')
        self._widgets['btn_preview_culvert_mesh_browser'].clicked.connect(dlg.on_btn_preview_mesh_browser)
        self._widgets['culvert_preview_layout'].addWidget(self._widgets['btn_preview_culvert_mesh_browser'])
        self._widgets['culvert_preview_layout'].addStretch()
        # toggle to add coverage to SMS on OK
        self._widgets['tog_add_culvert_coverage'] = QCheckBox('Add "culvert mesh" coverage to SMS on OK')
        vlayout_grp.addWidget(self._widgets['tog_add_culvert_coverage'])
        # toggle to add footprint mesh to SMS on OK
        self._widgets['tog_add_culvert_footprint_ugrid'] = QCheckBox('Add "culvert mesh" 2D mesh to SMS on OK')
        vlayout_grp.addWidget(self._widgets['tog_add_culvert_footprint_ugrid'])
        vlayout.addStretch()
        self._on_change_culvert_type()
        self.on_assign_arcs()

    def _on_change_culvert_type(self):
        """Change to the weir type combobox."""
        culvert_type = self._widgets['cbx_culvert_type'].currentText()
        circular_widgets = ['txt_culvert_diameter', 'edt_culvert_diameter']
        box_widgets = ['txt_culvert_rise', 'edt_culvert_rise', 'txt_culvert_span', 'edt_culvert_span']
        circular = box = False
        if culvert_type == 'Circular':
            circular = True
        else:
            box = True
        for item in circular_widgets:
            self._widgets[item].setVisible(circular)
        for item in box_widgets:
            self._widgets[item].setVisible(box)

    def on_assign_arcs(self):
        """Called when the assign properties to individual arcs toggle checked."""
        vis = self._widgets['tog_culvert_assign_arcs'].isChecked()
        self._widgets['grp_culvert_arc_props'].setVisible(vis)
        if vis:
            self._widgets['tog_culvert_abutments'].setChecked(False)
        self._widgets['tog_culvert_abutments'].setVisible(not vis)
