"""The help pane in the Structure properties dialog."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import numpy as np
import pandas as pd
from PySide2.QtCore import QModelIndex

# 3. Aquaveo modules
from xms.guipy.widgets.table_with_tool_bar import TableWithToolBar
from xms.tool_core.table_definition import ChoicesColumnType, StringColumnType, TableDefinition

# 4. Local modules


class StructureSummaryTab:
    """The profiles tab in the structure dialog."""
    def __init__(self, parent, widgets, plot_data):
        """Initializes the class.

        Args:
            parent (:obj:`QWidget`): Parent dialog
            widgets (:obj:`dict`): Dictionary of widgets
            plot_data (:obj:`PlotsAndTableDataSrh`): Plot data
        """
        self._parent = parent
        self._parent_widgets = widgets
        self._widgets = {}
        self._plot_data = plot_data
        self._summary_df = None
        self._table_def = None
        self._add_widgets()
        self._set_readonly_columns()

    def _add_widgets(self):
        """Setup the UI."""
        vlayout = self._parent_widgets['tab1_scroll_area_vert_layout']

        self._get_summary_data()
        self._table_widget = TableWithToolBar(parent=self._parent)
        self._widgets['summary_table'] = self._table_widget
        self._table_widget.setup(self._table_def, self._summary_df)
        self._table_widget.ui.tool_bar.hide()
        self._table_widget.data_changed.connect(self._on_data_changed)
        vlayout.addWidget(self._table_widget)
        vlayout.addStretch()

    def _set_readonly_columns(self):
        """Set the columns that are read only."""
        read_only = set(range(len(self._summary_df.columns)))
        read_only.remove(1)
        self._table_widget.ui.table.model().set_read_only_columns(read_only)

    def _get_summary_data(self):
        """Get the summary data."""
        plt_data = self._plot_data
        df_dict = _build_dict_from_summary_data(plt_data.simulations, plt_data.plot_data_files, plt_data.file_data)
        cols = list(df_dict.keys())
        self._summary_df = pd.DataFrame(df_dict)
        self._summary_df.index += 1
        table_cols = [
            StringColumnType(header='Simulation'),
            StringColumnType(header=cols[1], choices=2),
            ChoicesColumnType(header='Choices for column 2'),
        ]
        for col in cols[2:]:
            table_cols.append(StringColumnType(header=col))
        self._table_def = TableDefinition(table_cols)

    def _on_data_changed(self, top_left: QModelIndex, bottom_right: QModelIndex) -> None:
        df = self._table_widget.get_values()
        # update data for each row
        _update_dataframe_rows(df, self._plot_data)
        self._table_widget.set_values(df)
        self._set_readonly_columns()


def _build_dict_from_summary_data(sims, data_files, file_data):
    """Build a dictionary from the summary data.

    Args:
        sims (list[str]): List of simulation names
        data_files (dict[str, str]): Dictionary of data files
        file_data (dict[str, tuple[list[str], list[list[str]]]): Dictionary of file data
    """
    df_dict = {}
    cols = []
    for sim in sims:
        if sim not in data_files:
            continue
        data_file = data_files[sim]
        headings, data = file_data[data_file]
        if len(headings) > 2 and 'Q-Structure' not in headings[2]:
            continue  # skip this simulation
        if len(df_dict) == 0:
            cols = ['Simulation', headings[0], 'time_choices'] + [name for name in headings[1:]]
            df_dict = {name: [] for name in cols}
        df_dict['Simulation'].append(sim)
        df_dict[cols[1]].append(data[0][-1])
        df_dict[cols[2]].append([str(t) for t in data[0]])
        for j, name in enumerate(headings[1:]):
            df_dict[name].append(str(data[j + 1][-1]))
    if not cols:
        cols = [
            'Simulation', 'Time(hr)', 'time_choices', 'Q-Overtopping(cfs)', 'Q-Structure(cfs)', 'Q-Total(cfs)',
            'WSE_up(ft)', 'WSE_down(ft)', 'Elev_crest(ft)'
        ]
        df_dict = {name: [0.0] for name in cols}
        df_dict['Simulation'] = ['No simulations']
        df_dict['time_choices'] = [['0.0']]
    return df_dict


def _update_dataframe_rows(df, plot_data):
    """Update the rows in the dataframe.

    Args:
        df (pandas.DataFrame): The dataframe
        plot_data (:obj:`PlotsAndTableDataSrh`): Plot data
    """
    cols = df.columns
    sims = df[cols[0]].tolist()
    times = [float(t) for t in df[cols[1]].tolist()]
    for idx, sim in enumerate(sims):
        if sim not in plot_data.plot_data_files:
            continue
        data_file = plot_data.plot_data_files[sim]
        headings, data = plot_data.file_data[data_file]
        # get the index of the time value
        time_idx = np.where(data[0] == times[idx])[0][0]
        for col_idx, _col in enumerate(cols[3:]):
            df.iat[idx, col_idx + 3] = str(data[col_idx + 1][time_idx])
