"""Worker thread for the CMS-Flow exporter."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.components.display import windows_gui as win_gui
from xms.guipy.dialogs import xms_parent_dlg as xms_dlg
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackDlg, ProcessFeedbackThread
from xms.guipy.dialogs.xms_parent_dlg import ensure_qapplication_exists

# 4. Local modules
from xms.cmsflow._cmsflow_cmcards_export import CMSFlowCmcardsExporter
from xms.cmsflow._cmsflow_datasets_export import CMSFlowDatasetsExporter
from xms.cmsflow._cmsflow_mp_export import CMSFlowMpExporter
from xms.cmsflow._cmsflow_tel_export import CMSFlowTelExporter
from xms.cmsflow.dmi.xms_data import XmsData


class CmCardsExportWorkerThread(ProcessFeedbackThread):
    """Worker thread for exporting CMS-Flow model files."""
    def __init__(self, parent=None):
        """Construct the worker.

        Args:
            parent (QWidget): Parent of the QThread
        """
        super().__init__(parent=parent, do_work=self._do_work)
        self._logger = logging.getLogger('xms.cmsflow')

    def _do_work(self):
        """Thread runner that exports the fort.15."""
        # Export the .cmcards file
        try:
            self._logger.info('Exporting cmcards file')
            exporter = CMSFlowCmcardsExporter()
            exporter.export_cards()
            cov_mapper = exporter.get_cov_mapper()
        except Exception as e:
            self._logger.exception(f'Could not export CMS-Flow control file.:\n{e}')
            return  # No point in going on.
        q = exporter.query
        d = XmsData(q)

        # Export the _datasets.h5
        try:
            self._logger.info('Exporting spatial datasets file')
            exporter = CMSFlowDatasetsExporter()
            exporter.export_datasets(query=q)
        except Exception as e:
            self._logger.exception(f'Could not export CMS-Flow spatial datasets.:\n{e}')

        # Write the .mp boundary conditions file.
        try:
            self._logger.info('Exporting boundary conditions file')
            exporter = CMSFlowMpExporter(d, cov_mapper=cov_mapper)
            exporter.export()
        except Exception as e:
            self._logger.exception(f'Could not export CMS-Flow .mp boundary conditions file.:\n{e}')

        # Export the .tel geometry file.
        try:
            self._logger.info('Exporting geometry file')
            exporter = CMSFlowTelExporter()
            # Use the activity we computed when writing the .cmcards file.
            exporter.cell_activity = cov_mapper.get_cell_activity()
            exporter.write_tel_file(query=q)
        except Exception as e:
            logging.getLogger('xms.cmsflow').exception(f'Could not export CMS-Flow .tel geometry.:\n{e}')


def export_cmcards_with_feedback():
    """Export the CMS-Flow model with a feedback dialog."""
    ensure_qapplication_exists()

    # Parse parent HWNDs and icon path from commandline arguments.
    parent_hwnd, main_hwnd, _ = xms_dlg.parse_parent_window_command_args()
    win_cont = xms_dlg.get_parent_window_container(parent_hwnd)
    # Create the timer that keeps our Python dialog in the foreground of XMS.
    _ = win_gui.create_and_connect_raise_timer(main_hwnd, win_cont)  # Keep the timer in scope
    parent = win_cont

    worker = CmCardsExportWorkerThread(parent=parent)
    display_text = {
        'title': 'CMS-Flow Export Simulation',
        'working_prompt': 'Exporting CMS-Flow simulation files. Please wait...',
        'warning_prompt': 'Warning(s) encountered while exporting simulation. Review log output for more details.',
        'error_prompt': 'Error(s) encountered while exporting simulation. Review log output for more details.',
        'success_prompt': 'Successfully exported simulation',
        'note': '',
        'auto_load': 'Close this dialog automatically when exporting is finished.'
    }
    testing = XmEnv.xms_environ_running_tests() == 'TRUE'
    feedback_dlg = ProcessFeedbackDlg(display_text=display_text, logger_name='xms', worker=worker, parent=parent)
    feedback_dlg.testing = testing
    if feedback_dlg.exec() and not testing:  # User did not cancel, we are not testing
        if LogEchoQSignalStream.logged_error:  # If we have an error on simulation export, log the error for XMS
            XmEnv.report_export_error()
    else:  # User cancelled
        XmEnv.report_export_aborted()
    if main_hwnd is not None:
        win_gui.raise_main_xms_window(main_hwnd)  # Bring top-level Python window to foreground
