"""Module for the CulvertStructureWriter class."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"
__all__ = ['write_culvert_structures']

# 1. Standard Python modules
from functools import cached_property
import logging
from math import atan2, degrees
from typing import cast, Sequence, TextIO

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules
from xms.constraint import QuadtreeGrid2d
from xms.data_objects.parameters import Arc, Coverage
from xms.gmi.data_bases.coverage_base_data import CoverageBaseData
from xms.grid.ugrid import UGrid
from xms.guipy.data.target_type import TargetType
from xms.snap import SnapPoint

# 4. Local modules
from xms.cmsflow.data.model import get_model
from xms.cmsflow.file_io.card_writer import CardWriter


def write_culvert_structures(
    coverage: Coverage, data: CoverageBaseData, ugrid: QuadtreeGrid2d, logger: logging.Logger, wrote_header: bool,
    cards: TextIO
) -> bool:
    """
    Write all the data needed by culverts in a structure coverage.

    Args:
        coverage: Coverage containing geometry to write.
        data: Data manager for the coverage. Should have its component_id_map initialized.
        ugrid: The QuadTree to snap the coverage to.
        logger: Where to log any warnings or errors.
        wrote_header: Whether the structures header has already been written.
        cards: Where to write cards to. Typically obtained by calling `open(...)` on the *.cmcards file.

    Returns:
        Whether the `!Structures` header was written (either before or by calling this function).
    """
    writer = CulvertStructureWriter(coverage, data, ugrid, logger, wrote_header, cards)
    return writer.write()


class CulvertStructureWriter:
    """Class for writing all the data needed by culverts in a structure coverage."""
    def __init__(
        self, coverage: Coverage, data: CoverageBaseData, ugrid: QuadtreeGrid2d, logger: logging.Logger,
        wrote_header: bool, cards: TextIO
    ):
        """
        Initialize the writer.

        Args:
            coverage: Coverage containing geometry to write.
            data: Data manager for the coverage. Should have its component_id_map initialized.
            ugrid: The QuadTree to snap the coverage to.
            logger: Where to log any warnings or errors.
            cards: Where to write cards to. Typically obtained by calling `open(...)` on the *.cmcards file.
            wrote_header: Whether the structures header has already been written.
        """
        self._coverage = coverage
        self._data = data
        self._cogrid = ugrid
        self._ugrid = ugrid.ugrid
        self._cards = CardWriter(cards)
        self._logger = logger
        self._section = get_model().arc_parameters
        self._need_header = not wrote_header
        self._wrote_header = wrote_header

    @cached_property
    def _activity(self) -> Sequence[bool]:
        """
        The grid's cell activity.

        If the grid has no cell activity, then a sequence of True is returned (all cells active).
        """
        activity = self._cogrid.model_on_off_cells
        if not activity:
            activity = np.full(self._ugrid.cell_count, True, dtype=bool)
        return activity

    @cached_property
    def _elevations(self) -> Sequence[float]:
        """
        The grid's cell elevations.

        If the grid has no cell elevations, then a sequence of zeroes is returned.
        """
        elevations = self._cogrid.cell_elevations
        if not elevations:
            elevations = np.full(self._ugrid.cell_count, 0.0, dtype=float)
        return elevations

    def write(self) -> bool:
        """
        Write all the culvert data needed for the coverage.

        Returns:
            Whether the `!Structures` header was written (either before or by calling this function).
        """
        for arc in self._coverage.arcs:
            self._write_culvert(arc)

        return self._wrote_header

    def _ensure_header_written(self):
        """
        Ensure the header for the rubble mound structure section is written.

        Does nothing after the first time it was called.
        """
        if self._wrote_header:
            return

        self._logger.info('Writing culvert structures')
        self._wrote_header = True
        self._cards.write('!Structures', indent=0)

    def _write_culvert(self, arc: Arc):
        """
        Write a weir if necessary.

        The arc will not be written if it is invalid or not a culvert.

        Args:
            arc: The arc to write.
        """
        values = self._data.feature_values(TargetType.arc, feature_id=arc.id)
        self._section.restore_values(values)
        group = self._section.group('culvert')
        if not group.is_active:
            return

        cells = self._snapper.get_snapped_points([arc.start_node, arc.end_node])['id']
        if not self._check_cells(cells, arc.id):
            return

        self._ensure_header_written()

        self._cards.write('CULVERT_STRUCT_BEGIN', indent=0)

        value = ' '.join(str(cell + 1) for cell in cells)
        self._cards.write('CELLS', value)

        culvert_type = group.parameter('culvert_type').value
        culvert_type = 'CIRCLE' if culvert_type == 'Circular' else 'BOX'
        self._cards.write('TYPE', culvert_type)

        flap_gate = group.parameter('flap_gate').value
        flap_gate = 'ON' if flap_gate else 'OFF'
        self._cards.write('FLAP_GATE', flap_gate)

        if culvert_type == 'CIRCLE':
            radius = group.parameter('radius').value
            self._cards.write('RADIUS', radius)

        if culvert_type == 'BOX':
            width = group.parameter('width').value
            self._cards.write('WIDTH', width)

        if culvert_type == 'BOX':
            height = group.parameter('height').value
            self._cards.write('HEIGHT', height)

        length = group.parameter('length').value
        self._cards.write('LENGTH', length)

        friction = group.parameter('darcy_friction').value
        self._cards.write('DARCY_FRICTION_FACTOR', friction)

        mannings = group.parameter('manning_n').value
        self._cards.write('MANNINGS_COEFFICIENT', mannings)

        bay_elevation = self._elevations[cells[0]]
        bay_elevation = _float_to_string(bay_elevation)
        sea_elevation = self._elevations[cells[1]]
        sea_elevation = _float_to_string(sea_elevation)
        self._cards.write('INVERT_ELEVATIONS', bay_elevation, sea_elevation)

        entry_head_loss_bay = group.parameter('entry_head_loss_bay').value
        self._cards.write('ENTRY_HEAD_LOSS_BAY', entry_head_loss_bay)

        entry_head_loss_sea = group.parameter('entry_head_loss_sea').value
        self._cards.write('ENTRY_HEAD_LOSS_SEA', entry_head_loss_sea)

        exit_head_loss_bay = group.parameter('exit_head_loss_bay').value
        self._cards.write('EXIT_HEAD_LOSS_BAY', exit_head_loss_bay)

        exit_head_loss_sea = group.parameter('exit_head_loss_sea').value
        self._cards.write('EXIT_HEAD_LOSS_SEA', exit_head_loss_sea)

        bay, sea = self._compute_angles(cells)
        self._cards.write('OUTFLOW_ANGLES', bay, sea)

        self._cards.write('CULVERT_STRUCT_END', indent=0)

    @cached_property
    def _snapper(self) -> SnapPoint:
        """Set up the arc snapper."""
        snapper = SnapPoint()
        ugrid = cast(UGrid, self._cogrid)
        snapper.set_grid(ugrid, target_cells=True)
        return snapper

    def _check_cells(self, cells: list[int], feature_id: int) -> bool:
        """
        Check if a list of cells is a valid weir.

        Logs an error if they are not a valid weir.

        Args:
            cells: Cells to check.
            feature_id: Feature ID of the arc the cells represent. Used for error reporting.

        Returns:
            Whether the arc was good.
        """
        if cells[0] == cells[1]:
            self._logger.warning(f'Skipping culvert arc {feature_id}, nodes snapped to same cell.')
            return False

        # Empty indicates no activity, so all cells active.
        if not all(self._activity[cell] for cell in cells):
            self._logger.warning(f'Skipping culvert arc {feature_id}, nodes are on inactive cells.')
            return False

        return True

    def _compute_angles(self, cells: tuple[int, int]) -> tuple[str, str]:
        """Compute the outflow angles needed to write a culvert."""
        _exists, start = self._ugrid.get_cell_centroid(cells[0])
        _exists, end = self._ugrid.get_cell_centroid(cells[1])

        dx = end[0] - start[0]
        dy = end[1] - start[1]
        world_angle = degrees(atan2(dy, dx))
        if world_angle < 0.0:
            world_angle += 360.0

        grid_angle = world_angle - self._cogrid.angle
        if grid_angle < 0.0:
            grid_angle += 360.0

        opposite_angle = grid_angle + 180.0
        if opposite_angle > 360.0:
            opposite_angle -= 360.0

        return _angle_to_string(opposite_angle), _angle_to_string(grid_angle)


def _angle_to_string(angle: float) -> str:
    """Convert an angle to a string."""
    s = _float_to_string(angle)
    if s == '360.0':
        s = '0.0'
    return s


def _float_to_string(f: float) -> str:
    """Convert a float to a string with up to 6 digits after the decimal point."""
    s = f'{f:.6f}'
    s = s.rstrip('0')  # s always ends in .######, so stripping 0s won't erase magnitude
    if s[-1] == '.':
        s += '0'
    return s
