"""Class to manage display for StructuresDomponent."""

__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import shutil

# 2. Third party modules
from PySide2.QtGui import QColor
import xarray as xr

# 3. Aquaveo modules
from xms.components.display.display_options_io import (
    read_display_option_ids, read_display_options_from_json, write_display_option_ids, write_display_options_to_json
)
from xms.components.display.xms_display_message import XmsDisplayMessage
from xms.core.filesystem import filesystem
from xms.guipy.data.category_display_option import CategoryDisplayOption
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.data.polygon_texture import PolygonOptions
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.cmswave.data.structures_data import StructuresData
from xms.cmswave.gui.structures_parent_dialog import StructuresParentDialog

STRUCT_INITIAL_ATT_ID_FILE = 'initial_struct_polys.ids'
STRUCT_INITIAL_COMP_ID_FILE = 'initial_struct_comp.ids'


class StructuresComponentDisplay:
    """Class to manage display for StructuresComponent."""
    def __init__(self, comp):
        """Initializes the base component class.

        Args:
            comp (StructuresComponent): The component to manage
        """
        self._comp = comp

    def ensure_display_options_exist(self):
        """Ensure the component display options json file exists."""
        comp_dir = os.path.dirname(self._comp.main_file)
        self._comp.disp_opts_files[0] = os.path.join(comp_dir, 'structures_display_options.json')
        if not os.path.exists(self._comp.disp_opts_files[0]):
            # Copy over the unassigned structure category id file.
            default_id_file = os.path.join(
                os.path.dirname(os.path.dirname(__file__)), 'gui', 'resources', 'default_data',
                f'struct_{StructuresData.UNASSIGNED_STRUCT}.structid'
            )
            filesystem.copyfile(default_id_file, os.path.join(comp_dir, os.path.basename(default_id_file)))
            # Read the default display options, and save ourselves a copy with a randomized UUID.
            categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
            default_file = os.path.join(
                os.path.dirname(os.path.dirname(__file__)), 'gui', 'resources', 'default_data',
                'structure_default_display_options.json'
            )
            json_dict = read_display_options_from_json(default_file)
            categories.from_dict(json_dict)
            json_dict['comp_uuid'] = os.path.basename(os.path.dirname(self._comp.main_file))
            write_display_options_to_json(self._comp.disp_opts_files[0], categories)
            # Save our display list UUID to the main file
            self._comp.data.info.attrs['display_uuid'] = categories.uuid
            self._comp.data.commit()

    def initialize_display(self, query):
        """Initialize the XMS component display list.

        Args:
            query (:obj:`Query`): The XMS interprocess communication object.

        Returns:
            (:obj:`tuple([], [])`): Empty message and ActionRequest lists
        """
        self._comp.cov_uuid = query.parent_item_uuid()
        if not self._comp.cov_uuid:
            return [('ERROR', 'Could not get structure coverage UUID.')], []

        initial_att_file = os.path.join(os.path.dirname(self._comp.main_file), STRUCT_INITIAL_ATT_ID_FILE)
        if os.path.isfile(initial_att_file):  # Came from a model native read, initialize the component ids.
            att_ids = read_display_option_ids(initial_att_file)
            initial_comp_file = os.path.join(os.path.dirname(self._comp.main_file), STRUCT_INITIAL_COMP_ID_FILE)
            comp_ids = read_display_option_ids(initial_comp_file)
            filesystem.removefile(initial_att_file)
            filesystem.removefile(initial_comp_file)
            for att_id, comp_id in zip(att_ids, comp_ids):
                self._comp.update_component_id(TargetType.polygon, att_id, comp_id)

        self.update_display_options_file()
        self._comp.data.info.attrs['cov_uuid'] = self._comp.cov_uuid
        self._comp.data.commit()
        # Send the default structure list to XMS.
        self._comp.display_option_list.append(
            XmsDisplayMessage(file=self._comp.disp_opts_files[0], edit_uuid=self._comp.cov_uuid)
        )
        return [], []

    def update_display_options_file(self):
        """Update the XMS display options JSON file to match what we have in memory."""
        category_list = CategoryDisplayOptionList()
        category_list.target_type = TargetType.polygon
        category_list.uuid = str(self._comp.data.info.attrs['display_uuid'])
        category_list.comp_uuid = self._comp.uuid
        df = self._comp.data.structures.to_dataframe()
        for struct_row in range(0, len(df.index)):
            category = CategoryDisplayOption()
            category.id = int(df.iloc[struct_row]['id'])
            category.description = df.iloc[struct_row]['name']
            category.file = f'struct_{category.id}.structid'
            category.options = PolygonOptions()
            style = df.iloc[struct_row]['color'].split()
            category.options.color = QColor(int(style[0]), int(style[1]), int(style[2]), 255)
            category.options.texture = int(style[3])
            # Make bathymetry modification structure the default category.
            if category.id == 0:  # Should always have 0 as "structure id"
                category.is_unassigned_category = True
            category_list.categories.append(category)
        write_display_options_to_json(self._comp.disp_opts_files[0], category_list)

    def update_display_id_files(self, old_ids, new_ids):
        """Update the display files.

        Args:
            old_ids (:obj:`list[int]`): list of ids before editing structures
            new_ids (:obj:`list[int]`): list of current structure ids

        Returns:
            (:obj:`list[int]`): deleted ids
        """
        deleted_ids = old_ids
        path = os.path.dirname(self._comp.main_file)
        for struct_id in new_ids:
            if struct_id >= 0:
                id_file = f'struct_{struct_id}.structid'
                filename = os.path.join(path, id_file)
                write_display_option_ids(filename, [struct_id])
            if struct_id in deleted_ids:
                deleted_ids.remove(struct_id)

        for struct_id in deleted_ids:
            id_file = f'struct_{struct_id}.structid'
            filename = os.path.join(path, id_file)
            filesystem.removefile(filename)
        return deleted_ids

    def update_structures_list(self, query, old_ids):
        """Update the display options JSON file and display id files after editing the structures list.

        Args:
            query (:obj:`Query`): xmsapi interprocess communication object
            old_ids (:obj:`list[int]`): The old structure ids
        """
        # Check for structures removed from the list.
        new_ids = list(self._comp.data.structures['id'].data.astype(int))
        deleted_ids = self.update_display_id_files(old_ids, new_ids)
        self.unassign_structures(query, deleted_ids)
        # Write the display options file.
        self.update_display_options_file()
        self._comp.display_option_list.append(
            XmsDisplayMessage(file=self._comp.disp_opts_files[0], edit_uuid=self._comp.cov_uuid)
        )

    def unassign_structures(self, query, delete_ids):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query (:obj:`Query`): Object for communicating with XMS
            delete_ids (:obj:`list[int]`): List of the deleted structure ids.

        Returns:
            Empty message and ActionRequest lists
        """
        query.load_component_ids(self._comp, polygons=True)
        if self._comp.cov_uuid in self._comp.comp_to_xms and \
                TargetType.polygon in self._comp.comp_to_xms[self._comp.cov_uuid]:
            poly_map = self._comp.comp_to_xms[self._comp.cov_uuid][TargetType.polygon]
            for struct in delete_ids:
                # For some reason these are xarray DataArrays when coming from the tree item menu command. Didn't
                # bother to find out why. Just fix it here.
                if isinstance(struct, xr.DataArray):
                    struct = struct.item()
                if struct in poly_map:
                    for att_id in poly_map[struct]:
                        self._comp.update_component_id(TargetType.polygon, att_id, StructuresData.UNASSIGNED_STRUCT)

    def open_assign_poly_structures(self, query, params, parent):
        """Opens the Assign Structures dialog and saves component data state on OK.

        Args:
            query (:obj:`Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Contains selection map and component id files.
            parent (:obj:`QWidget`): The Qt parent window container.

        Returns:
            (:obj:`tuple(list, list)`):

                messages (:obj:`list[tuple(str, str)]`): List of tuples with the first element of the tuple being the
                message level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.

                action_requests (:obj:`list[ActionRequest]`): List of actions for XMS to perform.
        """
        # Get the XMS attribute ids of the selected polygons (if any)
        poly_ids = params.get('selection', [])
        num_polys = len(poly_ids)
        if num_polys == 0:
            return [('INFO', 'No polygons selected. Select one or more polygons to assign structures.')], []

        # Get the component id (structure id) map of the selected polygons (if any).
        current_struct_id = StructuresData.UNASSIGNED_STRUCT
        id_files = params.get('id_files', [])
        if id_files is not None and len(id_files) > 1:
            self._comp.load_coverage_component_id_map({'POLYGON': (id_files[0], id_files[1])})
            # Delete the id dumped by xms files.
            if poly_ids:
                # First if multiple selected
                current_struct_id = self._comp.get_comp_id(TargetType.polygon, poly_ids[0])
                if current_struct_id == -1:
                    current_struct_id = StructuresData.UNASSIGNED_STRUCT

        current_struct_idx = StructuresData.UNASSIGNED_STRUCT
        for idx, struct_id in enumerate(self._comp.data.structures['id'].data):
            if struct_id == current_struct_id:
                current_struct_idx = idx
                break

        # Get original structure ids, so we know if the user deletes one.
        old_ids = list(self._comp.data.structures['id'].data.astype(int))

        title = 'Assign Structure' if num_polys == 1 else 'Assign Structures'
        dlg = StructuresParentDialog(title, parent, self._comp.data, current_struct_idx)
        if num_polys > 1:  # Add the multi-select warning if needed.
            dlg.add_multi_polygon_select_warning()

        if dlg.exec():
            self._comp.data = dlg.structures_dialog.structure_data
            self.update_structures_list(query, old_ids)
            struct_id = int(self._comp.data.structures['id'].data.tolist()[dlg.selected_structure])
            for poly_id in poly_ids:
                self._comp.update_component_id(TargetType.polygon, poly_id, struct_id)
            self._comp.data.commit()

        # Delete the id dumped by xms files.
        shutil.rmtree(os.path.join(os.path.dirname(self._comp.main_file), 'temp'), ignore_errors=True)
        return [], []

    def open_structures_properties(self, query, parent):
        """Opens the Structures Properties dialog and saves component data state on OK.

        Args:
            query (:obj:`Query`): Object for communicating with XMS
            parent (QWidget): The Qt parent window container.

        Returns:
            (:obj:`tuple(list, list)`):
                messages (:obj:`list[tuple(str, str)]`): List of tuples with the first element of the tuple being the
                message level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.

                action_requests (:obj:`list[ActionRequest]`): List of actions for XMS to perform.
        """
        # get original structure ids
        ids = list(self._comp.data.structures['id'].astype(int))
        dlg = StructuresParentDialog('Structure List and Properties', parent, self._comp.data)
        if dlg.exec():
            self._comp.data = dlg.structures_dialog.structure_data
            self.update_structures_list(query, ids)
            self._comp.data.commit()
        return [], []
