"""A script for tracking the progress an CMS-Wave simulation run."""

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query

# 4. Local modules
from xms.cmswave.data.simulation_data import SimulationData


class ProgressTracker:
    """A class for tracking the progress of an CMS-Wave simulation run."""
    prog = None
    query = None
    echo_file = None
    echo_pos = 0
    num_timesteps = 0
    current_timestep = 0

    def __init__(self):
        """Constructor that does nothing."""
        pass

    @staticmethod
    def progress_function():
        """Progress is calculated and sent to query as a percentage."""
        if not ProgressTracker.echo_file:
            ProgressTracker.echo_file = ProgressTracker.prog.command_line_output_file

        found_iter = False
        try:
            with open(ProgressTracker.echo_file, "r") as f:
                f.seek(ProgressTracker.echo_pos)
                echo_line = f.readline()
                while echo_line:
                    if echo_line.endswith('\n') or echo_line.endswith('\r'):
                        # We currently only tracking the case times as part of the progress. Another variable that
                        # contributes to the length of the simulation run is the size of the grid. An improvement
                        # would be to update progress based on the number of columns processed in the current timestep,
                        # but STWAVE doesn't do this. Figured if it is good enough for him, it's good enough for us.
                        if echo_line.startswith('Wave Input Index ='):
                            found_iter = True
                            ProgressTracker.current_timestep += 1
                            ProgressTracker.echo_pos = f.tell()
                    echo_line = f.readline()
        except Exception:
            pass  # File might not exist yet

        if found_iter:
            percent_done = ((ProgressTracker.current_timestep - 1) / ProgressTracker.num_timesteps) * 100.0
            ProgressTracker.query.update_progress_percent(percent_done)

    @staticmethod
    def start_tracking():
        """Entry point for the CMS-Wave progress script."""
        ProgressTracker.query = Query(progress_script=True)
        session = ProgressTracker.query.xms_agent.session
        # Get the simulation's hidden component
        sim_uuid = ProgressTracker.query.current_item_uuid()  # Get the simulation uuid
        sim_comp = ProgressTracker.query.item_with_uuid(sim_uuid, model_name='CMS-Wave', unique_name='Sim_Component')
        data = SimulationData(sim_comp.main_file)
        # get the snap case times
        ProgressTracker.num_timesteps = data.case_times.sizes[next(iter(data.case_times.dims))]
        ProgressTracker.prog = session.progress_loop
        ProgressTracker.prog.set_progress_function(ProgressTracker.progress_function)
        ProgressTracker.prog.start_loop()
