"""Methods for reading and writing component display files sent between components and XMS."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from enum import IntEnum
import os

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class DrawType(IntEnum):  # Matches order of XMS enum
    """An enumeration of possible target types matching TargetDisplayTypes in XMS."""
    draw_at_ids = 0
    draw_at_locations = 1
    replace_at_ids = 2


class XmsDisplayMessage:
    """Container for sending component display data to XMS."""
    def __init__(self, edit_uuid='', file='', draw_type=DrawType.draw_at_ids):
        """The consructor of the display options message to be sent to XMS.

        Args:
            edit_uuid (str): A uuid of the geometry of the display option attributes being edited.
            file (str): A relative or absolute path to the JSON file whish is assumed to be from
                the component runner. The JSON file will refer to the id files internally. The JSON
                file should contain only one category display list with however many categories
                inside it. If a relative path is given, it must be at or below the component folder.
            draw_type (:obj: of type `DrawType`): The draw type of the category display list.
        """
        self.edit_uuid = edit_uuid
        self.file = file
        self.draw_type = draw_type

    def __iter__(self):
        """This is used to create a dictionary for JSON.

        Yields:
            str: The geometery uuid of the attributes being edited.
            str: The absolute path and filename of the JSON file.
            int: The draw type of the display list.
        """
        yield self.edit_uuid
        if os.path.isabs(self.file):
            abs_path = os.path.normpath(self.file)
        else:
            abs_path = os.path.normpath(os.path.join(os.getcwd(), self.file))
        yield abs_path
        yield int(self.draw_type)
