"""Runs XMS DMI component-based model run events."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
import argparse
import os
import sys
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ExecutableCommand, Query, XmsEnvironment as XmEnv
from xms.data_objects.parameters import Simulation

# 4. Local modules
from xms.components.display.display_options_io import write_display_options_to_json


def generate_temp_file_name(temp_dir):
    """Create a unique filename in a temporary directory.

    Args:
        temp_dir (str): Location of the temporary directory

    Returns:
        str: See description
    """
    return os.path.normpath(os.path.join(temp_dir, str(uuid.uuid4())))


def main():
    """Entry point for component model run events."""
    arguments = argparse.ArgumentParser(description="Simulation run runner.")
    arguments.add_argument(dest='script', type=str, help='script to run')
    arguments.add_argument(dest='module_name', type=str, help='module of the method to run')
    arguments.add_argument(dest='class_name', type=str, help='class of the method to run')
    arguments.add_argument(dest='file', type=str, help='absolute path to the export location')
    arguments.add_argument(dest='temp_dir', type=str, help='absolute path to the temp directory')
    arguments.add_argument(dest='command', type=str, help='One of: solution_load, model_run')
    parsed_args = arguments.parse_args()

    query = Query(timeout=300000)

    sim = None
    result = query._impl._instance.Get('none')
    if result['none'] and result['none'][0]:
        # Wrap the C++ exposed Simulation in a pure Python class.
        sim = Simulation(instance=result['none'][0])
    execs = None
    plot_dict = None

    # Disallow Query.send() when we are running the migration scripts.
    query._impl._instance.SetAllowSend(False)  # Only exposed on the C++ exposed interface
    root_place_mark = query._impl._instance.GetContext().GetRootInstance()

    try:
        mod = __import__(parsed_args.module_name, fromlist=[parsed_args.class_name])
        klass = getattr(mod, parsed_args.class_name)
        class_instance = klass()

        if parsed_args.command == 'model_run':
            # We are running the model.
            get_exec_method = class_instance.get_executables
            execs = get_exec_method(sim, query, parsed_args.file)

            # Get plot data for the model run.
            plots = class_instance.plots
            plot_dict = {}
            for plot in plots:
                temp_file = generate_temp_file_name(parsed_args.temp_dir)
                write_display_options_to_json(temp_file, plots[plot][0])
                plot_dict[plot] = [temp_file, plots[plot][1]]
        else:  # parsed_args.command == 'solution_load'
            # We are loading a solution from an existing model run.
            sol_load_method = class_instance.get_solution_load_actions
            actions = sol_load_method(sim, query, parsed_args.file)

            # Package up the solution load ActionRequests in an ExecutableCommand object so XMS can unpack them the
            # same way it does when running a solution
            cmd = ExecutableCommand()
            for action in actions:
                cmd.add_solution_file(action)
            execs = [cmd]
    except Exception as ex:
        XmEnv.report_error(ex)

    # clear place marks to make it harder to send extra data to XMS
    ctxt = query._impl._instance.GetContext()
    ctxt.ClearPlaceMarks()
    query._impl._instance.SetContext(ctxt)
    if execs:
        # Unwrap the pure Python ExecutableCommands.
        query._impl._instance.Set([{'executable': [exec_cmd._instance for exec_cmd in execs]}], root_place_mark)
    if plot_dict:
        query._impl._instance.Set([{'plot': plot_dict}], root_place_mark)

    # Re-enable Query.send()
    query._impl._instance.SetAllowSend(True)
    query.send(True)
    sys.exit()


if __name__ == "__main__":
    main()
