"""Code to store interface with the XMS mesh generator coverage."""

# 1. Standard Python modules

# 2. Third party modules
import h5py
import numpy as np

# 3. Aquaveo modules
from xms.api._xmsapi.dmi import DataDumpIOBase
from xms.data_objects.parameters import Coverage, FilterLocation

# 4. Local modules
# MeshPolyEnum from XMS
from xms.coverage import coverage_util

MESH_NONE = 0
MESH_PATCH = 1
MESH_PAVE = 2
MESH_SPAVE = 3
MESH_CPAVE = 4
MESH_UGRID = 5
MESH_END = 6
# bathpolytype enum from XMS
BATH_CONSTANT = 0
BATH_SCAT = 1
BATH_RASTER = 2
BATH_EXIST = 3
BATH_FILL_POLY_FROM_EDGES = 4
# interptype enum from XMS
LINEAR = 0
IDW = 1
NN = 2
LAPLACIAN = 3  # In XMS enum but not accessible from GUI
# extraptype enum from XMS
IDWEXTRAP = 0
SINGLEVALUE = 1
SIGEXTRAP = 2  # In XMS enum but not accessible from GUI
NEARESTVALUE = 3  # In XMS enum but not accessible from GUI
LAPLACIAN_EXTRAP = 4  # In XMS enum but not accessible from GUI
DSETEXTRAP = 5  # In XMS enum but not accessible from GUI
# weightsourcetype enum from XMS
NPOINTS = 0
NPOINTSINQUAD = 1
ALLPOINTS = 2
# Node mege option enum from XMS
NODESPLIT = 0
NODEMERGE = 1
NODEDEGEN = 2


class MeshGeneratorCoverage(DataDumpIOBase):
    """Class to interface with the XMS mesh generator coverage."""
    def __init__(self, file_name=''):
        """Construct the mesh generator coverage.

        Notes:
            All variables are parallel arrays.

        Args:
            file_name(str): Path to the H5 dump file
        """
        super().__init__()
        super().SetSelf(self)
        self._cov = None
        self._has_read = False
        self._file_name = file_name
        self._poly_ids = []
        self._mesh_type = []  # MESH_* constants
        self._bathy_type = []  # BATH_* constants
        self._bias = []
        self._cell_size = []
        self._const_elev = []
        self._interp_option = []  # interptype constants
        self._extrap_option = []  # extraptype constants
        self._idw_use = []  # weightsourcetype constants
        self._idw_num_use = []
        self._truncate = []
        self._truncate_min = []
        self._truncate_max = []
        self._timestep_idx = []
        self._extrap_const = []
        self._geom_uuid = []
        self._dset_uuid = []
        self._interp_option_bathy = []  # interptype constants
        self._extrap_option_bathy = []  # extraptype constants
        self._idw_use_bathy = []  # weightsourcetype constants
        self._idw_num_use_bathy = []
        self._truncate_bathy = []
        self._truncate_min_bathy = []
        self._truncate_max_bathy = []
        self._timestep_idx_bathy = []
        self._extrap_const_bathy = []
        self._geom_uuid_bathy = []
        self._dset_uuid_bathy = []
        self._raster_uuid = []
        self._ugrid_bridge_uuid = []
        # Polygon node merge options
        self._polygon_node_merge = {}
        # Refine point attributes
        self._point_ids = []
        self._refine_on = []
        self._create_mesh_node = []
        self._element_size = []
        self._merge_option = []  # NODE* constants

    def _read_if_needed(self):
        """Read the H5 file the first time anything is requested."""
        if self._has_read is False and self._file_name:
            self.ReadDump(self._file_name)

    def _read_poly_attributes(self, att_group):
        """Read the attributes of the polygons.

        Args:
            att_group (h5py.Group): The mesh generation attribute group
        """
        if 'Id' not in att_group:
            return  # No polygons?
        self._poly_ids = att_group['Id'][:]
        self._mesh_type = att_group['MeshingType'][:]
        self._bathy_type = att_group['BathyType'][:]
        self._bias = att_group['Bias'][:]
        self._cell_size = att_group['Cellsize'][:]
        self._const_elev = att_group['ConstantElev'][:]
        self._interp_option = att_group['SizeInterpOpt'][:]
        self._extrap_option = att_group['SizeExtrapOpt'][:]
        self._idw_use = att_group['SizeExtrapIdwUse'][:]
        self._idw_num_use = att_group['SizeExtrapIdwNum'][:]
        self._truncate = att_group['SizeTruncate'][:]
        self._truncate_min = att_group['SizeTruncateMin'][:]
        self._truncate_max = att_group['SizeTruncateMax'][:]
        self._timestep_idx = att_group['SizeTimestepIdx'][:]
        self._extrap_const = att_group['SizeExtrapVal'][:]
        self._geom_uuid = att_group['SizeGeomUuid'][:].astype(str)
        self._dset_uuid = att_group['SizeDsetUuid'][:].astype(str)
        self._interp_option_bathy = att_group['ElevInterpOpt'][:]
        self._extrap_option_bathy = att_group['ElevExtrapOpt'][:]
        self._idw_use_bathy = att_group['ElevExtrapIdwUse'][:]
        self._idw_num_use_bathy = att_group['ElevExtrapIdwNum'][:]
        self._truncate_bathy = att_group['ElevTruncate'][:]
        self._truncate_min_bathy = att_group['ElevTruncateMin'][:]
        self._truncate_max_bathy = att_group['ElevTruncateMax'][:]
        self._timestep_idx_bathy = att_group['ElevTimestepIdx'][:]
        self._extrap_const_bathy = att_group['ElevExtrapVal'][:]
        self._geom_uuid_bathy = att_group['ElevGeomUuid'][:].astype(str)
        self._dset_uuid_bathy = att_group['ElevDsetUuid'][:].astype(str)
        self._raster_uuid = att_group['RasterUuid'][:].astype(str)
        self._ugrid_bridge_uuid = att_group['BridgeUGridUuid'][:].astype(str)
        self._read_poly_node_merge_flags(att_group)

    def _read_poly_node_merge_flags(self, att_group):
        """Read the polygon node merge option arrays into a map.

        Args:
            att_group (h5py.Group): The mesh generation attribute group
        """
        # Build the polygon node merge option map
        poly_ids = att_group['PolyNodePolyId'][:]
        node_ids = att_group['PolyNodeId'][:]
        merge_opts = att_group['PolyNodeMergeOpt'][:]
        for poly_id, node_id, merge_opt in zip(poly_ids, node_ids, merge_opts):
            self._polygon_node_merge.setdefault(poly_id, {})[node_id] = merge_opt

    def _read_point_attributes(self, att_group):
        """Read the attributes of the points.

        Args:
            att_group (h5py.Group): The mesh generation attribute group
        """
        if 'PointId' not in att_group:
            return  # No points
        self._point_ids = att_group['PointId'][:]
        self._refine_on = att_group['RefineOn'][:]
        self._create_mesh_node = att_group['CreateMeshNode'][:]
        self._element_size = att_group['ElementSize'][:]
        self._merge_option = att_group['MergeOpt'][:]

    def _write_poly_attributes(self, att_group):
        """Write the attributes of the polygons.

        Args:
            att_group (h5py.Group): The mesh generation attribute group
        """
        dt = h5py.special_dtype(vlen=str)
        att_group.create_dataset('Id', dtype='i', data=self.poly_ids)
        att_group.create_dataset('MeshingType', dtype='i', data=self.mesh_type)
        att_group.create_dataset('BathyType', dtype='i', data=self.bathy_type)
        att_group.create_dataset('Bias', dtype='float64', data=self.bias)
        att_group.create_dataset('BridgeUGridUuid', data=np.array(self.ugrid_bridge_uuid, dtype=dt))
        att_group.create_dataset('Cellsize', dtype='float64', data=self.cell_size)
        att_group.create_dataset('ConstantElev', dtype='float64', data=self.const_elev)
        att_group.create_dataset('SizeInterpOpt', dtype='i', data=self.interp_option)
        att_group.create_dataset('SizeExtrapOpt', dtype='i', data=self.extrap_option)
        att_group.create_dataset('SizeExtrapIdwUse', dtype='i', data=self.idw_use)
        att_group.create_dataset('SizeExtrapIdwNum', dtype='i', data=self.idw_num_use)
        att_group.create_dataset('SizeTruncate', dtype='i', data=self.truncate)
        att_group.create_dataset('SizeTruncateMin', dtype='float64', data=self.truncate_min)
        att_group.create_dataset('SizeTruncateMax', dtype='float64', data=self.truncate_max)
        att_group.create_dataset('SizeTimestepIdx', dtype='i', data=self.timestep_idx)
        att_group.create_dataset('SizeExtrapVal', dtype='float64', data=self.extrap_constant)
        att_group.create_dataset('SizeGeomUuid', data=np.array(self.geom_uuid, dtype=dt))
        att_group.create_dataset('SizeDsetUuid', data=np.array(self.dset_uuid, dtype=dt))
        att_group.create_dataset('ElevInterpOpt', dtype='i', data=self.interp_option_bathy)
        att_group.create_dataset('ElevExtrapOpt', dtype='i', data=self.extrap_option_bathy)
        att_group.create_dataset('ElevExtrapIdwUse', dtype='i', data=self.idw_use_bathy)
        att_group.create_dataset('ElevExtrapIdwNum', dtype='i', data=self.idw_num_use_bathy)
        att_group.create_dataset('ElevTruncate', dtype='i', data=self.truncate_bathy)
        att_group.create_dataset('ElevTruncateMin', dtype='float64', data=self.truncate_min_bathy)
        att_group.create_dataset('ElevTruncateMax', dtype='float64', data=self.truncate_max_bathy)
        att_group.create_dataset('ElevTimestepIdx', dtype='i', data=self.timestep_idx_bathy)
        att_group.create_dataset('ElevExtrapVal', dtype='float64', data=self.extrap_constant_bathy)
        att_group.create_dataset('ElevGeomUuid', data=np.array(self.geom_uuid_bathy, dtype=dt))
        att_group.create_dataset('ElevDsetUuid', data=np.array(self.dset_uuid_bathy, dtype=dt))
        att_group.create_dataset('RasterUuid', data=np.array(self.raster_uuid, dtype=dt))
        self._write_poly_node_merge_flags(att_group)

    def _write_poly_node_merge_flags(self, att_group):
        """Write the polygon node merge option map as parallel arrays.

        Args:
            att_group (h5py.Group): The mesh generation attribute group
        """
        poly_ids = []
        node_ids = []
        merge_opts = []
        for poly_id, nodes in self._polygon_node_merge.items():
            for node_id, merge_opt in nodes.items():
                poly_ids.append(poly_id)
                node_ids.append(node_id)
                merge_opts.append(merge_opt)
        att_group.create_dataset('PolyNodePolyId', dtype='i', data=poly_ids)
        att_group.create_dataset('PolyNodeId', dtype='i', data=node_ids)
        att_group.create_dataset('PolyNodeMergeOpt', dtype='i', data=merge_opts)

    def _write_point_attributes(self, att_group):
        """Write the attributes of the points.

        Args:
            att_group (h5py.Group): The mesh generation attribute group
        """
        att_group.create_dataset('PointId', dtype='i', data=self.point_ids)
        att_group.create_dataset('RefineOn', dtype='i', data=self.refine_on)
        att_group.create_dataset('CreateMeshNode', dtype='i', data=self.create_mesh_node)
        att_group.create_dataset('ElementSize', dtype='float64', data=self.element_size)
        att_group.create_dataset('MergeOpt', dtype='i', data=self.merge_option)

    @property
    def m_cov(self):
        """Get the mesh generator coverage geometry.

        Returns:
            xms.data_objects.parameters.Coverage: The mesh generator coverage geometry
        """
        self._read_if_needed()
        return self._cov

    @m_cov.setter
    def m_cov(self, val):
        """Set the mesh generator coverage geometry.

        Args:
            val (xms.data_objects.parameters.Coverage): The mesh generator coverage geometry
        """
        self._cov = val

    @property
    def poly_ids(self):
        """Get the polygon feature ids.

        Returns:
            list[int]: The polygon feature ids
        """
        self._read_if_needed()
        return self._poly_ids

    @poly_ids.setter
    def poly_ids(self, val):
        """Set the polygon feature ids.

        Args:
            val (list[int]): The polygon feature ids
        """
        self._poly_ids = val

    @property
    def mesh_type(self):
        """Get the polygon meshing types.

        Returns:
            list[int]: See description
        """
        self._read_if_needed()
        return self._mesh_type

    @mesh_type.setter
    def mesh_type(self, val):
        """Set the polygon meshing types.

        Args:
            val (list[int]): The polygon meshing types
        """
        self._mesh_type = val

    @property
    def bathy_type(self):
        """Get the polygon bathy types.

        Returns:
            list[int]: See description
        """
        self._read_if_needed()
        return self._bathy_type

    @bathy_type.setter
    def bathy_type(self, val):
        """Set the polygon bathy types.

        Args:
            val (list[int]): The polygon bathy types
        """
        self._bathy_type = val

    @property
    def bias(self):
        """Get the polygon bias values.

        Returns:
            list[float]: See description
        """
        self._read_if_needed()
        return self._bias

    @bias.setter
    def bias(self, val):
        """Set the polygon bias values.

        Args:
            val (list[float]): The polygon bias values
        """
        self._bias = val

    @property
    def cell_size(self):
        """Get the polygon cell size values.

        Returns:
            list[float]: See description
        """
        self._read_if_needed()
        return self._cell_size

    @cell_size.setter
    def cell_size(self, val):
        """Set the polygon cell size values.

        Args:
            val (list[float]): The polygon cell size values
        """
        self._cell_size = val

    @property
    def const_elev(self):
        """Get the polygon constant elevation values.

        Returns:
            list[float]: See description
        """
        self._read_if_needed()
        return self._const_elev

    @const_elev.setter
    def const_elev(self, val):
        """Set the polygon constant elevation values.

        Args:
            val (list[float]): The polygon constant elevation values
        """
        self._const_elev = val

    @property
    def interp_option(self):
        """Get the polygon interp options.

        Returns:
            list[int]: See description
        """
        self._read_if_needed()
        return self._interp_option

    @interp_option.setter
    def interp_option(self, val):
        """Set the polygon interp options.

        Args:
            val (list[int]): The polygon interp options
        """
        self._interp_option = val

    @property
    def extrap_option(self):
        """Get the polygon extrap options.

        Returns:
            list[int]: See description
        """
        self._read_if_needed()
        return self._extrap_option

    @extrap_option.setter
    def extrap_option(self, val):
        """Set the polygon extrap options.

        Args:
            val (list[int]): The polygon extrap options
        """
        self._extrap_option = val

    @property
    def idw_use(self):
        """Get the polygon IDW use options.

        Returns:
            list[int]: See description
        """
        self._read_if_needed()
        return self._idw_use

    @idw_use.setter
    def idw_use(self, val):
        """Set the polygon IDW use options.

        Args:
            val (list[int]): The polygon IDW use options
        """
        self._idw_use = val

    @property
    def idw_num_use(self):
        """Get the polygon IDW num use options.

        Returns:
            list[int]: See description
        """
        self._read_if_needed()
        return self._idw_num_use

    @idw_num_use.setter
    def idw_num_use(self, val):
        """Set the polygon IDW num use options.

        Args:
            val (list[int]): The polygon IDW num use options
        """
        self._idw_num_use = val

    @property
    def truncate(self):
        """Get the polygon truncate options.

        Returns:
            list[int]: See description
        """
        self._read_if_needed()
        return self._truncate

    @truncate.setter
    def truncate(self, val):
        """Set the polygon truncate options.

        Args:
            val (list[int]): The polygon truncate options
        """
        self._truncate = val

    @property
    def truncate_min(self):
        """Get the polygon truncate minimums.

        Returns:
            list[float]: See description
        """
        self._read_if_needed()
        return self._truncate_min

    @truncate_min.setter
    def truncate_min(self, val):
        """Set the polygon truncate minimums.

        Args:
            val (list[float]): The polygon truncate minimums
        """
        self._truncate_min = val

    @property
    def truncate_max(self):
        """Get the polygon truncate maximums.

        Returns:
            list[float]: See description
        """
        self._read_if_needed()
        return self._truncate_max

    @truncate_max.setter
    def truncate_max(self, val):
        """Set the polygon truncate maximums.

        Args:
            val (list[float]): The polygon truncate maximums
        """
        self._truncate_max = val

    @property
    def timestep_idx(self):
        """Get the polygon timestep indices.

        Returns:
            list[int]: See description
        """
        self._read_if_needed()
        return self._timestep_idx

    @timestep_idx.setter
    def timestep_idx(self, val):
        """Set the polygon timestep indices.

        Args:
            val (list[int]): The polygon timestep indices
        """
        self._timestep_idx = val

    @property
    def extrap_constant(self):
        """Get the polygon extrap constants.

        Returns:
            list[float]: See description
        """
        self._read_if_needed()
        return self._extrap_const

    @extrap_constant.setter
    def extrap_constant(self, val):
        """Set the polygon extrap constants.

        Args:
            val (list[float]): The polygon extrap constants.
        """
        self._extrap_const = val

    @property
    def geom_uuid(self):
        """Get the polygon geometry UUIDs.

        Returns:
            list[str]: See description
        """
        self._read_if_needed()
        return self._geom_uuid

    @geom_uuid.setter
    def geom_uuid(self, val):
        """Set the polygon geometry UUIDs.

        Args:
            val (list[str]): The polygon geometry UUIDs
        """
        self._geom_uuid = val

    @property
    def dset_uuid(self):
        """Get the polygon dataset UUIDs.

        Returns:
            list[str]: See description
        """
        self._read_if_needed()
        return self._dset_uuid

    @dset_uuid.setter
    def dset_uuid(self, val):
        """Set the polygon dataset UUIDs.

        Args:
            val (list[str]): The polygon dataset UUIDs
        """
        self._dset_uuid = val

    @property
    def interp_option_bathy(self):
        """Get the polygon interp options (bathy).

        Returns:
            list[int]: See description
        """
        self._read_if_needed()
        return self._interp_option_bathy

    @interp_option_bathy.setter
    def interp_option_bathy(self, val):
        """Set the polygon interp options (bathy).

        Args:
            val (list[int]): The polygon interp options
        """
        self._interp_option_bathy = val

    @property
    def extrap_option_bathy(self):
        """Get the polygon extrap options (bathy).

        Returns:
            list[int]: See description
        """
        self._read_if_needed()
        return self._extrap_option_bathy

    @extrap_option_bathy.setter
    def extrap_option_bathy(self, val):
        """Set the polygon extrap options (bathy).

        Args:
            val (list[int]): The polygon extrap options
        """
        self._extrap_option_bathy = val

    @property
    def idw_use_bathy(self):
        """Get the polygon IDW use options (bathy).

        Returns:
            list[int]: See description
        """
        self._read_if_needed()
        return self._idw_use_bathy

    @idw_use_bathy.setter
    def idw_use_bathy(self, val):
        """Set the polygon IDW use options (bathy).

        Args:
            val (list[int]): The polygon IDW use options
        """
        self._idw_use_bathy = val

    @property
    def idw_num_use_bathy(self):
        """Get the polygon IDW num use options (bathy).

        Returns:
            list[int]: See description
        """
        self._read_if_needed()
        return self._idw_num_use_bathy

    @idw_num_use_bathy.setter
    def idw_num_use_bathy(self, val):
        """Set the polygon IDW num use options (bathy).

        Args:
            val (list[int]): The polygon IDW num use options
        """
        self._idw_num_use_bathy = val

    @property
    def truncate_bathy(self):
        """Get the polygon truncate options (bathy).

        Returns:
            list[int]: See description
        """
        self._read_if_needed()
        return self._truncate_bathy

    @truncate_bathy.setter
    def truncate_bathy(self, val):
        """Set the polygon truncate options (bathy).

        Args:
            val (list[int]): The polygon truncate options
        """
        self._truncate_bathy = val

    @property
    def truncate_min_bathy(self):
        """Get the polygon truncate minimums (bathy).

        Returns:
            list[float]: See description
        """
        self._read_if_needed()
        return self._truncate_min_bathy

    @truncate_min_bathy.setter
    def truncate_min_bathy(self, val):
        """Set the polygon truncate minimums (bathy).

        Args:
            val (list[float]): The polygon truncate minimums
        """
        self._truncate_min_bathy = val

    @property
    def truncate_max_bathy(self):
        """Get the polygon truncate maximums (bathy).

        Returns:
            list[float]: See description
        """
        self._read_if_needed()
        return self._truncate_max_bathy

    @truncate_max_bathy.setter
    def truncate_max_bathy(self, val):
        """Set the polygon truncate maximums (bathy).

        Args:
            val (list[float]): The polygon truncate maximums
        """
        self._truncate_max_bathy = val

    @property
    def timestep_idx_bathy(self):
        """Get the polygon timestep indices (bathy).

        Returns:
            list[int]: See description
        """
        self._read_if_needed()
        return self._timestep_idx_bathy

    @timestep_idx_bathy.setter
    def timestep_idx_bathy(self, val):
        """Set the polygon timestep indices (bathy).

        Args:
            val (list[int]): The polygon timestep indices
        """
        self._timestep_idx_bathy = val

    @property
    def extrap_constant_bathy(self):
        """Get the polygon extrap constants (bathy).

        Returns:
            list[float]: See description
        """
        self._read_if_needed()
        return self._extrap_const_bathy

    @extrap_constant_bathy.setter
    def extrap_constant_bathy(self, val):
        """Set the polygon extrap constants (bathy).

        Args:
            val (list[float]): The polygon extrap constants.
        """
        self._extrap_const_bathy = val

    @property
    def geom_uuid_bathy(self):
        """Get the polygon geometry UUIDs (bathy).

        Returns:
            list[str]: See description
        """
        self._read_if_needed()
        return self._geom_uuid_bathy

    @geom_uuid_bathy.setter
    def geom_uuid_bathy(self, val):
        """Set the polygon geometry UUIDs (bathy).

        Args:
            val (list[str]): The polygon geometry UUIDs
        """
        self._geom_uuid_bathy = val

    @property
    def dset_uuid_bathy(self):
        """Get the polygon dataset UUIDs (bathy).

        Returns:
            list[str]: See description
        """
        self._read_if_needed()
        return self._dset_uuid_bathy

    @dset_uuid_bathy.setter
    def dset_uuid_bathy(self, val):
        """Set the polygon dataset UUIDs (bathy).

        Args:
            val (list[str]): The polygon dataset UUIDs
        """
        self._dset_uuid_bathy = val

    @property
    def raster_uuid(self):
        """Get the polygon raster UUIDs.

        Returns:
            list[str]: See description
        """
        self._read_if_needed()
        return self._raster_uuid

    @raster_uuid.setter
    def raster_uuid(self, val):
        """Set the polygon raster UUIDs.

        Args:
            val (list[str]): The polygon raster UUIDs
        """
        self._raster_uuid = val

    @property
    def ugrid_bridge_uuid(self):
        """Get the polygon ugrid/bridge UUIDs.

        Returns:
            list[str]: See description
        """
        self._read_if_needed()
        return self._ugrid_bridge_uuid

    @ugrid_bridge_uuid.setter
    def ugrid_bridge_uuid(self, val):
        """Set the polygon ugrid/bridge UUIDs.

        Args:
            val (list[str]): The polygon ugrid/bridge UUIDs
        """
        self._ugrid_bridge_uuid = val

    @property
    def polygon_node_merge(self):
        """Get the polygon node merge flags.

        Returns:
            dict: Mapping of polygon ID to node ID to merge option flag
        """
        self._read_if_needed()
        return self._polygon_node_merge

    @polygon_node_merge.setter
    def polygon_node_merge(self, val):
        """Set the polygon node merge flags.

        Args:
            val (dict): Mapping of polygon ID to node ID to merge option flag
        """
        self._polygon_node_merge = val

    @property
    def point_ids(self):
        """Get the point feature ids.

        Returns:
            list[int]: The point feature ids
        """
        self._read_if_needed()
        return self._point_ids

    @point_ids.setter
    def point_ids(self, val):
        """Set the point feature ids.

        Args:
            val (list[int]): The point feature ids
        """
        self._point_ids = val

    @property
    def refine_on(self):
        """Get the point refine flags.

        Returns:
            list[int]: The point refine flags
        """
        self._read_if_needed()
        return self._refine_on

    @refine_on.setter
    def refine_on(self, val):
        """Set the point refine flags.

        Args:
            val (list[int]): The point refine flags
        """
        self._refine_on = val

    @property
    def create_mesh_node(self):
        """Get the point create mesh node flags.

        Returns:
            list[int]: The point create mesh node flags
        """
        self._read_if_needed()
        return self._create_mesh_node

    @create_mesh_node.setter
    def create_mesh_node(self, val):
        """Set the point create mesh node flags.

        Args:
            val (list[int]): The point create mesh node flags
        """
        self._create_mesh_node = val

    @property
    def element_size(self):
        """Get the refine point element sizes.

        Returns:
            list[float]: The refine point element sizes
        """
        self._read_if_needed()
        return self._element_size

    @element_size.setter
    def element_size(self, val):
        """Set the refine point element sizes.

        Args:
            val (list[float]): The refine point element sizes
        """
        self._element_size = val

    @property
    def merge_option(self):
        """Get the point merge options.

        Returns:
            list[int]: The point merge options
        """
        self._read_if_needed()
        return self._merge_option

    @merge_option.setter
    def merge_option(self, val):
        """Set the point merge options.

        Args:
            val (list[int]): The point merge optionss
        """
        self._merge_option = val

    def ReadDump(self, file_name):  # noqa: N802
        """Read mesh generation coverage geometry and attributes from an H5 file written by XMS.

        Args:
            file_name (str): Path to the file to read
        """
        self._has_read = True
        f = h5py.File(file_name, 'r')
        the_cov_name = coverage_util.read_coverage_name(f)
        if the_cov_name == '':
            f.close()
            return

        attributes_group = f['Map Data/' + the_cov_name + '/Attributes']
        self._read_poly_attributes(attributes_group)
        self._read_point_attributes(attributes_group)
        f.close()

        # read geometry
        self._cov = Coverage(file_name, '/Map Data/' + the_cov_name)
        self._cov.get_points(FilterLocation.LOC_NONE)  # force geometry to load from H5

    def WriteDump(self, file_name):  # noqa: N802,C901
        """Write the wind coverage geometry and attributes to an H5 file XMS can read.

        Args:
            file_name (str): Path to the output H5 file
        """
        self._cov.write_h5(file_name)
        f = h5py.File(file_name, 'a')
        cov_group = f['Map Data/Coverage1']
        att_group = cov_group.create_group('Attributes')
        self._write_poly_attributes(att_group)
        self._write_point_attributes(att_group)
        f.close()

    def Copy(self):  # noqa: N802
        """Return a reference to this object."""
        return self

    def GetDumpType(self):  # noqa: N802
        """Get the XMS coverage dump type."""
        return 'xms.coverage.mesh_generator'


def ReadDumpWithObject(file_name):  # noqa: N802
    """Read a mesh generator coverage dump file.

    Args:
        file_name (str): Filepath to the dumped coverage to read

    Returns:
        MeshGeneratorCoverage: The loaded mesh generator coverage
    """
    mesh_cov = MeshGeneratorCoverage(file_name)
    return mesh_cov
