"""Data class for RSM simulation component."""
# 1. Standard python modules
import os

# 2. Third party modules
import pkg_resources

# 3. Aquaveo modules
from xms.components.bases.xarray_base import XarrayBase
from xms.gmi.data.generic_model import GenericModel

# 4. Local modules

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"


class SimData(XarrayBase):
    """Manages data file for the hidden simulation component."""
    def __init__(self, data_file):
        """Initializes the data class.

        Args:
            data_file (:obj:`str`): The netcdf file (with path) associated with this instance data. Probably the owning
                component's main file.
        """
        super().__init__(data_file)
        self.main_file = data_file
        self.info.attrs['FILE_TYPE'] = 'CSTORM_SIMULATION'
        self.info.attrs['VERSION'] = pkg_resources.get_distribution('xmscstorm').version
        self._gm = None
        self._create_generic_model()
        if 'OPTIONS' in self.info.attrs:
            self._gm.global_parameters.restore_values(self.info.attrs['OPTIONS'])

        self._migrate_data()

        if not os.path.isfile(data_file):
            self.commit()

    def _migrate_data(self):
        """Check for missing data when opening an existing file."""
        pass

    @property
    def generic_model(self) -> GenericModel:
        """Property to get the generic model from the coverage data."""
        return self._gm

    def _create_generic_model(self):
        """Creates a generic model with the necessary parameters for 'data exchange'."""
        gm = GenericModel()
        self._gm = gm
        gp = gm.global_parameters

        grp = gp.add_group('model_control', 'Model Control', is_active=True)
        opts = [
            '1 - Tight one-way coupling ADCIRC->STWAVE (ADCIRC sends surge and wind)', '2 - [NOT ACTIVE]',
            '3 - Tight two-way coupling ADCIRC<->STWAVE (ADCIRC sends surge and wind; STWAVE sends non-zero wave '
            'stress gradients', '4 - [NOT ACTIVE]',
            '5 - Tight one-way coupling ADCIRC->STWAVE (ADCIRC sends surge, wind, and ice)',
            '6 - Tight two-way coupling ADCIRC<->STWAVE (ADCIRC sends surge, wind, and ice; STWAVE sends non-zer0 '
            'wave stress gradients)'
        ]
        grp.add_option('wsid', 'Wave service option (wsid)', opts[0], opts)
        grp.add_boolean(name='template', label='Export as a template', default=False)

    def commit(self):
        """Save current in-memory component parameters to data file."""
        # Always load all data into memory and vacuum the file. Data is small enough and prevents filesystem bloat.
        super().commit()  # Recreates the NetCDF file
