"""The Assign EWN Feature polygon attribute dialog."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtWidgets import (
    QDialogButtonBox, QGroupBox, QHBoxLayout, QLabel, QMessageBox, QPushButton, QSizePolicy, QVBoxLayout
)

# 3. Aquaveo modules
from xms.api.tree import tree_util
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules


class RefineQuadTreeDialog(XmsDlg):
    """A dialog for setting the refinement level of quadtree cells."""
    def __init__(self, parent, pe_tree, cov_uuid):
        """Initialize quadtree refinement dialog."""
        super().__init__(parent, 'xms.ewn.gui.refine_quadtree_dialog')
        self._target_uuid = None
        self._source_uuid = cov_uuid
        self._pe_tree = pe_tree

        self._int_validator = None

        self._ugrid_pe_tree = None
        self._filter_tree_to_quadtree_meshes()

        self._widgets = {}
        self._setup_ui()

    def _setup_ui(self):
        """Setup dialog widgets."""
        self.setWindowTitle("Refine Quadtree")
        self._add_main_layout()
        self._add_labels()
        self._add_widgets()
        self._add_button_box()
        self._setup_validators()
        self._setup_connections()

    def _add_main_layout(self):
        """Add main layout to dialog."""
        self._widgets['grp_main'] = QGroupBox('')
        self._widgets['main_vert_layout'] = QVBoxLayout()
        self._widgets['main_vert_layout'].addWidget(self._widgets['grp_main'])
        self._widgets['grp_main_vert_layout'] = QVBoxLayout()
        self._widgets['grp_main'].setLayout(self._widgets['grp_main_vert_layout'])
        self._widgets['horiz_layout'] = QHBoxLayout()
        self._widgets['grp_main_vert_layout'].addLayout(self._widgets['horiz_layout'])
        self._widgets['left_label_layout'] = QVBoxLayout()
        self._widgets['horiz_layout'].addLayout(self._widgets['left_label_layout'])
        self._widgets['right_widget_layout'] = QVBoxLayout()
        self._widgets['horiz_layout'].addLayout(self._widgets['right_widget_layout'])
        self._widgets['horiz_layout'].addStretch()
        self.setLayout(self._widgets['main_vert_layout'])

    def _add_labels(self):
        """Add lables to the dialog."""
        self._widgets['lbl_target'] = QLabel('Quadtree:')
        self._widgets['left_label_layout'].addWidget(self._widgets['lbl_target'])

    def _add_widgets(self):
        """Add widgets to the dialog."""
        self._widgets['target_h_layout'] = QHBoxLayout()
        self._widgets['right_widget_layout'].addLayout(self._widgets['target_h_layout'])
        self._widgets['btn_select_target'] = QPushButton('Select QuadTree UGrid...')
        self._widgets['target_h_layout'].addWidget(self._widgets['btn_select_target'])
        self._widgets['txt_target'] = QLabel('(none selected)')
        self._widgets['txt_target_size_policy'] = QSizePolicy(QSizePolicy.Preferred, QSizePolicy.Fixed)
        self._widgets['txt_target'].setSizePolicy(self._widgets['txt_target_size_policy'])
        self._widgets['target_h_layout'].addWidget(self._widgets['txt_target'])
        self._widgets['target_h_layout'].addStretch()

    def _add_button_box(self):
        """Adds the button box to the bottom of the dialog."""
        self._widgets['btn_box'] = QDialogButtonBox()
        btn_flags = QDialogButtonBox.Ok | QDialogButtonBox.Cancel  # | QDialogButtonBox.Help
        self._widgets['btn_box'].setStandardButtons(btn_flags)
        self._widgets['btn_box'].accepted.connect(self.accept)
        self._widgets['btn_box'].rejected.connect(self.reject)
        self._widgets['btn_horiz_layout'] = QHBoxLayout()
        self._widgets['btn_horiz_layout'].addWidget(self._widgets['btn_box'])
        self._widgets['main_vert_layout'].addLayout(self._widgets['btn_horiz_layout'])

    def _setup_validators(self):
        pass

    def _setup_connections(self):
        """Setup dialog connections."""
        self._widgets['btn_select_target'].clicked.connect(self._on_select_geom)

    def _on_select_geom(self):
        """Select a ugrid or mesh."""
        selector_dlg = TreeItemSelectorDlg(
            title='Select QuadTree Mesh',
            target_type='',
            pe_tree=self._ugrid_pe_tree,
            parent=self,
            selectable_xms_types=['TI_UGRID_SMS']
        )
        if selector_dlg.exec():
            self._target_uuid = selector_dlg.get_selected_item_uuid()
            if not self._target_uuid:
                self._widgets['txt_target'].setText('(none selected)')
                return
            tree_path = tree_util.build_tree_path(self._pe_tree, self._target_uuid)
            self._widgets['txt_target'].setText(tree_path)

    def _filter_tree_to_quadtree_meshes(self):
        creator = tree_util.ProjectExplorerTreeCreator()
        self._ugrid_pe_tree = creator.copy(self._pe_tree)

    def accept(self):
        """Save data to persistent storage on OK."""
        if not self._target_uuid:
            msgbox = QMessageBox(QMessageBox.Warning, 'SMS', 'Select a target QuadTree', QMessageBox.Ok, self)
            msgbox.exec()
            return
        super().accept()

    def reject(self):
        """Cancel command."""
        super().reject()

    def tool_inputs(self):
        """Get tool inputs."""
        return {
            'ugrid_uuid': self._target_uuid,
            'ewn_uuid': self._source_uuid,  # TODO: Need to set this...
        }
