"""Module for exporting a FUNWAVE simulation with feedback."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging

# 2. Third party modules
from PySide2.QtCore import Signal

# 3. Aquaveo modules
from xms.api.dmi import Query, XmsEnvironment as XmEnv
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackDlg, ProcessFeedbackThread
from xms.guipy.dialogs.xms_parent_dlg import ensure_qapplication_exists

# 4. Local modules
from xms.funwave.dmi.xms_data import XmsData
from xms.funwave.file_io.control_writer import ControlWriter


class ExportSimulationRunner(ProcessFeedbackThread):
    """Class for exporting a FUNWAVE simulation in a worker thread."""
    processing_finished = Signal()

    def __init__(self, parent):
        """Constructor.

        Args:
            parent (:obj:`QWidget`): Parent of the QThread, probably the hidden parent dialog created by XMS.
        """
        super().__init__(parent=parent, do_work=self._do_work)
        self._logger = logging.getLogger('xms.funwave')

    def _export_simulation(self, query):
        """Export a simulation and its linked child simulations recursively.

        Args:
            query (:obj:`Query`): XMS interprocess communicator
        """
        xms_data = XmsData(query)

        # Export the simulation
        writer = ControlWriter(xms_data)
        writer.write()

    def _do_work(self):
        """Exports the simulation."""
        try:
            query = Query()
            self._export_simulation(query)
        except Exception:
            self._logger.exception('Error exporting simulation:')


def export_simulation_with_feedback():
    """Export a simulation with a feedback dialog."""
    ensure_qapplication_exists()
    worker = ExportSimulationRunner(None)

    # debug = False
    # if debug:
    #     worker.do_work()
    # else:
    display_text = {
        'title': 'FUNWAVE Export Simulation',
        'working_prompt': 'Exporting FUNWAVE simulation files. Please wait...',
        'warning_prompt': 'Warning(s) encountered while exporting simulation. Review log output for more details.',
        'error_prompt': 'Error(s) encountered while exporting simulation. Review log output for more details.',
        'success_prompt': 'Successfully exported simulation',
        'note': '',
        'auto_load': 'Close this dialog automatically when exporting is finished.'
    }
    feedback_dlg = ProcessFeedbackDlg(display_text=display_text, logger_name='xms', worker=worker,
                                      parent=None)
    feedback_dlg.testing = XmEnv.xms_environ_running_tests() == 'TRUE'
    if feedback_dlg.exec():
        if LogEchoQSignalStream.logged_error:
            XmEnv.report_export_error()
    else:
        XmEnv.report_export_aborted()
