"""MappedGridComponent class."""

__copyright__ = "(C) Copyright Aquaveo 2022"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
from PySide2.QtGui import QIcon

# 3. Aquaveo modules
from xms.components.display.display_options_io import read_display_options_from_json, write_display_options_to_json
from xms.components.display.xms_display_message import DrawType, XmsDisplayMessage
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.dialogs.category_display_options_list import CategoryDisplayOptionsDialog
from xms.guipy.dialogs.xms_parent_dlg import get_xms_icon

# 4. Local modules
from xms.gencade.components.gencade_component import GenCadeComponent
from xms.gencade.data.mapped_grid_data import MappedGridData


MAPPED_GRID_MAINFILE = 'mapped_grid_comp.nc'
MAPPED_GRID_DISPLAY = 'mapped_grid_display.json'
MAPPED_GRID_LOCATIONS = 'grid.display_ids'


class MappedGridComponent(GenCadeComponent):
    """A hidden Dynamic Model Interface (DMI) component for the mapped 1D grid coverage."""

    def __init__(self, main_file):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)
        self.data = MappedGridData(self.main_file)
        self.class_name = 'MappedGridComponent'
        self.module_name = 'xms.gencade.components.mapped_grid_component'
        self.tree_commands = [
            ('Display Options...', 'open_display_options'),
        ]
        self.disp_opts_file = os.path.join(os.path.dirname(self.main_file), MAPPED_GRID_DISPLAY)

    def get_initial_display_options(self, query, params):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.

        Returns:
            Empty message and ActionRequest lists
        """
        # Send the default display options list to XMS.
        self.display_option_list = [
            XmsDisplayMessage(file=self.disp_opts_file, draw_type=DrawType.draw_at_locations),  # lines
        ]
        return [], []

    def open_display_options(self, query, params, win_cont):
        """Shows the display options dialog.

        Args:
            query (:obj:`Query`):
            params (:obj:`list[str]`):
            win_cont (:obj:`QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        # Get the arc/line display options.
        arc_categories = CategoryDisplayOptionList()
        json_dict = read_display_options_from_json(self.disp_opts_file)
        arc_categories.from_dict(json_dict)
        categories_list = [arc_categories]

        wkt = ''
        if 'wkt' in arc_categories.projection:
            wkt = arc_categories.projection['wkt']
        dlg = CategoryDisplayOptionsDialog(categories_list, win_cont, package_name='xmsgencade')
        dlg.setWindowIcon(QIcon(get_xms_icon()))
        dlg.setModal(True)
        if dlg.exec():
            # write files
            category_lists = dlg.get_category_lists()
            for category_list in category_lists:
                category_list.projection['wkt'] = wkt
                write_display_options_to_json(self.disp_opts_file, category_list)
                self.display_option_list.append(
                    XmsDisplayMessage(file=self.disp_opts_file, draw_type=DrawType.draw_at_locations)
                )
        return [], []
