"""Override to paint combobox with selection."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtGui import QBrush
from PySide2.QtWidgets import QApplication, QStyle, QStyleOptionComboBox

# 3. Aquaveo modules
from xms.guipy.delegates.qx_cbx_delegate import QxCbxDelegate

# 4. Local modules


class SelectedComboboxDelegate(QxCbxDelegate):
    """A combobox delegate."""

    def __init__(self, parent):
        """Initializes the class.

        Args:
            parent (Something derived from :obj:`QWidget`): The parent window.
        """
        super().__init__(parent)

    def _is_row_selected(self, index):
        """Check if the index's row is currently selected (regardless of column).

        Args:
            index (:obj:`QModelIndex`): The index in the model

        Returns:
            (:obj:`bool`): True if the currently selected row is the same as the index
        """
        if index.flags() & QStyle.State_Selected:
            return True  # Currently selected item is index
        selected = self.parent().table_view.selectionModel().currentIndex()
        if selected.row() == index.row():
            return True  # Currently selected item is on the same row as index
        return False

    def paint(self, painter, option, index):
        """Override the paint event.

        Args:
            painter (:obj:`QPainter`): The painter
            option (:obj:`QStyleOptionViewItem`): The style options
            index (:obj:`QModelIndex`): The index in the model
        """
        if not index.isValid():
            return

        current_text = index.data(Qt.EditRole)
        cbx_opt = QStyleOptionComboBox()
        cbx_opt.currentText = current_text
        cbx_opt.rect = option.rect
        cbx_opt.state = option.state
        if index.flags() & Qt.ItemIsEnabled:
            cbx_opt.state |= QStyle.State_Enabled
        if self._is_row_selected(index):
            # This seems a little extreme, but not enabling any of the following three state flags led to goofy
            # drawing. For this table, we want to make it very clear which row is selected because there is only
            # the vertical header and one combobox column. Changing rows changes the values of several widgets
            # in a side pane.
            cbx_opt.state |= QStyle.State_Selected
            cbx_opt.state |= QStyle.State_MouseOver
            cbx_opt.state |= QStyle.State_HasFocus
            sel_brush = QBrush(option.palette.highlight())
            painter.setBrush(sel_brush)
        else:
            # Don't let Qt paint mouse over styles because we have hijacked the style to make it clear which row
            # is selected.
            cbx_opt.state = cbx_opt.state & ~QStyle.State_Selected
            cbx_opt.state = cbx_opt.state & ~QStyle.State_MouseOver
            cbx_opt.state = cbx_opt.state & ~QStyle.State_HasFocus
        cbx_opt.editable = False

        QApplication.style().drawComplexControl(QStyle.CC_ComboBox, cbx_opt, painter)
        QApplication.style().drawControl(QStyle.CE_ComboBoxLabel, cbx_opt, painter)
