"""The CustomDialogBase dialog."""

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"

# 1. Standard Python modules
from typing import Optional

# 2. Third party modules
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules

# 4. Local modules
from xms.gmi.data.component_data import CurveAdder, CurveGetter
from xms.gmi.data.generic_model import GroupSet
from xms.gmi.gui.base_dlg import BaseDialog
from xms.gmi.gui.dataset_callback import DatasetCallback
from xms.gmi.gui.parameter_widgets import make_widgets


class CustomDialogBase(BaseDialog):
    """
    Base class for a custom dialog (not dynamically created GroupSetDialog) to be used with GenericModel.

    This dialog accepts an `xms.gmi.data.generic_model.Section` and looks for existing Qt widgets to associate
    with generic model parameters. Widgets are found by naming the widgets a certain way, or by providing a dict that
    maps widget names to Qt widgets (see qt_widgets_map in setup_widgets()). Both methods can be used together. The
    naming convention is as follows (see parameter_widgets.widget_name_from_strings()):

    [widget type prefix]_[group name]_[parameter name] e.g. "cbx_group1_GridTypes"

    Custom Qt widgets must be the same types as those used by the ParameterWidget object associated with the Parameter.
    Some ParameterWidgets use multiple Qt widgets, e.g. a DATASET parameter uses a DatasetWidget which uses a
    QPushButton and a QLabel. For widget type prefixes, see parameter_widgets.widget_name_from_parameter(). All widgets
    associated with the parameter must be named the same, except for the widget type prefix, e.g.
    "btn_group1_dataset1", "txt_group1_dataset1".
    """
    def __init__(
        self,
        parent: QWidget | None,
        section: GroupSet,
        group_names: list[str],
        get_curve: Optional[CurveGetter],
        add_curve: Optional[CurveAdder],
        dlg_name: str,
        dataset_callback: Optional[DatasetCallback] = None,
    ):
        """
        Initialize the dialog.

        Args:
            parent: Parent window.
            section: Section in the model to display and edit values for.
            group_names: List of group names in section associated with this dialog.
            get_curve: A callable that takes a curve ID and whether that curve uses dates, and returns the curve's
                X and Y values. The X values will be of type `np.datetime64` if using dates, or `float` otherwise.
                This can be `None` if there are no curve parameters.
            add_curve: A callable that takes a list of X values, a list of Y values, and whether to use dates, and
                returns a curve ID for a curve with the input values. X values will be of type `np.datetime64` if
                using dates, or `float` otherwise. This function should arrange so that calling `get_curve` later with
                the same curve ID and `use_dates` will return the same X and Y values passed to this function. This can
                be `None` if there are no curve parameters.
            dlg_name: Unique name for this dialog. site-packages import path would make sense.
                Used as registry key for loading and saving the dialog geometry.
            dataset_callback: A callback for picking a dataset or getting its label. See `DatasetCallback` for details.
                This can be `None` if there are no dataset parameters.
        """
        super().__init__(parent, dlg_name)

        self._dataset_callback = dataset_callback
        self.section = section
        self._group_names = group_names
        self._get_curve = get_curve
        self._add_curve = add_curve
        self._qt_widgets_map: dict[str, QWidget] | None = None  # Maps widget names to the Qt widget
        self.widgets: dict = {}  # Stores ParameterWidgets. Needed since they aren't added to any layout

    def setup_widgets(self, qt_widgets_map: dict[str, QWidget] = None):
        """
        Makes ParameterWidgets, finds the Qt widgets, and initializes their data.

        qt_widgets_map is an alternate way to find qt widgets if using object names isn't sufficient. The keys must be
        the name returned from widget_name_from_strings().

        Args:
            qt_widgets_map: Dict of widget name -> Qt widget.
        """
        self._qt_widgets_map = qt_widgets_map
        for group_name in self._group_names:
            self.widgets[group_name] = make_widgets(
                self._dataset_callback, self, group_name, self.section.group(group_name), self
            )

    def find_widget(self, widget_type: type, widget_name: str) -> QWidget | None:
        """Finds and returns the Qt widget, or None if not found.

        Args:
            widget_type (type): QCheckBox, QPushButton etc.
            widget_name (str): The widget name (see widget_name_from_parameter()).

        Returns:
            (QWidget|None): See description.
        """
        widget = None
        if self._qt_widgets_map:
            widget = self._qt_widgets_map.get(widget_name)
        if not widget:
            widget = self.findChild(widget_type, widget_name)
        return widget
